// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/System/Io/FileSystem/Util/hkFileSystemCallbacks.h>

static bool fileFound(hkFileSystem& fs, const char* filename, const char* dirname, const char* filter)
{
    hkBool found = false;

    hkFileSystem::Iterator dirIter(&fs, dirname, filter);
    while(dirIter.advance())
    {
        const hkFileSystem::Entry entry(dirIter.current());
        if(entry.isFile())
        {

            if(!hkString::strCmp(entry.getName(), filename))
            {
                HK_TEST(!found); found = true;
            }
        }
    }
    return found;
}

int fileops_main()
{
    hkFileSystem& fs = hkFileSystem::getInstance();

    // Compression tests
    {
        // Register callbacks with file system
        const char* ext[] = { ".gz" };
        hkFileSystemCallback::CompressCbData extensions(ext, sizeof(ext) / sizeof(char*));
        fs.registerOpenWriterCb(hkFileSystemCallback::compress, &extensions);
        fs.registerOpenReaderCb(hkFileSystemCallback::decompress);

        hkStringBuf originalBuffer = "Hello File System\n\nHello File System\n\nHello File System\n\nHello File System\n\n"
                                     "Hello File System\n\nHello File System\n\nHello File System\n\nHello File System\n\n"
                                     "Hello File System\n\nHello File System\n\nHello File System\n\nHello File System";

        // Test compressed/uncompressed writing when callbacks are registered
        {
            hkRefPtr<hkStreamWriter> swCompr = fs.openWriter("testCompression.gz");
            if (HK_TEST(swCompr))
            {
                HK_TEST(swCompr->write(originalBuffer.cString(), originalBuffer.getLength()) == originalBuffer.getLength());
            }
            hkRefPtr<hkStreamWriter> swText = fs.openWriter("testCompression.txt");
            if (HK_TEST(swText))
            {
                HK_TEST(swText->write(originalBuffer.cString(), originalBuffer.getLength()) == originalBuffer.getLength());
            }
        }
        // Test compressed/uncompressed reading when callbacks are registered
        {
            hkRefPtr<hkStreamReader> srCompr = fs.openReader("testCompression.gz");
            if (HK_TEST(srCompr))
            {
                char readBuffer[1024];
                if (HK_TEST(srCompr->read(readBuffer, originalBuffer.getLength()) == originalBuffer.getLength()))
                {
                    readBuffer[originalBuffer.getLength()] = 0;
                    HK_TEST(hkString::strCmp(readBuffer, originalBuffer) == 0);
                }
            }
            hkRefPtr<hkStreamReader> srText = fs.openReader("testCompression.txt");
            if (HK_TEST(srText))
            {
                char readBuffer[1024];
                if (HK_TEST(srText->read(readBuffer, originalBuffer.getLength()) == originalBuffer.getLength()))
                {
                    readBuffer[originalBuffer.getLength()] = 0;
                    HK_TEST(hkString::strCmp(readBuffer, originalBuffer) == 0);
                }
            }
        }

        // Test compressed writing with seek/tell
        {
            hkStringBuf originalText = "Hello File System\n\n";

            hkRefPtr<hkStreamWriter> sw = fs.openWriter("testCompression2.gz");
            if (HK_TEST(sw))
            {
                HK_TEST(sw->write(originalText.cString(), originalText.getLength()) == originalText.getLength());
                if (sw->seekTellSupported())
                {
                    HK_TEST(sw->tell() == originalText.getLength());
                    HK_TEST_SUCCESS(sw->seek(-8, hkStreamWriter::STREAM_CUR));
                    hkStringBuf changedText = "Change";
                    sw->write(changedText, 6);
                    HK_TEST_SUCCESS(sw->seek(0, hkStreamWriter::STREAM_SET));
                    HK_TEST(sw->tell() == 0);
                    HK_TEST_SUCCESS(sw->seek(0, hkStreamWriter::STREAM_END));
                    hkStringBuf appendText("Appended to file\n");
                    HK_TEST(sw->write(appendText, appendText.getLength()) == appendText.getLength());
                    sw = nullptr;

                    hkRefPtr<hkStreamReader> sr = fs.openReader("testCompression2.gz");
                    if (HK_TEST(sr))
                    {
                        char readBuffer[1024];
                        if (HK_TEST(sr->read(readBuffer, originalText.getLength()) == originalText.getLength()))
                        {
                            readBuffer[originalText.getLength()] = 0;
                            HK_TEST(hkString::strCmp(readBuffer, "Hello File Change\n\n") == 0);
                        }
                        if (HK_TEST(sr->read(readBuffer, appendText.getLength()) == appendText.getLength()))
                        {
                            readBuffer[appendText.getLength()] = 0;
                            HK_TEST(hkString::strCmp(readBuffer, appendText) == 0);
                        }
                    }
                }
            }
        }

        // Delete test files
        HK_TEST_SUCCESS(fs.remove("testCompression.gz"));
        HK_TEST_SUCCESS(fs.remove("testCompression.txt"));
        HK_TEST_SUCCESS(fs.remove("testCompression2.gz"));

        // Unregister callbacks with file system
        fs.unregisterOpenWriterCb(hkFileSystemCallback::compress, &extensions);
        fs.unregisterOpenReaderCb(hkFileSystemCallback::decompress);
    }

    hkStringBuf originalBuffer = "Hello File System\n\nFile System\n";

    {
        {
            hkRefPtr<hkStreamWriter> sw = fs.openWriter("testFileSystem.txt", hkFileSystem::OPEN_DEFAULT_WRITE);
            if(HK_TEST(sw))
                HK_TEST(sw->write(originalBuffer.cString(), originalBuffer.getLength()) == originalBuffer.getLength());
        }
        {
            hkRefPtr<hkStreamWriter> sw = fs.openWriter("testFileSystem2.txt", hkFileSystem::OPEN_DEFAULT_WRITE);
            if(HK_TEST(sw))
                HK_TEST(sw->write(originalBuffer.cString(), originalBuffer.getLength()) == originalBuffer.getLength());
        }
        {
            hkRefPtr<hkStreamWriter> sw = fs.openWriter("testFileSystem3.txt", hkFileSystem::OPEN_DEFAULT_WRITE);
            if(HK_TEST(sw))
                HK_TEST(sw->write(originalBuffer.cString(), originalBuffer.getLength()) == originalBuffer.getLength());
        }
    }

    // Test list directory
    {
        HK_TEST(fileFound(fs, "testFileSystem.txt", "", "*.txt"));
        HK_TEST(fileFound(fs, "testFileSystem2.txt", "", HK_NULL));
        HK_TEST(!fileFound(fs, "testFileSystem2.txt", "", "*.doesntexist"));
        HK_TEST(fileFound(fs, "testFileSystem3.txt", "", "*.txt"));
    }

    
#if defined(HK_PLATFORM_WIN32) || defined(HK_PLATFORM_MAC) || defined(HK_PLATFORM_LINUX) || defined(HK_PLATFORM_WINRT)

    // Test rename file
    // moveFile( file, newFile )
    {
        hkResult moveResult = fs.moveFile("testFileSystem.txt", "testFileSystemRenamed.txt");
        if(!moveResult.isExactly(HK_E_NOT_IMPLEMENTED)) // Some platforms may not implement this
        {
            if (HK_TEST_SUCCESS(moveResult))
            {
                hkFileSystem::Entry entry;
                HK_TEST(fs.stat("testFileSystem.txt", entry).isFailure());
                HK_TEST_SUCCESS(fs.stat("testFileSystemRenamed.txt", entry));
                HK_TEST(entry.isFile());

                // Restore for next test(s)
                HK_TEST_SUCCESS(fs.moveFile("testFileSystemRenamed.txt", "testFileSystem.txt"));
            }
        }

    }

    // Test move file
    // moveFile( file, path )
    {
        if (HK_TEST_SUCCESS(fs.mkdir("subdir", hkFileSystem::ALLOW_EXISTING)))
        {
            hkResult moveResult = fs.moveFile("testFileSystem.txt", "subdir");
            if (!moveResult.isExactly(HK_E_NOT_IMPLEMENTED)) // Some platforms may not implement this
            {
                if (HK_TEST_SUCCESS(moveResult))
                {
                    hkFileSystem::Entry entry;
                    HK_TEST(fs.stat("testFileSystem.txt", entry).isFailure());
                    HK_TEST_SUCCESS(fs.stat("subdir/testFileSystem.txt", entry));
                    HK_TEST(entry.isFile());

                    // Restore for next test(s)
                    HK_TEST_SUCCESS(fs.moveFile("subdir/testFileSystem.txt", "testFileSystem.txt"));
                }
            }

            // Restore for next test(s)
            HK_TEST_SUCCESS(fs.remove("subdir"));
        }
    }

    // Test move + rename file
    // moveFile( file, path/newFile )
    {
        if (HK_TEST_SUCCESS(fs.mkdir("subdir", hkFileSystem::ERROR_IF_EXISTS)))
        {
            hkResult moveResult = fs.moveFile("testFileSystem.txt", "subdir/testFileSystemMoved.txt");
            if (!moveResult.isExactly(HK_E_NOT_IMPLEMENTED)) // Some platforms may not implement this
            {
                if (HK_TEST_SUCCESS(moveResult))
                {
                    hkFileSystem::Entry entry;
                    HK_TEST(fs.stat("testFileSystem.txt", entry).isFailure());
                    HK_TEST_SUCCESS(fs.stat("subdir/testFileSystemMoved.txt", entry));
                    HK_TEST(entry.isFile());

                    // Restore for next test(s)
                    HK_TEST_SUCCESS(fs.moveFile("subdir/testFileSystemMoved.txt", "testFileSystem.txt"));
                }
            }

            // Restore for next test(s)
            HK_TEST_SUCCESS(fs.remove("subdir"));
        }
    }

#endif

    // Test simple read
    {
        hkRefPtr<hkStreamReader> sr = fs.openReader("testFileSystem.txt", hkFileSystem::OPEN_DEFAULT_READ);

        if(HK_TEST(sr))
        {
            char readBuffer[1024];
            HK_TEST(sr->read(readBuffer, 17) == 17);
            readBuffer[17] = 0;
            HK_TEST(!hkString::strCmp(readBuffer, "Hello File System"));
            HK_TEST(sr->read(readBuffer, 1024) == 14);
            readBuffer[14] = 0;
            HK_TEST(!hkString::strCmp(readBuffer, "\n\nFile System\n"));
        }
    }
    HK_TEST_SUCCESS(fs.remove("testFileSystem.txt"));

    // Test list directory after file removed
    {
        HK_TEST(!fileFound(fs, "testFileSystem.txt", "", HK_NULL));
        HK_TEST(fileFound(fs, "testFileSystem2.txt", "", "*.txt"));
        HK_TEST(fileFound(fs, "testFileSystem3.txt", "", "*.txt"));
    }

    // Test buffered read
    {
        hkRefPtr<hkStreamReader> containingSr = fs.openReader("testFileSystem2.txt", hkFileSystem::OPEN_DEFAULT_READ);
        if(HK_TEST(containingSr))
        {
            if (hkSeekableStreamReader* sr = containingSr->isSeekTellSupported())
            {
                char readBuffer[1024];
                HK_TEST(sr->peek(readBuffer, 1) == 1); // Peek 'H'
                HK_TEST(readBuffer[0] == 'H');
                HK_TEST(sr->read(readBuffer, 6) == 6); // Read over "Hello "
                readBuffer[6] = 0;
                HK_TEST(!hkString::strCmp(readBuffer, "Hello "));
                HK_TEST(sr->peek(readBuffer, 4) == 4); // Peek "File"
                readBuffer[4] = 0;
                HK_TEST(!hkString::strCmp(readBuffer, "File"));
                // Seek back to start
                HK_TEST_SUCCESS(sr->seek(0, hkSeekableStreamReader::STREAM_SET));
                HK_TEST(sr->peek(readBuffer, 5) == 5); // Peek "Hello"
                readBuffer[5] = 0;
                HK_TEST(!hkString::strCmp(readBuffer, "Hello"));
                HK_TEST(sr->read(readBuffer, 17) == 17); // Read over "Hello File System"
                readBuffer[17] = 0;
                HK_TEST(!hkString::strCmp(readBuffer, "Hello File System"));
                HK_TEST(sr->peek(readBuffer, 1024) == 14); // Peek to end of file
                readBuffer[14] = 0;
                HK_TEST(!hkString::strCmp(readBuffer, "\n\nFile System\n"));

                HK_TEST(sr->isOk()); // Peek has not finished file

                HK_TEST(sr->read(readBuffer, 1024) == 14); // Read to end of file
                readBuffer[14] = 0;
                HK_TEST(!hkString::strCmp(readBuffer, "\n\nFile System\n"));

                HK_TEST_SUCCESS(sr->seek(-7, hkSeekableStreamReader::STREAM_END)); // Seek back a few bytes
                HK_TEST(sr->read(readBuffer, 1024) == 7); // Read to end of file
                readBuffer[7] = 0;
                HK_TEST(!hkString::strCmp(readBuffer, "System\n"));

                HK_TEST(sr->peek(readBuffer, 1024) == 0); // Nothing left to peek
                HK_TEST(!sr->isOk()); // Peek finishes the file

                HK_TEST(sr->read(readBuffer, 1024) == 0); // Nothing left to read
                HK_TEST(!sr->isOk()); // File finished
            }
        }
    }

    HK_TEST_SUCCESS(fs.remove("testFileSystem2.txt"));

    // Test open append
    {
        hkRefPtr<hkStreamWriter> sw = fs.openWriter("testFileSystem3.txt", hkFileSystem::OpenFlags(hkFileSystem::OPEN_DEFAULT_WRITE & (~hkFileSystem::OPEN_TRUNCATE)));
        if(HK_TEST(sw))
        {
            sw->seek(0, hkStreamWriter::STREAM_END);
            hkStringBuf appendText("Appended to file\n");
            sw->write(appendText.cString(), appendText.getLength());
            sw = HK_NULL;
        }

        char readBuffer[1024];

        hkRefPtr<hkStreamReader> sr = fs.openReader("testFileSystem3.txt", hkFileSystem::OPEN_DEFAULT_READ);
        if(HK_TEST(sr))
        {
            HK_TEST(sr->peek(readBuffer, 1024) == 48); // Peek all of file
            readBuffer[48] = 0;
            HK_TEST(!hkString::strCmp(readBuffer, "Hello File System\n\nFile System\nAppended to file\n"));

            HK_TEST(sr->read(readBuffer, 1024) == 48); // Read to end of file
            readBuffer[48] = 0;
            HK_TEST(!hkString::strCmp(readBuffer, "Hello File System\n\nFile System\nAppended to file\n"));
        }
    }

    HK_TEST_SUCCESS(fs.remove("testFileSystem3.txt"));

    // Test list directory after file removed
    {
        HK_TEST(!fileFound(fs, "testFileSystem.txt", "", HK_NULL));
        HK_TEST(!fileFound(fs, "testFileSystem2.txt", "", HK_NULL));
        HK_TEST(!fileFound(fs, "testFileSystem3.txt", "", HK_NULL));
    }

    // Test directory iterator on known files
    {
        const char Resources_Common_FileSystem[] = "Resources/Common/FileSystem";
        {
            hkFileSystem::Entry entry;
            HK_TEST2(fs.stat(Resources_Common_FileSystem, entry).isSuccess(), "Folder  missing (missing files from deploy?)");
            HK_TEST(entry.isDir());
        }

        hkFileSystem::Iterator dirIter(&fs, Resources_Common_FileSystem);
        hkArray<hkFileSystem::Entry> files;
        while(dirIter.advance())
        {
            hkFileSystem::Entry entry(dirIter.current());
            if(entry.isFile())
            {
                files.pushBack(entry);
            }
            else if(entry.isDir())
            {
                dirIter.recurseInto(entry.getPath());
            }
        }

        // This dir should contain 3 files, of size 11. Each file contents is equal to its name
        HK_TEST(files.getSize() == 3);
        while(files.getSize())
        {
            hkFileSystem::Entry file = files.back();
            hkRefPtr<hkStreamReader> sr = fs.openReader(file.getPath(), hkFileSystem::OPEN_DEFAULT_READ);
            HK_TEST(file.getSize() == 11);
            HK_TEST(file.isFile());
            char readBuffer[1024];
            HK_TEST(sr->read(readBuffer, 1024) == 11); // Peek all of file
            readBuffer[11] = 0;
            HK_TEST(!hkString::strCmp(readBuffer, file.getName()));
            files.popBack();
        }
    }

    // Test seek past the end of the file fills with zero
    {
        hkStringBuf shortString("In File");
        {
            hkRefPtr<hkStreamWriter> sw = fs.openWriter("testSeekPastEnd.txt", hkFileSystem::OPEN_DEFAULT_WRITE);
            if(sw)
            {
                HK_TEST(sw->write(shortString.cString(), shortString.getLength()) == shortString.getLength());

                // Seek past the end of the file should fill with zeros
                HK_TEST_SUCCESS(sw->seek(50, hkStreamWriter::STREAM_SET));
                HK_TEST(sw->write(shortString.cString(), shortString.getLength()) == shortString.getLength());
            }
        }

        char readBuffer[1024];
        // First string
        {
            hkRefPtr<hkStreamReader> sr = fs.openReader("testSeekPastEnd.txt", hkFileSystem::OPEN_DEFAULT_READ);

            if(HK_TEST(sr))
            {
                HK_TEST(sr->read(readBuffer, 7) == 7);
                readBuffer[7] = 0;
                HK_TEST(!hkString::strCmp(readBuffer, "In File"));

                HK_TEST(sr->read(readBuffer, 43) == 43);

                // Xbox 360 does not initialize the intervening bytes
                // The filled in bytes should all be zero
                for(int i=0;i<43;i++)
                {
                    HK_TEST(readBuffer[i] == 0);
                }

                // Then the write occurs after the seek
                HK_TEST(sr->read(readBuffer, 1024) == 7);
                readBuffer[7] = 0;
                HK_TEST(!hkString::strCmp(readBuffer, "In File"));
            }
        }

        HK_TEST_SUCCESS(fs.remove("testSeekPastEnd.txt"));
    }

// Server file system doesn't implement HK_E_ALREADY_EXISTS
#if !defined(HK_PLATFORM_DURANGO) && !defined(HK_PLATFORM_PS4) && !defined(HK_PLATFORM_ANDROID) && !defined(HK_PLATFORM_IOS)
    // Test mkdir and hkResult toString
    {
        hkResult r = fs.mkdir("TestDir", hkFileSystem::ALLOW_EXISTING);
        hkStringBuf sb;
        r.toString(sb);
        HK_TEST_SUCCESS(r);
        HK_TEST(hkString::strCmp(sb.cString(), "HK_SUCCESS") == 0);

        r = fs.mkdir("TestDir", hkFileSystem::ERROR_IF_EXISTS);
        r.toString(sb);
        HK_TEST(r.isExactly(HK_E_ALREADY_EXISTS));
        HK_TEST(hkString::strCmp(sb.cString(), "HK_E_ALREADY_EXISTS") == 0);

        HK_TEST_SUCCESS(fs.remove("TestDir"));
    }
#endif

    return 0;
}

HK_TEST_REGISTER(fileops_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__ );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
