// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/System/Log/hkLog.h>
#include <Common/Base/Serialize/hkSerialize.h>

#define DEBUG_LOG_IDENTIFIER "test.hkLog"
#include <Common/Base/System/Log/hkLog.hxx>

template<typename T>
static void use(T) {}

static void test_dlog_2()
{
    // log with scope
    Log_Scope(Dev,"Dev Scope Used1");
    Log_Auto("This one shows up {}", 44);
    Log_Scope(Dev, "Dev Scope Used2");
    Log_Info("Info message");

    {
        // captures start their own block & set of autos
        UnitTest::Raii::CaptureThreadLogOutput threadOutput;
        Log_Dev("INNER Dev1");
        Log_Auto("INNER Should show in captured");
        Log_Dev("INNER Dev2");
        hkStringBuf sb;
        const char* txt = threadOutput.getText(sb);
        HK_TEST(hkString::strStr(txt, "Should"));
    }

    // info, then dev
    {
        Log_Auto("Nested Scope 1");
        Log_Info("Info 1");
        Log_Auto("Nested Scope 2");
        Log_Dev("Dev 2");
        Log_Auto("Auto Scope Unused2");
    }
    // dev then info
    {
        Log_Auto("Nested Scope 1");
        Log_Dev("Dev 2");
        Log_Auto("Nested Scope 2");
        Log_Info("Info 1a");
        Log_Info("Info 1b");
        Log_Auto("Auto Scope Unused2");
    }
    Log_Dev("Same level as Info message");
}

static void test_dlog_1()
{
    Log_Dev("Dev1");

    // log with scope
    Log_Scope(Dev, "Dev Scope Used1");
    Log_Auto("This one shows up {}", 44);
    Log_Scope(Dev, "Dev Scope Used2");
    Log_Dev("Dev1");
    Log_Auto("But not this one because the next message is skipped");
    // there is no shortcut for MaxLevel, write it longhand
    hkLog_Log(DEBUG_LOG_OBJECT, hkLog::Level::MaxLevel, "test");

    // should not show up as nothing is inside the auto
    Log_Auto("Auto Scope Unused1");
    Log_Auto("Auto Scope Unused2");
}

static void test_warn_1()
{
    Log_Warning("Warning");
}

static void testStopPropagation(bool stop)
{
    using namespace hkLog;

    // stop output at the file level (to avoid polluting the global log)
    UnitTest::LogStopT< TextOutput > stopOutput;
    Raii::AddOutput raii(DEBUG_LOG_OBJECT.accessOutput(), &stopOutput);

    // we toggle this one for the test
    UnitTest::Raii::CaptureThreadLogOutput threadOutput;
    threadOutput.m_stopMessage = stop;

    // Log a message
    test_warn_1();

    hkStringBuf sb1;
    HK_TEST( hkString::strStr(threadOutput.getText(sb1), "Warning") );

    hkStringBuf sb2;
    const char* txt2 = stopOutput.getText(sb2);
    if(stop)
    {
        HK_TEST_EQ(txt2, (const char*)HK_NULL);
    }
    else
    {
        HK_TEST( hkString::strStr(txt2, "Warning") );
    }
}

static void testAddRemove()
{
    hkLog::Output* o1 = hkLog::connectToFile("ReadFormat", "log.txt", hkLog::Level::Dev);
    hkLog::Output* o2 = hkLog::connectToFile("s11n", "log.txt", hkLog::Level::Warning);
    HK_TEST(o1==o2);

    hkLog::disconnect("ReadFormat", o1);
    hkLog::disconnect("s11n", o2);
}

static int testCallCount()
{
    static int callCount = 0;
    ++callCount;
    return callCount;
}

static int log_main()
{
    using namespace hkLog;
    {
        // check that we get nested debug logs
        UnitTest::Raii::CaptureThreadLogOutput threadOutput;
        Raii::SetLevel level(DEBUG_LOG_OBJECT, Level::Debug);
        test_dlog_1();
        hkStringBuf sb;
        const char* txt = threadOutput.getText(sb);
        if(HK_TEST(txt))
        {
            HK_TEST(hkString::strStr(txt,"Used1") != HK_NULL);
            HK_TEST(hkString::strStr(txt, "Used2") != HK_NULL);
            HK_TEST(hkString::strStr(txt, "Unused1") == HK_NULL);
            HK_TEST(hkString::strStr(txt, "Unused2") == HK_NULL);
        }
    }
    {
        // check that we get nested debug logs
        UnitTest::Raii::CaptureThreadLogOutput threadOutput;
        Raii::SetLevel level(DEBUG_LOG_OBJECT, Level::Debug);
        test_dlog_2();
        hkStringBuf sb;
        const char* txt = threadOutput.getText(sb);
        if (HK_TEST(txt))
        {
            HK_TEST(hkString::strStr(txt, "INNER")==nullptr);
        }
    }
    {
        // check that we don't get output if the level is too low
        UnitTest::Raii::CaptureThreadLogOutput threadOutput;
        test_dlog_1();
        hkStringBuf sb;
        const char* txt = threadOutput.getText(sb);
        HK_TEST_EQ(txt, (const char*)HK_NULL);
    }

    {
        testStopPropagation(true);
        // testStopPropagation(false); // this pollutes the test log
    }

//  {   
//      UnitTest::LogStopT< hkLog::TextOutput > globalOut;
//      Raii::AddOutput raii(hkLog::accessGlobalOutput(), &globalOut);
//      test_warn_1();
//      hkStringBuf sb;
//      HK_TEST(hkString::strStr(globalOut.getText(sb), "Warning"));
//  }

    {
        hkStringBuf sb;
        for(hkLog::OriginIterator it; it.advance();)
        {
            if(const hkLog::RegisteredOrigin* cur = it.current())
            {
                sb.appendJoin(cur->getName(), "\n");
            }
        }
        sb.clear();
    }

    testAddRemove();

    if(0)
    {
        
        hkLog::connectToFile("ReadFormat", "log.txt", hkLog::Level::Dev);
        hkLog::connectToFile("s11n", "log.txt", hkLog::Level::Warning);
        

        typedef int Foo;
        
        hkLog::Raii::CaptureOutput message;
        if(Foo* f = hkSerialize::Load().toObject<Foo>("foo.hkt"))
        {
            // ok, use f
        }
        else
        {
            hkStringBuf buf;
            const char* whyNot = message.getText(buf);
            use(whyNot);
        }
        

        
        for(hkLog::OriginIterator it; it.advance();)
        {
            use(it.current()->getName());
        }
        
    }

    // test that function calls are not duplicated by macro expansion
    {
        DEBUG_LOG_OBJECT.setLevel( hkLog::Level::Info );
        const int callCountBefore = testCallCount(); // 1
        Log_Info( "Test call count: {}", testCallCount() ); // 2
        const int callCountAfter = testCallCount(); // 3
        HK_TEST( callCountAfter - callCountBefore == 2 );
    }

    // test safety of use with c/c++ block
    // e.g. with an "if" without curly braces, check that the if() applies to the macros as a whole
    {
        const int callCountBefore = testCallCount(); // 4
        if ( callCountBefore == -42 ) Log_Info( "Test call count: {}", testCallCount() ); // -
        const int callCountAfter = testCallCount(); // 5
        HK_TEST( callCountAfter - callCountBefore == 1 );
    }

    return 0;
}

HK_TEST_REGISTER( log_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__ );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
