// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Container/String/hkUtf8.h>
#include <Common/Base/Container/String/hkUtf8Detail.h>
#include <wchar.h>

static const hkUint8 englishUtf8[] = { 0x45, 0x6e, 0x67, 0x6c, 0x69, 0x73, 0x68, 0x0 };
static const hkUint16 englishUtf16[] = { 0x45, 0x6e, 0x67, 0x6c, 0x69, 0x73, 0x68, 0x0 };
static const hkUint32 englishUtf32[] = { 0x45, 0x6e, 0x67, 0x6c, 0x69, 0x73, 0x68, 0x0 };

static const hkUint8 russianUtf8[] = { 0xd0, 0xa0, 0xd1, 0x83, 0xd1, 0x81, 0xd1, 0x81, 0xd0, 0xba, 0xd0, 0xb8, 0xd0, 0xb9, 0x0 };
static const hkUint16 russianUtf16[] = { 0x420, 0x443, 0x441, 0x441, 0x43a, 0x438, 0x439, 0x0 };
static const hkUint32 russianUtf32[] = { 0x420, 0x443, 0x441, 0x441, 0x43a, 0x438, 0x439, 0x0 };

static const hkUint8 japaneseUtf8[] = { 0xe6, 0x97, 0xa5, 0xe6, 0x9c, 0xac, 0xe8, 0xaa, 0x9e, 0x0 };
static const hkUint16 japaneseUtf16[] = { 0x65e5, 0x672c, 0x8a9e, 0x0 };
static const hkUint32 japaneseUtf32[] = { 0x65e5, 0x672c, 0x8a9e, 0x0 };

static const hkUint8 chineseUtf8[] = { 0xe4, 0xb8, 0xad, 0xe6, 0x96, 0x87, 0x0 };
static const hkUint16 chineseUtf16[] = { 0x4e2d, 0x6587, 0x0 };
static const hkUint32 chineseUtf32[] = { 0x4e2d, 0x6587, 0x0 };

static const hkUint8 mathUtf8[] = { 0xf0, 0x9d, 0x95, 0x84, 0xf0, 0x9d, 0x94, 0xb8, 0xf0, 0x9d, 0x95, 0x8b, 0xe2, 0x84, 0x8d, 0x0 };
static const hkUint16 mathUtf16[] = { 0xd835, 0xdd44, 0xd835, 0xdd38, 0xd835, 0xdd4b, 0x210d, 0x0 };
static const hkUint32 mathUtf32[] = { 0x1d544, 0x1d538, 0x1d54b, 0x210d, 0x0 };

static const hkUint8 mixedUtf8[] = { 0xf0, 0x9d, 0x95, 0x84, 0x20, 0x45, 0x6e, 0x67, 0x20, 0xe4, 0xb8, 0xad, 0x20, 0xd0, 0xa0, 0xd1, 0x83, 0xd1, 0x81, 0xf0, 0x9d, 0x94, 0xb8, 0xd1, 0x81, 0xd0, 0xba, 0xd0, 0xb8, 0xd0, 0xb9, 0x20, 0x6c, 0x69, 0x73, 0x68, 0x20, 0xe6, 0x97, 0xa5, 0xe6, 0x9c, 0xac, 0xe8, 0xaa, 0x9e, 0x20, 0xf0, 0x9d, 0x95, 0x8b, 0xe2, 0x84, 0x8d, 0x0 };
static const hkUint16 mixedUtf16[] = { 0xd835, 0xdd44, 0x20, 0x45, 0x6e, 0x67, 0x20, 0x4e2d, 0x20, 0x420, 0x443, 0x441, 0xd835, 0xdd38, 0x441, 0x43a, 0x438, 0x439, 0x20, 0x6c, 0x69, 0x73, 0x68, 0x20, 0x65e5, 0x672c, 0x8a9e, 0x20, 0xd835, 0xdd4b, 0x210d, 0x0 };
static const hkUint32 mixedUtf32[] = { 0x1d544, 0x20, 0x45, 0x6e, 0x67, 0x20, 0x4e2d, 0x20, 0x420, 0x443, 0x441, 0x1d538, 0x441, 0x43a, 0x438, 0x439, 0x20, 0x6c, 0x69, 0x73, 0x68, 0x20, 0x65e5, 0x672c, 0x8a9e, 0x20, 0x1d54b, 0x210d, 0x0 };

template <typename T>
static bool strEq(const T* a, const T* b)
{
    while(true)
    {
        if(*a != *b)
        {
            return false;
        }
        else if(*a == 0)
        {
            return true;
        }

        a++;
        b++;
    }
}

template <typename T>
static bool isPrefix(const T* a, const T* b)
{
    while(true)
    {
        if(*a == 0)
        {
            return true;
        }

        if(*a != *b)
        {
            return false;
        }

        a++;
        b++;
    }
}

template <typename T>
static int strLen(const T* str)
{
    int ret = 0;
    while(str[ret])
    {
        ret++;
    }
    return ret;
}

// hkUtf8
namespace
{
    // hkUtf8::strLen()
    void test_strLen()
    {
        HK_TEST( hkUtf8::strLen( HK_NULL ) == 0 );
        HK_TEST( hkUtf8::strLen( "" ) == 0 );
        HK_TEST( hkUtf8::strLen( "test" ) == 4 );
        HK_TEST( hkUtf8::strLen( (const char*)englishUtf8 ) == 7 );
        HK_TEST( hkUtf8::strLen( (const char*)russianUtf8 ) == 7 );
        HK_TEST( hkUtf8::strLen( (const char*)japaneseUtf8 ) == 3 );
        HK_TEST( hkUtf8::strLen( (const char*)chineseUtf8 ) == 2 );
        HK_TEST( hkUtf8::strLen( (const char*)mathUtf8 ) == 4 );
        HK_TEST( hkUtf8::strLen( (const char*)mixedUtf8 ) == 28 );
    }

    // hkUtf8::utf8FromCodePoint()
    void test_utf8FromCodePoint()
    {
        char dst[6];

        // 0x0 - 0x7F
        for ( hkUint32 cp = 0u; cp <= 0x7Fu; ++cp )
        {
            HK_TEST( hkUtf8::utf8FromCodePoint( dst, cp ) == 1 );
            HK_TEST( hkUint8(dst[0]) == cp );
        }

        // 0x80 - 0x7FF
        for ( hkUint32 cp = 0x80u; cp <= 0x7FFu; ++cp )
        {
            HK_TEST( hkUtf8::utf8FromCodePoint( dst, cp ) == 2 );
            HK_TEST( hkUint8(dst[0] & 0x1F) == (cp >> 6) );
            HK_TEST( hkUint8(dst[1] & 0x3F) == (cp & 0x3F) );
        }

        // 0x800 - 0xFFFF
        for ( hkUint32 cp = 0x800u; cp <= 0xFFFFu; ++cp )
        {
            HK_TEST( hkUtf8::utf8FromCodePoint( dst, cp ) == 3 );
            HK_TEST( hkUint8(dst[0] & 0x0F) == (cp >> 12) );
            HK_TEST( hkUint8(dst[1] & 0x3F) == ((cp >> 6) & 0x3F) );
            HK_TEST( hkUint8(dst[2] & 0x3F) == (cp & 0x3F) );
        }

        // 0x10000 - 0x1FFFFF
        for ( hkUint32 cp = 0x10000u; cp <= 0x1FFFFFu; ++cp )
        {
            HK_TEST( hkUtf8::utf8FromCodePoint( dst, cp ) == 4 );
            HK_TEST( hkUint8(dst[0] & 0x07) == (cp >> 18) );
            HK_TEST( hkUint8(dst[1] & 0x3F) == ((cp >> 12) & 0x3F) );
            HK_TEST( hkUint8(dst[2] & 0x3F) == ((cp >> 6) & 0x3F) );
            HK_TEST( hkUint8(dst[3] & 0x3F) == (cp & 0x3F) );
        }
    }
}

static void testUtf8FromUtf16(const hkUint16* utf16, const hkUint8* refUtf8)
{
    int refLen = strLen(refUtf8) + 1;

    {
        char utf8[64];
        int len = hkUtf8::Detail::utf8FromUtf16(utf8, 64, utf16);
        HK_TEST(strEq(utf8, (const char*)refUtf8));
        HK_TEST(len == refLen);
    }

    {
        char utf8[64];
        int len = hkUtf8::Detail::utf8FromUtf16(utf8, refLen / 2, utf16);
        HK_TEST(isPrefix(utf8, (const char*)refUtf8));
        HK_TEST(len == refLen);
    }

    {
        int len = hkUtf8::Detail::utf8LengthForUtf16(utf16);
        HK_TEST(len == refLen);
    }
}

static void testUtf16FromUtf8(const hkUint8* utf8, const hkUint16* refUtf16)
{
    int refLen = strLen(refUtf16) + 1;

    {
        hkUint16 utf16[64];
        int len = hkUtf8::Detail::utf16FromUtf8(utf16, 64, (const char*)utf8);
        HK_TEST(strEq(utf16, refUtf16));
        HK_TEST(len == refLen);
    }

    {
        hkUint16 utf16[64];
        int len = hkUtf8::Detail::utf16FromUtf8(utf16, refLen / 2, (const char*)utf8);
        HK_TEST(isPrefix(utf16, refUtf16));
        HK_TEST(len == refLen);
    }

    {
        int len = hkUtf8::Detail::utf16LengthForUtf8((const char*)utf8);
        HK_TEST(len == refLen);
    }
}


static void testUtf8FromUtf32(const hkUint32* utf32, const hkUint8* refUtf8)
{
    int refLen = strLen(refUtf8) + 1;

    {
        char utf8[64];
        int len = hkUtf8::Detail::utf8FromUtf32(utf8, 64, utf32);
        HK_TEST(strEq(utf8, (const char*)refUtf8));
        HK_TEST(len == refLen);
    }

    {
        char utf8[64];
        int len = hkUtf8::Detail::utf8FromUtf32(utf8, refLen / 2, utf32);
        HK_TEST(isPrefix(utf8, (const char*)refUtf8));
        HK_TEST(len == refLen);
    }

    {
        int len = hkUtf8::Detail::utf8LengthForUtf32(utf32);
        HK_TEST(len == refLen);
    }
}

static void testUtf32FromUtf8(const hkUint8* utf8, const hkUint32* refUtf32)
{
    int refLen = strLen(refUtf32) + 1;

    {
        hkUint32 utf32[64];
        int len = hkUtf8::Detail::utf32FromUtf8(utf32, 64, (const char*)utf8);
        HK_TEST(strEq(utf32, refUtf32));
        HK_TEST(len == refLen);
    }

    {
        hkUint32 utf32[64];
        int len = hkUtf8::Detail::utf32FromUtf8(utf32, refLen / 2, (const char*)utf8);
        HK_TEST(isPrefix(utf32, refUtf32));
        HK_TEST(len == refLen);
    }

    {
        int len = hkUtf8::Detail::utf32LengthForUtf8((const char*)utf8);
        HK_TEST(len == refLen);
    }
}

static void testTemp(const wchar_t* wide, const char* utf8)
{
    HK_TEST(strEq(hkUtf8::Utf8FromWide(wide).cString(), utf8));
    HK_TEST(strEq(hkUtf8::WideFromUtf8(utf8).cString(), wide));
}

static void testValidateUtf8()
{
    HK_TEST(hkUtf8::Detail::findFirstInvalidUtf8((const char*)englishUtf8) == -1);

    const char strayTail[] = "ab\xcf\x84\x84""d";
    HK_TEST(hkUtf8::Detail::findFirstInvalidUtf8(strayTail) == 4);

    const char overlong[] = "ps\xe0\x85\xb3";
    HK_TEST(hkUtf8::Detail::findFirstInvalidUtf8(overlong) == 2);
}

int utf8_main()
{
    test_strLen();
    test_utf8FromCodePoint();

    testUtf8FromUtf16(englishUtf16, englishUtf8);
    testUtf8FromUtf16(russianUtf16, russianUtf8);
    testUtf8FromUtf16(japaneseUtf16, japaneseUtf8);
    testUtf8FromUtf16(chineseUtf16, chineseUtf8);
    testUtf8FromUtf16(mathUtf16, mathUtf8);
    testUtf8FromUtf16(mixedUtf16, mixedUtf8);

    testUtf16FromUtf8(englishUtf8, englishUtf16);
    testUtf16FromUtf8(russianUtf8, russianUtf16);
    testUtf16FromUtf8(japaneseUtf8, japaneseUtf16);
    testUtf16FromUtf8(chineseUtf8, chineseUtf16);
    testUtf16FromUtf8(mathUtf8, mathUtf16);
    testUtf16FromUtf8(mixedUtf8, mixedUtf16);

    testUtf8FromUtf32(englishUtf32, englishUtf8);
    testUtf8FromUtf32(russianUtf32, russianUtf8);
    testUtf8FromUtf32(japaneseUtf32, japaneseUtf8);
    testUtf8FromUtf32(chineseUtf32, chineseUtf8);
    testUtf8FromUtf32(mathUtf32, mathUtf8);
    testUtf8FromUtf32(mixedUtf32, mixedUtf8);

    testUtf32FromUtf8(englishUtf8, englishUtf32);
    testUtf32FromUtf8(russianUtf8, russianUtf32);
    testUtf32FromUtf8(japaneseUtf8, japaneseUtf32);
    testUtf32FromUtf8(chineseUtf8, chineseUtf32);
    testUtf32FromUtf8(mathUtf8, mathUtf32);
    testUtf32FromUtf8(mixedUtf8, mixedUtf32);

#if WCHAR_MAX == 0xffff
    testTemp((const wchar_t*)englishUtf16, (const char*)englishUtf8);
    testTemp((const wchar_t*)russianUtf16, (const char*)russianUtf8);
    testTemp((const wchar_t*)japaneseUtf16, (const char*)japaneseUtf8);
    testTemp((const wchar_t*)chineseUtf16, (const char*)chineseUtf8);
    testTemp((const wchar_t*)mathUtf16, (const char*)mathUtf8);
    testTemp((const wchar_t*)mixedUtf16, (const char*)mixedUtf8);
#else
    HK_COMPILE_TIME_ASSERT(WCHAR_MAX == 0xffffffff || WCHAR_MAX == 0x7fffffff);
    testTemp((const wchar_t*)englishUtf32, (const char*)englishUtf8);
    testTemp((const wchar_t*)russianUtf32, (const char*)russianUtf8);
    testTemp((const wchar_t*)japaneseUtf32, (const char*)japaneseUtf8);
    testTemp((const wchar_t*)chineseUtf32, (const char*)chineseUtf8);
    testTemp((const wchar_t*)mathUtf32, (const char*)mathUtf8);
    testTemp((const wchar_t*)mixedUtf32, (const char*)mixedUtf8);
#endif

    testValidateUtf8();

    return 0;
}

HK_TEST_REGISTER( utf8_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__ );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
