// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Container/Set/hkSet.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Container/Hash/hkHashSet.h>

namespace SetTest
{
    template<typename K>
    struct Set : public hkSet<K> { Set() {} };
}

template<class TypeOfSetElement, template<typename E> class SET_TYPE >
struct CopyCtorTester
{
    static void exec( const SET_TYPE<TypeOfSetElement>& set, const hkArray<TypeOfSetElement>& testArray, int numKeys )
    {
        typedef SET_TYPE<TypeOfSetElement> TestSet;

        TestSet set2;
        set2 = set;

        HK_TEST( set.getSize() == set2.getSize() );

        for ( int i = 0; i < numKeys; ++i )
        {
            // Ensure elements are in the copied map
            HK_TEST( set2.contains( testArray[i] ) );
            typename TestSet::Iterator iter2 = set2.findElement( testArray[i] );
            HK_TEST( set2.isValid( iter2 ) );
            const TypeOfSetElement& elem2 = set2.getElement( iter2 );
            HK_TEST( elem2 == testArray[i] );

            // Ensure elements are copies (have different address from original set)
            typename TestSet::Iterator iter = set.findElement( testArray[i] );
            HK_TEST( set.isValid( iter ) );
            HK_TEST( &elem2 != &set.getElement( iter ) );
        }
    }
};

// Specialization for hkSet, because getElement() returns a copy instead of a const reference.
template <class TypeOfSetElement>
struct CopyCtorTester<TypeOfSetElement, SetTest::Set>
{
    static void exec( const SetTest::Set<TypeOfSetElement>& set, const hkArray<TypeOfSetElement>& testArray, int numKeys )
    {
        typedef SetTest::Set<TypeOfSetElement> TestSet;

        TestSet set2;
        set2 = set;

        HK_TEST( set.getSize() == set2.getSize() );

        for ( int i = 0; i < numKeys; ++i )
        {
            // Ensure elements are in the copied map
            HK_TEST( set2.contains( testArray[i] ) );
            typename TestSet::Iterator iter2 = set2.findElement( testArray[i] );
            HK_TEST( set2.isValid( iter2 ) );
            const TypeOfSetElement& elem2 = set2.getElement( iter2 );
            HK_TEST( elem2 == testArray[i] );
        }

        // Ensure copy ctor made an actual copy
        // Hack around getElement() returning a copy, so impossible to know from
        // individual elements if they were copied or not.
        class SetTester : public TestSet
        {
        public:
            void test_address( const SetTester& other ) const
            {
                HK_TEST( this->m_elem.begin() != other.m_elem.begin() );
            }
        };
        ((SetTester&)set2).test_address( ((const SetTester&)set) );
    }
};

template<class TypeOfSetElement, template<typename E> class SET_TYPE >
static void test_set()
{
    typedef SET_TYPE<TypeOfSetElement> TestSet;

    hkPseudoRandomGenerator randgen(0);
    int numKeys = randgen.getRandChar(127);


    {
        TestSet set;
        set.reserve( numKeys );
        hkArray<TypeOfSetElement> testArray;

        // insert() / tryInsert()
        for(int i = 0; i < numKeys; ++i)
        {
            TypeOfSetElement elem = (TypeOfSetElement) randgen.getRand32();
            hkBool32 inserted = set.insert( elem );
            HK_TEST( inserted );
            inserted = set.insert( elem );
            HK_TEST( !inserted );
            hkResult res = HK_FAILURE;
            inserted = set.tryInsert( elem, res );
            HK_TEST( !inserted );
            HK_TEST( res.isSuccess() );
            testArray.pushBack( elem );
        }

        // contains() / findElement() + getElement()
        for(int i = 0; i < numKeys; ++i)
        {
            HK_TEST(  set.contains( testArray[i])  );
            typename TestSet::Iterator iter = set.findElement( testArray[i] );
            HK_TEST( set.getElement(iter) == testArray[i] );
        }
        HK_TEST( set.getSize() == testArray.getSize() );

        // Iterator
        {
            hkArray<TypeOfSetElement> testArray2( testArray );

            typename TestSet::Iterator iter = set.getIterator();
            for (; set.isValid(iter); iter = set.getNext(iter) )
            {
                TypeOfSetElement elem = set.getElement(iter);

                int idx = testArray2.indexOf(elem);
                HK_TEST(idx >= 0);
                testArray2.removeAt(idx);
            }
            HK_TEST( testArray2.isEmpty() );
        }

        // Copy construct
        {
            CopyCtorTester<TypeOfSetElement, SET_TYPE>::exec( set, testArray, numKeys );
        }

        // remove()
        for (int i=0; i<numKeys; i++)
        {
            TypeOfSetElement elem = testArray[i];

            HK_TEST( set.contains(elem) );
            hkResult res = set.remove( elem );
            HK_TEST( !set.contains(elem) );
            HK_TEST( res.isSuccess() );
            res = set.remove( elem );
            HK_TEST( res.isFailure() );
        }
    }

}

static void test_shouldResize()
{
    // Check that we can add elements to an hkSet without triggering a resize
    for (int size = 4; size < 256; size++)
    {
        const int stackSize = size;

        int setSize = hkSet<hkUint32>::getSizeInBytesFor(stackSize);
        hkLocalArray<char> setStorage(setSize);

        hkSet<hkUint32> set( setStorage.begin(), setSize );
        const int initialCapacity = set.getCapacity();
        HK_TEST(set.getCapacity() >= stackSize);
        HK_TEST(set.getSize() == 0);

        while( !set.shouldResize() )
        {
            hkUint32 val = set.getSize(); // doesn't matter what we add, as long as it's distinct
            set.insert( val );

            HK_TEST(initialCapacity == set.getCapacity());
        }

        // Make sure we actually added something
        HK_TEST( set.getSize() > 0 );

    }
}

static void test_reduceCapacity()
{
    hkSet<int> set;
    for (int i=0; i<1000; i++)
    {
        set.insert(i);
    }
    int oldCap = set.getCapacity();
    for (int i=10; i<1000; i++)
    {
        set.remove(i);
    }
    set.optimizeCapacity();
    int newCap = set.getCapacity();

    HK_TEST( newCap < oldCap );

    hkArray<int> elems;
    hkSet<int>::Iterator iter;
    for (iter = set.getIterator(); set.isValid(iter); iter = set.getNext(iter) )
    {
        elems.pushBack( set.getElement(iter) );
    }

    hkSort( elems.begin(), elems.getSize() );
    for (int i=0; i<10; i++)
    {
        HK_TEST( i == elems[i] );
    }

    hkSet<int> newSet;
    for (int i=0; i<10; i++)
    {
        newSet.insert(i);
    }

    HK_TEST( newSet.getCapacity() == newCap );

}

int set_main()
{
    test_set<int, SetTest::Set>();
    test_set<int, hkHashSet>();

    test_set<hkUint32, SetTest::Set>();
    test_set<hkUint32, hkHashSet>();
    test_set<hkUint64, SetTest::Set>();
    test_set<hkUint64, hkHashSet>();
    test_set<hkUlong, SetTest::Set>();
    test_set<hkUlong, hkHashSet>();

    test_shouldResize();
    test_reduceCapacity();
    return 0;
}

HK_TEST_REGISTER(set_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__     );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
