// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Container/BitField/hkBitField.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Math/Vector/hkIntVector.h>

#include <Common/Base/UnitTest/Container/BitField.inl>

namespace OffsetBitFieldImpl
{
    static HK_INLINE void HK_CALL setAllBytes(int val, hkIntVector& vOut)
    {
        hkIntVector vInts;      vInts.setAll(val);                      // [0x000000XX, 0x000000XX, 0x000000XX, 0x000000XX]
        hkIntVector vShorts;    vShorts.setShiftLeft32<8>(vInts);       // [0x0000XX00, 0x0000XX00, 0x0000XX00, 0x0000XX00]
                                vShorts.setOr(vShorts, vInts);          // [0x0000XXXX, 0x0000XXXX, 0x0000XXXX, 0x0000XXXX]
                                vInts.setShiftLeft32<16>(vShorts);      // [0xXXXX0000, 0xXXXX0000, 0xXXXX0000, 0xXXXX0000]
        vOut.setOr(vInts, vShorts);                                     // [0xXXXXXXXX, 0xXXXXXXXX, 0xXXXXXXXX, 0xXXXXXXXX]
    }

    static HK_INLINE void HK_CALL setBytes(const hkUint32* HK_RESTRICT intPtr, hkIntVector& vOut)
    {
        hkIntVector vInt32; vInt32.set(intPtr[3], intPtr[2], intPtr[1], intPtr[0]);
        hkIntVector vInt16; vInt16.setCombine8To16(vInt32);
        vOut.setCombine16To32(vInt16);
    }
}

struct SetAscendingIter
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE, SetAscendingIter);
    HK_INLINE SetAscendingIter(hkOffsetBitFieldLocal& bf)
    :   m_bf(bf)
    {}

    HK_INLINE void operator()(int bitIdx)
    {
        m_bf.set(bitIdx + 1);
    }

    hkOffsetBitFieldLocal& m_bf;
};

static void testSetAscendingIterator()
{
    hkOffsetBitFieldLocal bf(128, hkBitFieldValue::UNINITIALIZED);
    SetAscendingIter sai(bf);

    for (int e = 0; e < 127; e++)
    {
        for (int s = 0; s <= e; s++)
        {
            // Set the start bit to 1
            bf.assignAll<0>();
            bf.assign(s, 1);

            // Set all other bits to 1 with the iterator
            bf.forEach<hkBitFieldLoop::ASCENDING, hkBitFieldBit::SET>(s, e - s + 1, sai);

            // Test. Should be one from s to e + 1 inclusive!
            hkIntVector vActual;    OffsetBitFieldImpl::setBytes(bf.getWords(), vActual);
            hkIntVector vDesired;   vDesired.setAll(-1);
            const int numLeftZeros  = s;
            const int numZeros      = (127 - (e + 1)) + numLeftZeros;
            hkIntVector vShr;       OffsetBitFieldImpl::setAllBytes(numZeros, vShr);
            hkIntVector vShl;       OffsetBitFieldImpl::setAllBytes(numLeftZeros, vShl);
            vDesired.setBitShiftRight128(vDesired, vShr);
            vDesired.setBitShiftLeft128(vDesired, vShl);
            HK_TEST(vDesired.compareEqualS32(vActual).allAreSet());
        }
    }
}

struct ClearAscendingIter
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE, ClearAscendingIter);
    HK_INLINE ClearAscendingIter(hkOffsetBitFieldLocal& bf)
    :   m_bf(bf)
    {}

    HK_INLINE void operator()(int bitIdx)
    {
        m_bf.clear(bitIdx + 1);
    }

    hkOffsetBitFieldLocal& m_bf;
};

static void testClearAscendingIterator()
{
    hkOffsetBitFieldLocal bf(128, hkBitFieldValue::UNINITIALIZED);
    ClearAscendingIter cai(bf);

    for (int e = 0; e < 127; e++)
    {
        for (int s = 0; s <= e; s++)
        {
            // Set the start bit to 0
            bf.assignAll<1>();
            bf.assign(s, 0);

            // Set all other bits to 0 with the iterator
            bf.forEach<hkBitFieldLoop::ASCENDING, hkBitFieldBit::CLEARED>(s, e - s + 1, cai);

            // Test. Should be zero from s to e + 1 inclusive!
            hkIntVector vActual;    OffsetBitFieldImpl::setBytes(bf.getWords(), vActual);
            hkIntVector vDesired;   vDesired.setAll(-1);
            const int numLeftZeros  = s;
            const int numZeros      = (127 - (e + 1)) + numLeftZeros;
            hkIntVector vShr;       OffsetBitFieldImpl::setAllBytes(numZeros, vShr);
            hkIntVector vShl;       OffsetBitFieldImpl::setAllBytes(numLeftZeros, vShl);
            vDesired.setBitShiftRight128(vDesired, vShr);
            vDesired.setBitShiftLeft128(vDesired, vShl);
            vDesired.setNot(vDesired);
            HK_TEST(vDesired.compareEqualS32(vActual).allAreSet());
        }
    }
}

struct SetDescendingIter
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE, SetDescendingIter);
    HK_INLINE SetDescendingIter(hkOffsetBitFieldLocal& bf)
    :   m_bf(bf)
    {}

    HK_INLINE void operator()(int bitIdx)
    {
        m_bf.set(bitIdx - 1);
    }

    hkOffsetBitFieldLocal& m_bf;
};

static void testSetDescendingIterator()
{
    hkOffsetBitFieldLocal bf(128, hkBitFieldValue::UNINITIALIZED);
    SetDescendingIter sdi(bf);

    for (int s = 1; s < 128; s++)
    {
        for (int e = 127; e >= s; e--)
        {
            // Set the end bit to 1
            bf.assignAll<0>();
            bf.assign(e, 1);

            // Set all other bits to 1 with the iterator
            bf.forEach<hkBitFieldLoop::DESCENDING, hkBitFieldBit::SET>(s, e - s + 1, sdi);

            // Test. Should be one from s-1 to e inclusive!
            hkIntVector vActual;    OffsetBitFieldImpl::setBytes(bf.getWords(), vActual);
            hkIntVector vDesired;   vDesired.setAll(-1);
            const int numLeftZeros  = (s - 1);
            const int numZeros      = (127 - e) + numLeftZeros;
            hkIntVector vShr;       OffsetBitFieldImpl::setAllBytes(numZeros, vShr);
            hkIntVector vShl;       OffsetBitFieldImpl::setAllBytes(numLeftZeros, vShl);
            vDesired.setBitShiftRight128(vDesired, vShr);
            vDesired.setBitShiftLeft128(vDesired, vShl);
            HK_TEST(vDesired.compareEqualS32(vActual).allAreSet());
        }
    }
}

struct ClearDescendingIter
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE, ClearDescendingIter);
    HK_INLINE ClearDescendingIter(hkOffsetBitFieldLocal& bf)
    :   m_bf(bf)
    {}

    HK_INLINE void operator()(int bitIdx)
    {
        m_bf.clear(bitIdx - 1);
    }

    hkOffsetBitFieldLocal& m_bf;
};

static void testClearDescendingIterator()
{
    hkOffsetBitFieldLocal bf(128, hkBitFieldValue::UNINITIALIZED);
    ClearDescendingIter cdi(bf);

    for (int s = 1; s < 128; s++)
    {
        for (int e = 127; e >= s; e--)
        {
            // Set the end bit to 0
            bf.assignAll<1>();
            bf.assign(e, 0);

            // Set all other bits to 1 with the iterator
            bf.forEach<hkBitFieldLoop::DESCENDING, hkBitFieldBit::CLEARED>(s, e - s + 1, cdi);

            // Test. Should be zero from s-1 to e inclusive!
            hkIntVector vActual;    OffsetBitFieldImpl::setBytes(bf.getWords(), vActual);
            hkIntVector vDesired;   vDesired.setAll(-1);
            const int numLeftZeros  = (s - 1);
            const int numZeros      = (127 - e) + numLeftZeros;
            hkIntVector vShr;       OffsetBitFieldImpl::setAllBytes(numZeros, vShr);
            hkIntVector vShl;       OffsetBitFieldImpl::setAllBytes(numLeftZeros, vShl);
            vDesired.setBitShiftRight128(vDesired, vShr);
            vDesired.setBitShiftLeft128(vDesired, vShl);
            vDesired.setNot(vDesired);
            HK_TEST(vDesired.compareEqualS32(vActual).allAreSet());
        }
    }
}

static void testComparison()
{
    // (Note: hkOffsetBitField aligns start bit and size to suitable word boundaries!)
    {
        // Different sizes and offsets, but same contents within common range: the longer bitfield must compare greater.
        hkOffsetBitField bf1(43, 160, hkBitFieldValue::ZERO);
        hkOffsetBitField bf2(75, 128, hkBitFieldValue::ZERO);
        bf1.set(83);
        bf2.set(83);
        verifyBitFieldRelation(bf1, bf2, 1);
        // If the contents differ within the common range, that is the deciding factor
        bf2.set(81);
        verifyBitFieldRelation(bf1, bf2, -1);
    }

    {
        // Equal sizes and offsets
        hkOffsetBitField bf1(32, 128, hkBitFieldValue::ZERO);
        hkOffsetBitField bf2(32, 128, hkBitFieldValue::ZERO);
        verifyBitFieldRelation(bf1, bf2, 0);

        bf1.set(48);
        verifyBitFieldRelation(bf1, bf2, 1);
        bf2.set(48);
        verifyBitFieldRelation(bf1, bf2, 0);

        bf2.set(67);
        verifyBitFieldRelation(bf1, bf2, -1);
        bf1.set(67);
        verifyBitFieldRelation(bf1, bf2, 0);
    }

    {
        // Comparison between regular and offset bitfield
        hkBitField bf1(128, hkBitFieldValue::ZERO);
        hkOffsetBitField bf2(32, 128, hkBitFieldValue::ZERO);
        verifyBitFieldRelation(bf1, bf2, 0);

        bf1.set(10); // set a bit not contained in the offset bit field
        verifyBitFieldRelation(bf1, bf2, 1);
        bf1.clear(10);

        bf1.set(48);
        verifyBitFieldRelation(bf1, bf2, 1);
        bf2.set(48);
        verifyBitFieldRelation(bf1, bf2, 0);

        bf2.set(67);
        verifyBitFieldRelation(bf1, bf2, -1);
        bf1.set(67);
        verifyBitFieldRelation(bf1, bf2, 0);
    }
}

static int offsetBitField_main()
{
    testSetAscendingIterator();
    testSetDescendingIterator();
    testClearAscendingIterator();
    testClearDescendingIterator();
    testComparison();

    return 0;
}

HK_TEST_REGISTER(offsetBitField_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__     );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
