// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Container/PointerMultiMap/hkPointerMultiMap.h>

namespace
{
/// COM-4484 Workaround compiler bug in VS2017 Update 5
bool randAnd16(hkPseudoRandomGenerator& rand)
{
    hkUint32 rand32 = rand.getRand32();
    return rand32 & 16;
}
}

static void pointerMultiMap_selfTest()
{
    // General checking
    {
        hkPointerMultiMap<int, char> map;

        map.insert(10, 'a');
        map.insert(10, 'b');
        map.insert(11, 'b');
        map.insert(10, 'b');

        HK_TEST(map.findNumEntries(10) == 3);
        HK_TEST(map.findNumEntries(10, 'a') == 1);

        int numRemoved = map.removeAll(10);
        HK_TEST(numRemoved == 3);

        map.insert(10, 'a');
        map.insert(10, 'b');
        map.insert(10, 'c');

        int bits = 0;
        hkPointerMultiMap<int, char>::Iterator iter = map.findKey(10);
        for (; map.isValid(iter); iter = map.getNext(iter, 10))
        {
            char value = map.getValue(iter);
            HK_TEST(value >= 'a' && value <= 'c');
            int bit = 1 << ( value - 'a' );
            HK_TEST((bits & bit) == 0);

            bits |= bit;
        }

        HK_TEST( bits == 0x7);
    }


    // Checking if removals correctly handles case of intermixed as, and bs
    {
        hkPseudoRandomGenerator rand(0x1321);

        hkMultiMap<hkUlong, hkUlong, hkMultiMapIntegralOperations> map;

        for (int i = 0; i < 100; i++)
        {
            // We don't want to clash with invalid
            int a = int(rand.getRand32()) & 0x7fffffff;
            int b = a + 1;
            /*
            do
            {
                b = int(rand.getRand32()) & 0x7fffffff;
            } while (b == a); */

            int numAs = 0;
            int numBs = 0;

            for (int j = 0; j < 100; j++)
            {
                if (rand.getRand32() & 8)
                {
                    map.insert(a, 1);
                    numAs ++;
                }
                else
                {
                    map.insert(b, 1);
                    numBs ++;
                }
            }

            while (map.getSize() > 0)
            {
                HK_TEST(map.findNumEntries(a) == numAs);
                HK_TEST(map.findNumEntries(b) == numBs);

                if (randAnd16(rand))
                {
                    if (numAs > 0)
                    {
                        map.remove(a);
                        numAs--;
                    }
                }
                else
                {
                    if (numBs > 0)
                    {
                        map.remove(b);
                        numBs--;
                    }
                }
            }
        }
    }
}

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    template< typename V >
    V getTestValue()
    {
        return V( 0 );
    }

    template<  >
    hkUint64 getTestValue<hkUint64>()
    {
        return 0x1111111111111111ULL;
    }

    template<  >
    int getTestValue<int>()
    {
        return 123;
    }

    template< typename K, typename V>
    static void testMap()
    {
        hkPointerMultiMap<K, V> map;
        K k1 = getTestValue<K>();
        V v1 = getTestValue<V>();
        HK_ASSERT_NO_MSG( 0x2c2af402, v1 != 0 );

        map.insert( k1, v1 );

        typename hkPointerMultiMap<K, V>::Iterator iter1 = map.getIterator();
        HK_TEST( map.isValid( iter1 ) );

        V v2 = map.getValue( iter1 );
        HK_TEST( v2 == v1 );
    }

    static void testTruncation()
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;
        // test to make sure that there's no truncation when using keys/values of different sizes
        
        testMap<hkUint64, int>();
        testMap<int, hkUint64>();
        testMap<hkInt64, hkUint64>();
        testMap<int, int>();
    }


    template< typename K, typename V>
    static void testMapRealValues()
    {
        hkPointerMultiMap<K, V> map;
        V val = V( 2.5 );
        K k1 = K( 100 );
        map.insert( k1, val );

        HK_TEST( map.getWithDefault( k1, V( 12.34 ) ) == val );
    }

    template< typename K, typename V>
    static void testMapRealKeys()
    {
        hkPointerMultiMap<K, V> map;
        K keys[3] = { K( 2.1 ), K( 2.2 ), K( 2.3 ) };

        for (int i = 0; i < 3; i++)
        {
            map.insert( keys[i], V( i ) );
        }

        for (int i = 0; i < 3; i++)
        {
            HK_TEST( map.getWithDefault( keys[i], V( 1234 ) ) == V( i ) );
        }
    }


    static void testReal()
    {
        testMapRealValues<int, float>();
        testMapRealValues<hkInt64, float>();
        testMapRealValues<int, double>();
        testMapRealValues<hkInt64, double>();

        testMapRealKeys< float, int >();
        testMapRealKeys< float, hkInt64 >();
        testMapRealKeys< double, int >();
        testMapRealKeys< double, hkInt64 >();
    }
}

int pointerMultiMap_main()
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    testTruncation();
    testReal();
    pointerMultiMap_selfTest();
    return 0;
}

HK_TEST_REGISTER(pointerMultiMap_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__     );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
