// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/UnitTest/UnitTestUtils.h>

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    struct A
    {
        HK_DECLARE_CLASS( A, New );

        char m_vc[32];
    };

    struct B
    {
        HK_DECLARE_CLASS( B, New );

        hkInplaceArray<A, 1> m_a;
    };

    static void inplace_array()
    {
        {
            typedef hkInplaceArrayAligned16<hkReal, 1> Array;
            Array a;
            a.pushBack( 4 );
            HK_TEST( a.wasReallocated() == false );
            a.pushBack( 10 );
            HK_TEST( a.wasReallocated() );
        }

        {


            typedef hkInplaceArray<A, 1> Array;
            Array a;
            A item;
            a.pushBack( item );
            HK_TEST( a.wasReallocated() == false );
            a.pushBack( item );
            HK_TEST( a.wasReallocated() );
        }

        {
            B container;
            A item;
            container.m_a.pushBack( item );
            HK_TEST( container.m_a.wasReallocated() == false );
            container.m_a.pushBack( item );
            HK_TEST( container.m_a.wasReallocated() );
        }

        {
            B* container = new B;
            A item;
            container->m_a.pushBack( item );
            HK_TEST( container->m_a.wasReallocated() == false );
            container->m_a.pushBack( item );
            HK_TEST( container->m_a.wasReallocated() );
            delete container;
        }
    }

    class NonPodArrayElement
    {
        public:
        NonPodArrayElement() { m_someObjectData = s_numLivingObjects % 3; ++s_numLivingObjects; }
        ~NonPodArrayElement() { --s_numLivingObjects; }
        bool operator==( const NonPodArrayElement& other ) { return other.m_someObjectData == m_someObjectData; }
        bool operator!=( const NonPodArrayElement& other ) { return other.m_someObjectData != m_someObjectData; }

        static int s_numLivingObjects;
        int m_someObjectData;
    };

    template<int N>
    struct hkPascalString
    {
        hkUint8 len;
        char str[N + 1];
    };
    //#define HK_PASCAL_STRING(STR) hkPascalString<HK_COUNT_OF(STR)-1>
    static hkPascalString<3> pascalstring = { 3, "foo" };

    static void array_view_nonconst(hkArrayView<int> a)
    {
        a[2] = -10;
    }

    static int array_view_const(hkArrayView<const int> a)
    {
        // array_view_nonconst(a); // mustn't compile because you cannot implicitly convert from 'const T' to 'T'.
        return a[2] + 10;
    }

    static void array_hk_nonconst(const hkArray<int>& a)
    {
        // array_view_nonconst(a); // shouldn't compile because 'a' is read-only here.
        array_view_const(a);
    }

    static void array_view()
    {
        int xxx[10] = { 10,20,30,40,50,60,70,80,90,100 };
        {
            // check fixed array ctor and slicing
            hkArrayView<int> av = hkArrayViewT::make( xxx );
            HK_TEST( av.getSize() == 10 );
            HK_TEST( av[1] == 20 );
            HK_TEST( av[9] == 100 );

            hkArrayView<int> avl3 = av.ltrim( 3 );
            HK_TEST( avl3[0] == 40 );
            HK_TEST( avl3.getSize() == 7 );

            hkArrayView<int> avr4 = av.rtrim( 4 );
            HK_TEST( avr4[0] == 10 );
            HK_TEST( avr4.getSize() == 6 );
        }
        {
            hkArrayView<int> av( &xxx[0], HK_COUNT_OF( xxx ) );
            array_view_nonconst( av );
        }
        {
            const hkArray<int> con(&xxx[0], HK_COUNT_OF(xxx), HK_COUNT_OF(xxx));
            array_hk_nonconst(con);
            // array_view_nonconst(con); // mustn't compile
            array_view_const(con);
        }
        {
            hkArrayView<int> ints = hkArrayViewT::make(xxx);
            hkArrayView<void> writableData = ints;
            hkArrayView<const void> readonlyData = ints;
            HK_TEST( hkGetByteOffset( writableData.begin(), ints.begin() ) == 0 );
            HK_TEST( hkGetByteOffset( writableData.end(),   ints.end() )   == 0 );
            HK_TEST( hkGetByteOffset( readonlyData.begin(), ints.begin() ) == 0 );
            HK_TEST( hkGetByteOffset( readonlyData.end(),   ints.end() )   == 0 );
        }
        {
            hkArrayView<void> data(&xxx[0], 10);
            HK_TEST( data.getSize() == 10 * sizeof(int) );
            HK_TEST( data.begin() == &xxx[0] );
            HK_TEST( data.end() == &xxx[9] + 1 );
        }
        {
            // None of the following should compile.
            // hkArrayView<float> floats = hkArrayView<int>();
            // hkArrayView<unsigned> uints = hkArrayView<signed>();
            // hkArrayView<int*> intPtrs = hkArrayView<const int*>();
        }
    }

    void array_removeAt()
    {
        hkArray<int> arr;
        arr.reserve( 5 );

        arr.pushBack( 1 );
        arr.pushBack( 2 );
        arr.pushBack( 3 );
        if ( HK_TEST( arr.getSize() == 3 ) )
        {
            HK_TEST( arr[0] == 1 );
            HK_TEST( arr[1] == 2 );
            HK_TEST( arr[2] == 3 );

            arr.removeAt( 0 );
            if ( HK_TEST( arr.getSize() == 2 ) )
            {
                // removeAt() swap elements, does not preserve order
                HK_TEST( arr[0] == 3 );
                HK_TEST( arr[1] == 2 );

                arr.removeAt( 1 );
                if ( HK_TEST( arr.getSize() == 1 ) )
                {
                    HK_TEST( arr[0] == 3 );

                    arr.removeAt( 0 );
                    HK_TEST( arr.isEmpty() );
                }
            }
        }
    }

    void array_erase()
    {
        int data[] = { 1, 2, 3, 4, 5, 6, 7 };
        hkArray<int> arr( data, 7, 7 );
        arr.reserve( 8 ); // force resize -> alloc+copy
        HK_TEST( arr.getSize() == 7 );

        auto it = arr.erase( arr.begin() + 2 );
        HK_TEST( arr.getSize() == 6 );
        HK_TEST( it == arr.begin() + 2 );
        // erase(it) swap elements, does not preserve order
        HK_TEST( arr[0] == 1 );
        HK_TEST( arr[1] == 2 );
        HK_TEST( arr[2] == 7 );
        HK_TEST( arr[3] == 4 );
        HK_TEST( arr[4] == 5 );
        HK_TEST( arr[5] == 6 );

        arr.erase( arr.begin() );
        arr.erase( arr.begin() );
        arr.erase( arr.begin() );
        arr.erase( arr.begin() );
        arr.erase( arr.begin() );
        it = arr.erase( arr.begin() );
        HK_TEST( arr.isEmpty() );
        HK_TEST( it == arr.end() );
    }

    void array_erase_slice()
    {
        int data[] = { 1, 2, 3, 4, 5, 6, 7 };
        hkArray<int> arr( data, 7, 7 );
        arr.reserve( 8 ); // force resize -> alloc+copy
        HK_TEST( arr.getSize() == 7 );

        auto it = arr.eraseAndCopy( arr.begin() + 2, arr.begin() + 4 );
        HK_TEST( arr.getSize() == 5 );
        HK_TEST( it == arr.begin() + 2 );
        HK_TEST( arr[0] == 1 );
        HK_TEST( arr[1] == 2 );
        HK_TEST( arr[2] == 5 );
        HK_TEST( arr[3] == 6 );
        HK_TEST( arr[4] == 7 );

        it = arr.eraseAndCopy( arr.begin() + 3, arr.end() );
        HK_TEST( arr.getSize() == 3 );
        HK_TEST( it == arr.end() );
        HK_TEST( arr[0] == 1 );
        HK_TEST( arr[1] == 2 );
        HK_TEST( arr[2] == 5 );

        it = arr.eraseAndCopy( arr.begin(), arr.end() );
        HK_TEST( arr.isEmpty() );
        HK_TEST( it == arr.end() );
    }

    static void array_emplace()
    {
        {  // hkArray::emplace() should support perfect forwarding
            using namespace UnitTest;
            hkArray<ForwardingTestingHelper> arr;

            static const int testArraySize = 20;
            static const int indices[20] = { 0, 1, 1, 2, 1, 0, 6, 0 , 5, 4, 2, 11, 2, 3, 4, 5, 6, 7, 8, 19 };

            for (int i = 0; i < testArraySize; ++i)
            {
                ForwardingTestingHelper src;
                arr.emplace(indices[i],
                    hk::move(src.t1),
                    hk::move(src.t2),
                    hk::move(src.t3),
                    hk::move(src.t4),
                    hk::move(src.t5));
                HK_TEST(arr.getSize() == (i + 1));
                HK_TEST(src.isEverythingMovedFrom());
                HK_TEST(arr.back().isEverythingMoveConstructed());
            }

            for (const auto& element : arr)
            {  // Resizing did not affect any existing elements
                HK_TEST(element.isEverythingMoveConstructed());
            }
        }
        {  // hkArray::emplace() should work with non-copyable and non-default-constructible types
            struct NonCopyableNonDefaultConstructible
            {
                NonCopyableNonDefaultConstructible(int data) : m_data(data) {}
                NonCopyableNonDefaultConstructible(NonCopyableNonDefaultConstructible&& other) {}

                int m_data;
            private:
                NonCopyableNonDefaultConstructible() {}
                NonCopyableNonDefaultConstructible(const NonCopyableNonDefaultConstructible&) {}
                void operator=(const NonCopyableNonDefaultConstructible&) {}
            };

            hkArray<NonCopyableNonDefaultConstructible> arr;
            static const int testArraySize = 10;
            static const int indices[10] = { 0, 1, 1, 2, 1, 0, 6, 0 , 5, 4 };
            static const int expectedArray[10] = { 7, 5, 0, 4, 9, 2, 8, 3, 1, 6 };

            for (int i = 0; i < testArraySize; ++i)
            {
                arr.emplace(indices[i], i);
                HK_TEST(arr.getSize() == (i + 1));
            }

            // Test if elements are ordered properly
            for (int i = 0; i < testArraySize; ++i)
            {
                HK_TEST(arr[i].m_data == expectedArray[i]);
            }
        }
    }

    static void array_emplaceBack()
    {
        {  // hkArray::emplaceBack() should support perfect forwarding
            using namespace UnitTest;
            hkArray<ForwardingTestingHelper> arr;

            static const int testArraySize = 20;

            for (int i = 0; i < testArraySize; ++i)
            {
                ForwardingTestingHelper src;
                arr.emplaceBack(
                    hk::move(src.t1),
                    hk::move(src.t2),
                    hk::move(src.t3),
                    hk::move(src.t4),
                    hk::move(src.t5));
                HK_TEST(arr.getSize() == (i + 1));
                HK_TEST(src.isEverythingMovedFrom());
                HK_TEST(arr.back().isEverythingMoveConstructed());
            }

            for (const auto& element : arr)
            {  // Resizing did not affect any existing elements
                HK_TEST(element.isEverythingMoveConstructed());
            }
        }
        {  // hkArray::emplaceBack() should work with non-copyable and non-default-constructible types
            struct NonCopyableNonDefaultConstructible
            {
                NonCopyableNonDefaultConstructible(int) {}
                NonCopyableNonDefaultConstructible(NonCopyableNonDefaultConstructible&& other) {}
            private:
                NonCopyableNonDefaultConstructible() {}
                NonCopyableNonDefaultConstructible(const NonCopyableNonDefaultConstructible&) {}
                void operator=(const NonCopyableNonDefaultConstructible&) {}
            };

            hkArray<NonCopyableNonDefaultConstructible> arr;
            static const int testArraySize = 20;

            for (int i = 0; i < testArraySize; ++i)
            {
                arr.emplaceBack(42);
                HK_TEST(arr.getSize() == (i + 1));
            }
        }
    }

    int NonPodArrayElement::s_numLivingObjects = 0;

    struct NonPodTracker;
    struct PodTracker;
    struct NonPodTracker2;
    struct PodTracker2;

    int g_assigns = 0;
    int g_dtors = 0;
    int g_default_ctors = 0;
    int g_copy_ctors = 0;
    int g_other_ctors = 0;

#define HK_DEFINE_TEST_CLASS(CLASSNAME, ...) \
    struct CLASSNAME \
    { \
        HK_DECLARE_CLASS(CLASSNAME, __VA_ARGS__); \
 \
        CLASSNAME(int val = 0) \
        { \
            ++g_default_ctors; \
            m_val = val; \
        } \
 \
        CLASSNAME(NonPodTracker const& other); \
        CLASSNAME(NonPodTracker2 const& other); \
        CLASSNAME(PodTracker const& other); \
        CLASSNAME(PodTracker2 const& other); \
\
        ~CLASSNAME() { \
            ++g_dtors; \
        } \
 \
        CLASSNAME& operator=(NonPodTracker const& other); \
        CLASSNAME& operator=(NonPodTracker2 const& other); \
        CLASSNAME& operator=(PodTracker const& other); \
        CLASSNAME& operator=(PodTracker2 const& other); \
 \
        int m_val; \
    };

    HK_DEFINE_TEST_CLASS(NonPodTracker, New)
        HK_DEFINE_TEST_CLASS(PodTracker, New, Pod)
        HK_DEFINE_TEST_CLASS(NonPodTracker2, New)
        HK_DEFINE_TEST_CLASS(PodTracker2, New, Pod)

#define HK_DEFINE_ASSIGNMENT_FROM(CLASSTYPE, OTHERTYPE) \
    CLASSTYPE& CLASSTYPE::operator=(OTHERTYPE const& other) \
    { \
        ++g_assigns; \
        m_val = other.m_val; \
        return *this; \
    }

        HK_DEFINE_ASSIGNMENT_FROM(NonPodTracker, NonPodTracker)
        HK_DEFINE_ASSIGNMENT_FROM(NonPodTracker, NonPodTracker2)
        HK_DEFINE_ASSIGNMENT_FROM(NonPodTracker, PodTracker)
        HK_DEFINE_ASSIGNMENT_FROM(NonPodTracker, PodTracker2)

        HK_DEFINE_ASSIGNMENT_FROM(PodTracker, NonPodTracker)
        HK_DEFINE_ASSIGNMENT_FROM(PodTracker, NonPodTracker2)
        HK_DEFINE_ASSIGNMENT_FROM(PodTracker, PodTracker)
        HK_DEFINE_ASSIGNMENT_FROM(PodTracker, PodTracker2)

        HK_DEFINE_ASSIGNMENT_FROM(NonPodTracker2, NonPodTracker)
        HK_DEFINE_ASSIGNMENT_FROM(NonPodTracker2, NonPodTracker2)
        HK_DEFINE_ASSIGNMENT_FROM(NonPodTracker2, PodTracker)
        HK_DEFINE_ASSIGNMENT_FROM(NonPodTracker2, PodTracker2)

        HK_DEFINE_ASSIGNMENT_FROM(PodTracker2, NonPodTracker)
        HK_DEFINE_ASSIGNMENT_FROM(PodTracker2, NonPodTracker2)
        HK_DEFINE_ASSIGNMENT_FROM(PodTracker2, PodTracker)
        HK_DEFINE_ASSIGNMENT_FROM(PodTracker2, PodTracker2)

#define HK_DEFINE_CONSTRUCTOR_FROM(CLASSTYPE, OTHERTYPE) \
    CLASSTYPE::CLASSTYPE(OTHERTYPE const& other) { \
        if (hkTrait::TypesAreEqual<CLASSTYPE, OTHERTYPE>::result) \
            ++g_copy_ctors; \
        else \
            ++g_other_ctors; \
        m_val = other.m_val; \
    }

    HK_DEFINE_CONSTRUCTOR_FROM(NonPodTracker, NonPodTracker)
    HK_DEFINE_CONSTRUCTOR_FROM(NonPodTracker, NonPodTracker2)
    HK_DEFINE_CONSTRUCTOR_FROM(NonPodTracker, PodTracker)
    HK_DEFINE_CONSTRUCTOR_FROM(NonPodTracker, PodTracker2)

    HK_DEFINE_CONSTRUCTOR_FROM(PodTracker, NonPodTracker)
    HK_DEFINE_CONSTRUCTOR_FROM(PodTracker, NonPodTracker2)
    HK_DEFINE_CONSTRUCTOR_FROM(PodTracker, PodTracker)
    HK_DEFINE_CONSTRUCTOR_FROM(PodTracker, PodTracker2)

    HK_DEFINE_CONSTRUCTOR_FROM(NonPodTracker2, NonPodTracker)
    HK_DEFINE_CONSTRUCTOR_FROM(NonPodTracker2, NonPodTracker2)
    HK_DEFINE_CONSTRUCTOR_FROM(NonPodTracker2, PodTracker)
    HK_DEFINE_CONSTRUCTOR_FROM(NonPodTracker2, PodTracker2)

    HK_DEFINE_CONSTRUCTOR_FROM(PodTracker2, NonPodTracker)
    HK_DEFINE_CONSTRUCTOR_FROM(PodTracker2, NonPodTracker2)
    HK_DEFINE_CONSTRUCTOR_FROM(PodTracker2, PodTracker)
    HK_DEFINE_CONSTRUCTOR_FROM(PodTracker2, PodTracker2)
}

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    void array_assign_pod_and_non_pod()
    {
        // Assign array to array of same POD type
        {
            hkArray<PodTracker> arr1;
            arr1.pushBack(PodTracker());
            arr1.pushBack(PodTracker());
            hkArray<PodTracker> arr2;
            arr2.pushBack(PodTracker());
            arr2.pushBack(PodTracker());
            arr2.pushBack(PodTracker());
            arr2.pushBack(PodTracker());

            g_default_ctors = 0;
            g_copy_ctors = 0;
            g_other_ctors = 0;
            g_dtors = 0;
            g_assigns = 0;
            arr1 = arr2;
            HK_TEST(g_dtors == 0);
            HK_TEST(g_default_ctors == 0);
            HK_TEST(g_copy_ctors == 0);
            HK_TEST(g_other_ctors == 0);
            HK_TEST(g_assigns == 4);
        }
        // Assign array to array of different type, but still POD
        {
            hkArray<PodTracker> arr1;
            arr1.pushBack(PodTracker());
            arr1.pushBack(PodTracker());
            hkArray<PodTracker2> arr2;
            arr2.pushBack(PodTracker2());
            arr2.pushBack(PodTracker2());
            arr2.pushBack(PodTracker2());
            arr2.pushBack(PodTracker2());

            g_default_ctors = 0;
            g_copy_ctors = 0;
            g_other_ctors = 0;
            g_dtors = 0;
            g_assigns = 0;
            arr1 = arr2;
            HK_TEST(g_dtors == 0);
            HK_TEST(g_default_ctors == 0);
            HK_TEST(g_copy_ctors == 0);
            HK_TEST(g_other_ctors == 0);
            HK_TEST(g_assigns == 4);
        }
        // Assign array to array of same type, non-POD
        {
            hkArray<NonPodTracker> arr1;
            arr1.pushBack(NonPodTracker());
            arr1.pushBack(NonPodTracker());
            hkArray<NonPodTracker> arr2;
            arr2.pushBack(NonPodTracker());
            arr2.pushBack(NonPodTracker());
            arr2.pushBack(NonPodTracker());
            arr2.pushBack(NonPodTracker());

            g_default_ctors = 0;
            g_copy_ctors = 0;
            g_other_ctors = 0;
            g_dtors = 0;
            g_assigns = 0;
            arr1 = arr2;
            HK_TEST(g_dtors == 0);
            HK_TEST(g_default_ctors == 0);
            HK_TEST(g_copy_ctors == 2);
            HK_TEST(g_other_ctors == 0);
            HK_TEST(g_assigns == 2);
        }
        // Assign array to array of different type, non-POD
        {
            hkArray<NonPodTracker> arr1;
            arr1.pushBack(NonPodTracker());
            arr1.pushBack(NonPodTracker());
            hkArray<NonPodTracker2> arr2;
            arr2.pushBack(NonPodTracker2());
            arr2.pushBack(NonPodTracker2());
            arr2.pushBack(NonPodTracker2());
            arr2.pushBack(NonPodTracker2());

            g_default_ctors = 0;
            g_copy_ctors = 0;
            g_other_ctors = 0;
            g_dtors = 0;
            g_assigns = 0;
            arr1 = arr2;
            HK_TEST(g_dtors == 0);
            HK_TEST(g_default_ctors == 0);
            HK_TEST(g_copy_ctors == 0);
            HK_TEST(g_other_ctors == 2);
            HK_TEST(g_assigns == 2);
        }
        // Assign array of non-POD to array POD. Assignments to
        // destination array from non-POD using operator=()
        {
            hkArray<PodTracker> arr1;
            arr1.pushBack(PodTracker());
            arr1.pushBack(PodTracker());
            hkArray<NonPodTracker> arr2;
            arr2.pushBack(NonPodTracker());
            arr2.pushBack(NonPodTracker());
            arr2.pushBack(NonPodTracker());
            arr2.pushBack(NonPodTracker());

            g_default_ctors = 0;
            g_copy_ctors = 0;
            g_other_ctors = 0;
            g_dtors = 0;
            g_assigns = 0;
            arr1 = arr2;
            HK_TEST(g_dtors == 0);
            HK_TEST(g_default_ctors == 0);
            HK_TEST(g_copy_ctors == 0);
            HK_TEST(g_other_ctors == 0);
            HK_TEST(g_assigns == 4);
        }
        // Assign array of POD to array of non-POD. Assignments to
        // existing elements, and construction of additional elements
        {
            hkArray<NonPodTracker> arr1;
            arr1.pushBack(NonPodTracker());
            arr1.pushBack(NonPodTracker());
            hkArray<PodTracker> arr2;
            arr2.pushBack(PodTracker());
            arr2.pushBack(PodTracker());
            arr2.pushBack(PodTracker());
            arr2.pushBack(PodTracker());

            g_default_ctors = 0;
            g_copy_ctors = 0;
            g_other_ctors = 0;
            g_dtors = 0;
            g_assigns = 0;
            arr1 = arr2;
            HK_TEST(g_dtors == 0);
            HK_TEST(g_default_ctors == 0);
            HK_TEST(g_copy_ctors == 0);
            HK_TEST(g_other_ctors == 2);
            HK_TEST(g_assigns == 2);
        }
    }
}  // namespace HK_UNITY_ANONYMOUS_NAMESPACE

int array_main()
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    NonPodArrayElement::s_numLivingObjects = 0; // Otherwise test will fail on second pass...

    array_view();

    inplace_array();

    array_removeAt();
    array_erase();
    array_erase_slice();
    array_emplace();
    array_emplaceBack();
    array_assign_pod_and_non_pod();

    {
        int a12345[] = {1,2,3,4,5};
        hkArray<int> a( a12345, HK_COUNT_OF(a12345), HK_COUNT_OF(a12345) ); a.reserve(10); // ensure copied
        hkArray<int> b( a12345, HK_COUNT_OF(a12345), HK_COUNT_OF(a12345) ); b.reserve(10);
        hkArray<int> c( a12345, HK_COUNT_OF(a12345), HK_COUNT_OF(a12345) ); c.reserve(10);

        const int a678[] = {6,7,8};
        a.spliceInto( 1, 1, a678, HK_COUNT_OF(a678) );
        b.spliceInto( 1, 4, a678, HK_COUNT_OF(a678) );
        c.spliceInto( 1, 3, a678, HK_COUNT_OF(a678) );

        const int aout[] = { 1,6,7,8,3,4,5 };
        const int bout[] = { 1,6,7,8 };
        const int cout[] = { 1,6,7,8,5 };

        HK_TEST( HK_COUNT_OF(aout) == a.getSize() );
        HK_TEST( hkString::memCmp( a.begin(), aout, a.getSize() * sizeof(int) ) == 0 );
        HK_TEST( HK_COUNT_OF(bout) == b.getSize() );
        HK_TEST( hkString::memCmp( b.begin(), bout, b.getSize() * sizeof(int) ) == 0 );
        HK_TEST( HK_COUNT_OF(cout) == c.getSize() );
        HK_TEST( hkString::memCmp( c.begin(), cout, c.getSize() * sizeof(int) ) == 0 );
    }
    {
        hkArray<int> a;
        const int a12[] = {1,2};
        a.append(a12, HK_COUNT_OF(a12) );
        HK_TEST( a.getSize() == HK_COUNT_OF(a12) );
        const int a3456789[] = {3,4,5,6,7,8,9};
        a.append(a3456789, HK_COUNT_OF(a3456789) );
        HK_TEST( a.getSize() == HK_COUNT_OF(a12)+HK_COUNT_OF(a3456789) );
        for( int i = 0; i < a.getSize(); ++i )
        {
            HK_TEST(a[i]==i+1);
        }
    }
    {
        char ctest[] = {'H','e','l','l','o'};
        hkArray<char> c( ctest, HK_COUNT_OF(ctest), HK_COUNT_OF(ctest) ); c.reserve(10);
        c.spliceInto( 2, 0, ctest, HK_COUNT_OF(ctest) );
        HK_TEST( c.getSize() == 10 );
        HK_TEST( hkString::memCmp( &c[2], ctest, HK_COUNT_OF(ctest) ) == 0 );
        HK_TEST( hkString::memCmp( &c[7], &ctest[2], 3 ) == 0 );
        c.spliceInto( 0, 2, ctest, HK_COUNT_OF(ctest) );
        HK_TEST( c.getSize() == 13 );
        HK_TEST( hkString::memCmp( c.begin(), ctest, HK_COUNT_OF(ctest) ) == 0 );
        HK_TEST( hkString::memCmp( &c[5], ctest, HK_COUNT_OF(ctest) ) == 0 );
        HK_TEST( hkString::memCmp( &c[10], &ctest[2], 3 ) == 0 );
    }
    {
        // Test removeAll on nonPod objects.
        hkArray<NonPodArrayElement> a( 9 );
        NonPodArrayElement zero;
        NonPodArrayElement one;
        NonPodArrayElement two;
        a.removeAllAndCopy( zero );
        HK_TEST( NonPodArrayElement::s_numLivingObjects == 9 );
        HK_TEST( a.getSize() == 6 );
        a.removeAllAndCopy( one );
        HK_TEST( NonPodArrayElement::s_numLivingObjects == 6 );
        HK_TEST( a.getSize() == 3 );
        a.removeAllAndCopy( two );
        HK_TEST( NonPodArrayElement::s_numLivingObjects == 3 );
        HK_TEST( a.getSize() == 0 );
    }
    {
        int tempData[5] = {1,2,3,4,5};
        hkArrayBase<int> tempArray(tempData, 5, 5);
        HK_TEST( tempArray.getSize() == 5 );
        // hkArrayBase destructor shouldn't assert, even though the size is non-zero, because we're using a POD type.
    }
    {
        NonPodArrayElement tempData[5];
        hkArrayBase<NonPodArrayElement> tempArray(tempData, 5, 5);
        HK_TEST( tempArray.getSize() == 5 );
        HK_TEST( NonPodArrayElement::s_numLivingObjects == 5 );

        tempArray.clear();      // hkArrayBase destructor will assert if we don't clear
        HK_TEST( NonPodArrayElement::s_numLivingObjects == 0 );
    }

    {
        hkArray<int> intArray;
        HK_TEST(intArray.getCapacity() == 0);
        HK_TEST(intArray.getSize() == 0);
        HK_TEST(!intArray.begin());
        HK_TEST(intArray.reflectValidate());

        intArray.pushBack(42);
        HK_TEST(intArray.getCapacity() != 0);
        HK_TEST(intArray.getSize() == 1);
        HK_TEST(intArray.begin());
        HK_TEST(intArray.reflectValidate());

        intArray.removeAt(0);
        HK_TEST(intArray.getCapacity() != 0);
        HK_TEST(intArray.getSize() == 0);
        HK_TEST(intArray.begin());
        HK_TEST(intArray.reflectValidate());
    }

    return 0;
}

HK_TEST_REGISTER(array_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__     );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
