// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>

#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Types/Uuid/hkUuid.h>
#include <Common/Base/Types/hkEndian.h>

static void baseTypes_halfTest()
{
    hkPseudoRandomGenerator random(1);
    {
        for (int i =0; i < 1000; i++)
        {
            hkReal x = random.getRandRange( -FLT_MAX*0.5f, FLT_MAX*0.5f );
            hkReal maxError = hkMath::fabs(x * 0.01f);
            hkHalf16 half; half.setReal<true>(x);
            hkReal uncompressed = half.getReal();
            HK_TEST( hkMath::equal(uncompressed, x, maxError));
        }
    }
    {
        for (int i =0; i < 1000; i++)
        {
            hkReal x = random.getRandRange( -100.0f, 100.0f );
            hkReal maxError = hkMath::fabs(x * 0.01f);
            hkHalf16 half; half.setReal<false>(x);
            hkReal uncompressed = half.getReal();
            HK_TEST( hkMath::equal(uncompressed, x, maxError));
        }
    }
    {
        for (int i =0; i < 1000; i++)
        {
            hkReal x = random.getRandRange( -0.001f, 0.001f );
            hkReal maxError = hkMath::fabs(x * 0.01f);
            hkHalf16 half; half.setReal<true>(x);
            hkReal uncompressed = half.getReal();
            HK_TEST( hkMath::equal(uncompressed, x, maxError));
        }
    }
}

static hkRefLoan<hkReferencedObject> borrowedRef(hkReferencedObject& r)
{
    return &r;
}

// simulate returning a new ref on an existing object
static hkRefNew<hkReferencedObject> newRef(hkReferencedObject& r)
{
    r.addReference();
    return &r;
}

static void baseTypes_refptrTest()
{
    hkReferencedObject r0;
    r0.setMemorySizeAndFlags(sizeof(hkReferencedObject)); // Explicitly initialize to non zero value - zero has special meaning for packfiles.
    {
        HK_TEST( r0.getReferenceCount() == 1 );
        borrowedRef(r0); // ok, ignore return
        hkReferencedObject* r = borrowedRef(r0); // ok, use return
        HK_TEST(r->getReferenceCount() == 1);
        hkRefPtr<hkReferencedObject> rr = borrowedRef(r0); // take another ref
        hkRefPtr<hkReferencedObject> r2 = HK_NULL;
        HK_TEST( r0.getReferenceCount() == 2 );
        r2 = borrowedRef(r0);
        HK_TEST( r0.getReferenceCount() == 3 );
    }
    HK_TEST( r0.getReferenceCount() == 1 );
    {
        HK_TEST( r0.getReferenceCount() == 1 );
        hkReferencedObject* r = newRef(r0).stealOwnership();
        HK_TEST( r0.getReferenceCount() == 2 );
        r->removeReference();
        HK_TEST( r0.getReferenceCount() == 1 );

        //hkReferencedObject* s = newRef(r0); // shouldn't compile
        hkRefPtr<hkReferencedObject> rr = newRef(r0); // take another ref
        hkRefPtr<hkReferencedObject> r2 = HK_NULL;
        HK_TEST( r0.getReferenceCount() == 2 );
        r2 = newRef(r0);
        HK_TEST( r0.getReferenceCount() == 3 );
    }
    HK_TEST( r0.getReferenceCount() == 1 );

    // Test setAndDontIncrementRefCount.
    {
        hkRefPtr<hkReferencedObject> refPtr;
        refPtr.setAndDontIncrementRefCount(&r0);
        HK_TEST(r0.getReferenceCount() == 1);
        refPtr.setAndDontIncrementRefCount(&r0);
        HK_TEST(r0.getReferenceCount() == 1);
        refPtr.setAndDontIncrementRefCount(&r0);
        HK_TEST(r0.getReferenceCount() == 1);
        refPtr.unsetAndDontDecrementRefCount();
        HK_TEST(r0.getReferenceCount() == 1);

    }
    HK_TEST(r0.getReferenceCount() == 1);
}

//
// HK_REMOVE_REFERENCE tests
//

// A referenced object that counts its number of instances
class hkCountedReferencedObject : public hkReferencedObject
{
    public:

        static int m_numInstances;

        hkCountedReferencedObject() : hkReferencedObject() { m_numInstances++;}

        ~hkCountedReferencedObject() { m_numInstances--; }

        static void initTest() { m_numInstances = 0; }

        static void endTest() { HK_TEST(m_numInstances == 0); }
};

int hkCountedReferencedObject::m_numInstances = 0;

// A typical class holding a reference injected on construction or through a setter
struct ClassHoldingReference
{
    ClassHoldingReference(hkReferencedObject* obj) : m_refObject (obj)
    {
        HK_TEST( obj->getReferenceCount()==2 );
    };

    void doSomethingWithObject (hkReferencedObject* obj)
    {
        HK_TEST( obj->getReferenceCount()==1 );
    };

    void setObject (hkReferencedObject* obj)
    {
        HK_TEST( obj->getReferenceCount()==1 );
        m_refObject = obj;
        HK_TEST( obj->getReferenceCount()==2 );
    }

    hkRefPtr<hkReferencedObject> m_refObject;
};

static void baseTypes_removeReferenceTest()
{
    hkCountedReferencedObject::initTest();
    {
        hkRefPtr<hkReferencedObject> refPtr( hkAutoRemoveReference( new hkCountedReferencedObject() ));
        HK_TEST( refPtr->getReferenceCount()==1 );

        hkRefPtr<hkReferencedObject> refPtr2;
        refPtr2 = hkAutoRemoveReference( new hkCountedReferencedObject() );
        HK_TEST( refPtr2->getReferenceCount()==1 );

        {
            ClassHoldingReference holder( hkAutoRemoveReference( new hkCountedReferencedObject() ) );

            HK_TEST( holder.m_refObject->getReferenceCount()==1 );

            holder.doSomethingWithObject( hkAutoRemoveReference( new hkCountedReferencedObject() ) );

            holder.setObject( hkAutoRemoveReference( new hkCountedReferencedObject() ) );

            HK_TEST( holder.m_refObject->getReferenceCount()==1 );
        }
    }
    hkCountedReferencedObject::endTest();
}

static void baseTypes_ufloat8Test()
{
    hkPseudoRandomGenerator random(1);
    {
        for (int i =0; i < 1000; i++)
        {
            hkReal x = random.getRandRange( 0, hkUFloat8_maxValue );
            hkReal maxError = hkMath::max2( hkReal(hkUFloat8_eps), hkMath::fabs(x * 0.1f) );
            hkUFloat8 half = float(x);
            hkReal uncompressed = half;
            HK_TEST( hkMath::equal(uncompressed, x, maxError));
        }
    }
    {
        for (int i =0; i < 1000; i++)
        {
            hkReal x = random.getRandRange( 0.0f, 100.0f );
            hkReal maxError = hkMath::max2(  hkReal(hkUFloat8_eps), hkMath::fabs(x * 0.1f) );
            hkUFloat8 half = float(x);
            hkReal uncompressed = half;
            HK_TEST( hkMath::equal(uncompressed, x, maxError));
        }
    }
    {
        for (int i =0; i < 1000; i++)
        {
            hkReal x = random.getRandRange( 0.0, 0.1f );
            hkReal maxError = hkMath::max2(  hkReal(hkUFloat8_eps), hkMath::fabs(x * 0.1f) );
            hkUFloat8 half = float(x);
            hkReal uncompressed = half;
            HK_TEST( hkMath::equal(uncompressed, x, maxError));
        }
    }
}

static void baseTypes_hkPtrTest()
{
    hkCountedReferencedObject::initTest();
    {
        hkRefPtr<hkReferencedObject> refPtr( hkAutoRemoveReference( new hkCountedReferencedObject() ));
        HK_TEST( refPtr->getReferenceCount()==1 );

        hkRefPtr<hkReferencedObject> refPtr2;
        refPtr2 = hkAutoRemoveReference( new hkCountedReferencedObject() );
        HK_TEST( refPtr2->getReferenceCount()==1 );

        // self assignment test
        {
            hkReferencedObject* value = refPtr.val();
            refPtr = refPtr;
            if(HK_TEST(refPtr.val() == value)) // unchanged
            {
                HK_TEST(refPtr->getReferenceCount() == 1); // reference count is unchanged
            }

            hkViewPtr<hkReferencedObject> ptr2 = refPtr2;
            value = ptr2;
            ptr2 = ptr2;
            HK_TEST(ptr2 == value); // unchanged
        }
    }
    hkCountedReferencedObject::endTest();
}

template<typename NTYPE, typename ATYPE>
void testEndianType(ATYPE src)
{
    hkEndian::EndianType<NTYPE,false> d0 = src;
    hkEndian::EndianType<NTYPE,true> d1 = src;
    HK_TEST( (NTYPE)d0 == src );
    HK_TEST( (NTYPE)d1 == src );
}


static void baseTypes_hkEndianTest()
{
    HK_TEST( hkEndian::swap(0x01020304) == 0x04030201);
    HK_TEST( hkEndian::swap(hkUint16(0x0102)) == 0x0201);
    HK_TEST( hkEndian::swap(hkInt8(-66)) == -66);
    HK_TEST( hkEndian::swap(0xf1f2f3f4) == 0xf4f3f2f1);
    HK_TEST( hkEndian::swap(0xf1f2f3f4f5f6f7f8ull) == 0xf8f7f6f5f4f3f2f1ull);
    HK_TEST( hkEndian::swap(hkEndian::swap(hkInt64(-55))) == -55);

    hkFloat32 fval = 101.1;
    HK_TEST( hkEndian::swap(hkEndian::swap(fval)) == fval);
    hkDouble64 dval = 202.1;
    HK_TEST( hkEndian::swap(hkEndian::swap(dval)) == dval);

    testEndianType<hkFloat32>(1.12421f);
    testEndianType<hkFloat32>(11);
    testEndianType<hkFloat32>(-11);
};

static void baseTypes_hkUuidTest()
{
    hkUuid uuid = hkUuid::getNil();
    HK_TEST(uuid == hkUuid::getNil());

    hkStringBuf buffer;
    uuid.toString(buffer);
    HK_TEST(buffer.compareToIgnoreCase("00000000-0000-0000-0000-000000000000") == 0);

    uuid.setFromFormattedString("6ba7b810-9dAD-11d1-80b4-00c04fd430C8");
//  HK_TEST(uuid != hkUuid::getNil());
//  HK_TEST(uuid.m_timeLow == 0x6ba7b810);
//  HK_TEST(uuid.m_timeMid == 0x9dad);
//  HK_TEST(uuid.m_timeHiAndVersion == 0x11d1);
//  HK_TEST(uuid.m_clockSeqHiAndReserved == 0x80);
//  HK_TEST(uuid.m_clockSeqLow == 0xb4);
//  HK_TEST(uuid.m_node[0] == 0x00);
//  HK_TEST(uuid.m_node[1] == 0xc0);
//  HK_TEST(uuid.m_node[2] == 0x4f);
//  HK_TEST(uuid.m_node[3] == 0xd4);
//  HK_TEST(uuid.m_node[4] == 0x30);
//  HK_TEST(uuid.m_node[5] == 0xc8);

    buffer.clear();
    uuid.toString(buffer);
    HK_TEST(buffer.compareToIgnoreCase("6ba7b810-9dad-11d1-80b4-00c04fd430c8") == 0);

    hkUuid uuid1; uuid1.setRandom();
    hkUuid uuid2; uuid2.setRandom();
    hkUuid uuid3; uuid3.setRandom();

    HK_TEST(uuid1 != uuid2);
    HK_TEST(uuid1 != uuid3);
    HK_TEST(uuid2 != uuid3);
}

int baseTypes_main()
{
    baseTypes_halfTest();
    baseTypes_ufloat8Test();
    baseTypes_refptrTest();
    baseTypes_removeReferenceTest();
    baseTypes_hkPtrTest();
    baseTypes_hkUuidTest();
    baseTypes_hkEndianTest();
    return 0;
}

HK_TEST_REGISTER(baseTypes_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__     );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
