// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>

namespace sortimpl
{
    struct ListElement
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE, sortimpl::ListElement);

        static HK_INLINE bool HK_CALL less(const ListElement& eA, const ListElement& eB)    {   return &eA < &eB;   }
        static HK_INLINE ListElement*& HK_CALL next(ListElement* p)                         {   return p->m_next;   }

        ListElement* m_next;
    };
}

template <class T>
static int validateSorted(T* elems, int numElems)
{
    for(int i = 1; i < numElems; i++)
    {
        if(elems[i] < elems[i - 1])
        {
            return i;
        }
    }
    return numElems;
}

int sort_main()
{
    const int NTEST = 50;
    const int ASIZE = 500;
    const int MAX = 100000;

    for(int i=0; i<NTEST; ++i)
    {
        int array[ASIZE];
        const int n = int(hkUnitTest::randRange( 0, hkReal(ASIZE) ));
        _Analysis_assume_(0 <= n && n < ASIZE);

        for(int j=0; j<n; ++j)
        {
            array[j] = (int)hkUnitTest::randRange(1, (float)MAX);
        }

        if (i&1)
        {
            hkAlgorithm::quickSort(array, n);
        }
        else
        {
            hkAlgorithm::heapSort(array, n);
        }

        {
            int sorted = validateSorted(array, n);
            HK_TEST2(sorted == n, "Sort failed on iteration " << i << " element "<< sorted << (i&1?" (quicksort).":" (heapsort).") );
        }

        // String sort
        {
            hkStringPtr tags[ASIZE];
            for(int j=0; j<n; ++j)
            {
                hkStringBuf tag;
                tag.format("elem {}", (int)hkUnitTest::randRange(1, (float)MAX));
                tags[j] = tag;
            }

            if (i&1)
            {
                hkAlgorithm::quickSort(tags, n);
            }
            else
            {
                hkAlgorithm::heapSort(tags, n);
            }

            {
                int sorted = validateSorted(tags, n);
                HK_TEST2(sorted == n, "Sort failed on iteration " << i << " element "<< sorted << (i&1?" (quicksort).":" (heapsort).") );
            }
        }

    }

    // RADIX sort
    {
        for(int i=0; i<NTEST; ++i)
        {
            hkRadixSort::SortData32 array[ASIZE];
            hkRadixSort::SortData32 buffer[ASIZE];
            const int n = (~0x3) & int(hkUnitTest::randRange( 0, hkReal(ASIZE) ));
            int j;

            for(j=0; j<n; ++j)
            {
                array[j].m_key = (int)hkUnitTest::randRange(1, hkReal(0x10000000));
            }

            hkRadixSort::sort32(array, n, buffer);

            for(j=0; j<n-1; ++j)
            {
                if(array[j].m_key > array[j+1].m_key)
                {
                    break;
                }
            }

            if( n > 0)
            {
                HK_TEST2(j == n-1, " radix sort failed on iteration " << i << " element "<< j << (i&1?"(quicksort)":"(heapsort)") );
            }
        }
    }

    // List sort
    {
        hkPseudoRandomGenerator prng(0);
        for( int asize = 2; asize < 100; ++asize )
        {
            for( int iter = 0; iter < 10; ++iter )
            {
                // create random linked list
                sortimpl::ListElement* head;
                hkArray<sortimpl::ListElement>::Temp arr(asize);
                {
                    hkArray<int>::Temp ind(asize);
                    for( int i = 0; i < asize; ++i )
                    {
                        ind[i] = i;
                    }
                    prng.shuffle( ind.begin(), asize);

                    for( int i = 0; i < asize-1; ++i )
                    {
                        arr[ ind[i] ].m_next = &arr[ ind[i+1] ];
                    }
                    arr[ ind[asize-1] ].m_next = HK_NULL;
                    head = &arr[ ind[0] ];
                }

                // sort
                head = hkAlgorithm::sortList( head, sortimpl::ListElement::next, sortimpl::ListElement::less );

                // test it was sorted
                int count = 0;
                while( head != HK_NULL )
                {
                    count += 1;
                    HK_TEST( head->m_next == HK_NULL || head < head->m_next );
                    head = head->m_next;
                }
                HK_TEST( count == asize );
            }
        }
    }

    // quickfind-K
    for(int i=0; i<NTEST; ++i)
    {
        int array[ASIZE];
        const int n = int(hkUnitTest::randRange( 0, hkReal(ASIZE) ));
        int j;

        for(j=0; j<n; ++j)
        {
            array[j] = (int)hkUnitTest::randRange(1, (float)MAX);
        }

        const int k = int(hkUnitTest::randRange( 0, hkReal(n) ));

        hkAlgorithm::quickFindK(array, n, k);

        // sort the lowest-k and highest-n-minus-k ranges
        hkAlgorithm::quickSort(array, k);
        hkAlgorithm::quickSort(array+k, n-k);

        // make sure they are disjoint
        HK_TEST(k == 0 || array[k-1] <= array[k]);
    }

    return 0;
}

HK_TEST_REGISTER(sort_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__     );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
