// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
#include <Common/Base/Container/Hash/hkHashMap.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>

template< template<typename K, typename V> class MAP_TYPE, typename KEY>
static void test_maps()
{
    static const int s_removeEvery = 4;

    for( int rep = 1; rep < 200; ++rep )
    {
        const int N = rep;

        typedef MAP_TYPE<KEY, int> MapType;
        MapType pmap;
        hkArray<KEY> keys(N);

        int i;
        for( i = 0; i < N; ++i )
        {
            while(1)
            {
                KEY key = int( hkUnitTest::rand01() * 1000 * rep );
                if( (keys.indexOf(key) == -1) )
                {
                    keys[i] = key;
                    break;
                }
            }
            pmap.insert(keys[i], i);
        }

        // Remove a subset of the keys.
        for( i = 0; i < N; ++i )
        {
            if ( i % s_removeEvery == 0 )
            {
                pmap.remove(keys[i]);
            }
        }

        const int expectedSize = (int) (hkMath::floor( ( 1.0f - (1.0f / s_removeEvery) ) * N ) );
        HK_TEST( pmap.getSize() == expectedSize );
        for( i = 0; i < N; ++i )
        {
            if ( i % s_removeEvery != 0 )
            {
                int out = 0;
                HK_TEST( pmap.get(keys[i], &out).isSuccess() );
                HK_TEST( out == i );
            }
        }
        for( i = 0; i < keys.getSize(); ++i )
        {
            // Remove those keys which have already removed from the map.
            if ( i % (s_removeEvery - 1) == 0 )
            {
                keys.removeAtAndCopy(i);
            }
        }
        {
            for(typename MapType::Iterator it = pmap.getIterator();
                pmap.isValid(it);
                it = pmap.getNext(it) )
            {
                int idx = keys.indexOf( pmap.getKey(it) );
                HK_TEST( idx != -1 );
                keys.removeAt( idx );
            }
            HK_TEST( keys.getSize() == 0 );
        }

        {
            for(typename MapType::Iterator it = pmap.getIterator();
                pmap.isValid(it);
                it = pmap.getNext(it) )
            {
                int val = pmap.getValue(it);
                pmap.setValue(it, val*10);
                HK_TEST( pmap.getValue(it) == val*10 );
            }
        }
    }
}


namespace
{
    // Objects for testing non-POD maps
    struct PodCountLiveObjects
    {
        PodCountLiveObjects() : m_live(true)
        {
            s_totalLiveObjectCount++;
        }

        PodCountLiveObjects(int value) : m_live(true), m_value(value)
        {
            s_totalLiveObjectCount++;
        }

        PodCountLiveObjects(PodCountLiveObjects const& other) : m_live(true), m_value(other.m_value)
        {
            m_live = true;
            s_totalLiveObjectCount++;
        }

        PodCountLiveObjects& operator=(PodCountLiveObjects const& other)
        {
            HK_TEST(m_live);
            m_value = other.m_value;
            return *this;
            // Do nothing
        }

        ~PodCountLiveObjects()
        {
            HK_TEST(m_live);
            s_totalLiveObjectCount--;
            m_value = -1;
            m_live = false;
        }

        int m_live;
        int m_value;
        static int s_totalLiveObjectCount;
    };

    int PodCountLiveObjects::s_totalLiveObjectCount = 0;

    struct ObjKey
    {
        ObjKey()
        {
            s_totalLiveObjectCount++;
        }

        ObjKey(int id) : m_id(id)
        {
            s_totalLiveObjectCount++;
        }

        ObjKey(const ObjKey& other) : m_id(other.m_id)
        {
            s_totalLiveObjectCount++;
        }

        ObjKey& operator=(ObjKey const& other)
        {
            // We're already alive. Map probably never does this
            m_id = other.m_id;
            return *this;
        }
        bool operator==(const ObjKey& o) const
        {
            return m_id == o.m_id;
        }

        ~ObjKey()
        {
            s_totalLiveObjectCount--;
            m_id = -1;
        }

        hkUint32 getHash() const { using namespace hkHash; return hkHashValue(m_id); }

        int m_id;
        static int s_totalLiveObjectCount;
    };

    int ObjKey::s_totalLiveObjectCount = 0;
}

template <>
struct hkMapOperations<ObjKey>
{
    inline static unsigned hash( const ObjKey& key, unsigned mod )
    {
        return unsigned(key.m_id) & mod;
    }
    inline static void invalidate( ObjKey& key )
    {
        // Destructor has already been called, this is just invalidating the memory
        *reinterpret_cast<void**>(&key) = HK_NULL;
    }
    inline static hkBool32 isValid( const ObjKey& key )
    {
        // In general this is not a valid hkRefPtr, so just check the memory
        return *reinterpret_cast<const void* const*>(&key) != HK_NULL;
    }
    inline static hkBool32 equal( const ObjKey& key0, const ObjKey& key1 )
    {
        return key0.m_id == key1.m_id;
    }
};


template< template<typename K,typename V> class MAP_TYPE>
static void test_nonPodMaps()
{
    // Test destruction counts
    {
        {
            typedef MAP_TYPE<int, PodCountLiveObjects> MapType;
            MapType map;

            hkPseudoRandomGenerator rng(2222);
            hkArray<int> keys;
            for(int i=0; i < 8; i++)
            {
                keys.pushBack(rng.getRandInt16(1000));
                map.insert(keys.back(), PodCountLiveObjects(i));
            }

            HK_TEST(PodCountLiveObjects::s_totalLiveObjectCount == 8);

            // Test retrieval
            {
                PodCountLiveObjects valOut(-20);
                for(int i=0; i < 8; i++)
                {
                    HK_TEST(map.get(keys[i], &valOut).isSuccess());
                    HK_TEST(valOut.m_value == i);
                }
            }

            // Overwrite members
            map.insert(keys[7], PodCountLiveObjects(-1));
            map.insert(keys[6], PodCountLiveObjects(-2));
            map.insert(keys[5], PodCountLiveObjects(-3));

            HK_TEST(PodCountLiveObjects::s_totalLiveObjectCount == 8);

            // Test retrieval
            {
                PodCountLiveObjects valOut(-20);
                for(int i=0; i < 5; i++)
                {
                    HK_TEST(map.get(keys[i], &valOut).isSuccess());
                    HK_TEST(valOut.m_value == i);
                }
                HK_TEST(map.get(keys[5], &valOut).isSuccess());
                HK_TEST(valOut.m_value == -3);
                HK_TEST(map.get(keys[6], &valOut).isSuccess());
                HK_TEST(valOut.m_value == -2);
                HK_TEST(map.get(keys[7], &valOut).isSuccess());
                HK_TEST(valOut.m_value == -1);

                HK_TEST(!map.get(20000, &valOut).isSuccess());
                HK_TEST(!map.get(-10, &valOut).isSuccess());

            }
            // Still only 8 live objects, we should have destructed all of the temporaries we created along the way
            HK_TEST(PodCountLiveObjects::s_totalLiveObjectCount == 8);
        }
        HK_TEST(PodCountLiveObjects::s_totalLiveObjectCount == 0);
    }

    // Test using a non-POD key
    {
        ObjKey::s_totalLiveObjectCount = 0;
        PodCountLiveObjects::s_totalLiveObjectCount = 0;
        {
            typedef MAP_TYPE<ObjKey, PodCountLiveObjects> MapType;
            MapType map;

            ObjKey key1(1);
            ObjKey key2(2);
            ObjKey key3(3);
            ObjKey key4(4);
            ObjKey key5(5);

            // Just 5 keys
            HK_TEST(ObjKey::s_totalLiveObjectCount == 5);

            map.insert(key1, PodCountLiveObjects(1));
            map.insert(key2, PodCountLiveObjects(2));
            map.insert(key3, PodCountLiveObjects(3));
            map.insert(key4, PodCountLiveObjects(4));
            map.insert(key5, PodCountLiveObjects(5));

            // 5 Keys here plus 5 in the map
            HK_TEST(ObjKey::s_totalLiveObjectCount == 10);
            HK_TEST(PodCountLiveObjects::s_totalLiveObjectCount == 5);

            {
                // Test retrieval
                HK_TEST(map.getWithDefault(key1, PodCountLiveObjects(-100)).m_value == 1);
                HK_TEST(map.getWithDefault(key2, PodCountLiveObjects(-100)).m_value == 2);
                HK_TEST(map.getWithDefault(key3, PodCountLiveObjects(-100)).m_value == 3);
                HK_TEST(map.getWithDefault(key4, PodCountLiveObjects(-100)).m_value == 4);
                HK_TEST(map.getWithDefault(key5, PodCountLiveObjects(-100)).m_value == 5);
            }

            // Temporaries all destroyed, 5 Keys here plus 5 in the map
            HK_TEST(ObjKey::s_totalLiveObjectCount == 10);
            HK_TEST(PodCountLiveObjects::s_totalLiveObjectCount == 5);

            map.insert(key3, PodCountLiveObjects(6));

            // Test it has actually been inserted
            HK_TEST(map.getWithDefault(key3, PodCountLiveObjects(-100)).m_value == 6);

            // 5 Keys here plus 5 in the map
            // Should have destructed the old ones
            HK_TEST(ObjKey::s_totalLiveObjectCount == 10);
            HK_TEST(PodCountLiveObjects::s_totalLiveObjectCount == 5);

            map.insert(ObjKey(6), PodCountLiveObjects(7));

            // Test it has actually been inserted
            HK_TEST(map.getWithDefault(ObjKey(6), PodCountLiveObjects(-100)).m_value == 7);

            // One key, one value added to map
            HK_TEST(ObjKey::s_totalLiveObjectCount == 11);
            HK_TEST(PodCountLiveObjects::s_totalLiveObjectCount == 6);

            map.insert(ObjKey(2), PodCountLiveObjects(8));

            // Test it has actually been inserted
            HK_TEST(map.getWithDefault(ObjKey(2), PodCountLiveObjects(-100)).m_value == 8);

            // Replace an existing key, replace an existing value
            HK_TEST(ObjKey::s_totalLiveObjectCount == 11);
            HK_TEST(PodCountLiveObjects::s_totalLiveObjectCount == 6);

            map.remove(ObjKey(1));

            // Removed one key, one value
            HK_TEST(ObjKey::s_totalLiveObjectCount == 10);
            HK_TEST(PodCountLiveObjects::s_totalLiveObjectCount == 5);

            // Test it has actually been inserted
            HK_TEST(map.getWithDefault(ObjKey(1), PodCountLiveObjects(-100)).m_value == -100);

            hkResult res = map.remove(ObjKey(1));

            // Nothing removed
            HK_TEST(!res.isSuccess());
            HK_TEST(ObjKey::s_totalLiveObjectCount == 10);
            HK_TEST(PodCountLiveObjects::s_totalLiveObjectCount == 5);
        }
        HK_TEST(ObjKey::s_totalLiveObjectCount == 0);
        HK_TEST(PodCountLiveObjects::s_totalLiveObjectCount == 0);
    }
}

template< template<typename K, typename V> class MAP_TYPE>
static void test_stringMaps()
{
    typedef MAP_TYPE<const char*, int> MapType;
    MapType map;
    map.insert("foo", 1);
    hkStringBuf foo("f", "o", "o");
    HK_TEST(map.getWithDefault(foo, 0) == 1);
}

namespace
{

    template <typename T>
    struct PseudoOrdered : public T
    {
        bool operator<(const PseudoOrdered& p) const { return m_order < p.m_order; }
        hkUint32 m_order;
    };
}

template< template<typename K, typename V> class MAP_TYPE>
static void test_copying()
{
    typedef MAP_TYPE<hkStringPtr, hkStringPtr> MapType;
    {
        MapType c0;
        {
            MapType orig;
            orig.insert("key0", "val0");
            orig.insert("key1", "val1");
            orig.insert("key2", "val2");
            orig.insert("key3", "val3");

            c0 = orig;
        }
        //assignment from empty
        HK_TEST_EQ(c0.getWithDefault("key2", HK_NULL), "val2");

        //self assignment
        c0 = c0;
        HK_TEST_EQ(c0.getWithDefault("key2", HK_NULL), "val2");

        //assignment when non-empty
        MapType c1(c0);
        c1.insert("key4", "val4");
        c0 = c1;
        HK_TEST_EQ(c0.getWithDefault("key4", HK_NULL), "val4");
    }

    {
        hkUnitTest::Prng prng;
        hkArray< PseudoOrdered< MapType > > maps;
        maps.setSize(5);
        for(int mi = 0; mi < maps.getSize(); ++mi)
        {
            maps[mi].m_order = prng.nextUint32();
            for(int i = 0; i < mi; ++i)
            {
                hkStringBuf k; k.format("{}", i * 7);
                hkStringBuf v; v.format("{}", i * 13);
                maps[mi].insert(k.cString(), v.cString());
            }
        }
        hkSort( maps.begin(), maps.getSize() );
    }
}


namespace MapTest
{

    struct TestKey
    {
        TestKey(int i = 0) : m_int( i ) {}
        TestKey(const TestKey& other) : m_int( other.m_int ) {}

        // A deliberately clashing hash (always returns 14, 15 or 16).
        hkUint32 hash() const { return 14 + (m_int % 3); }

        bool operator==(const TestKey& other) const { return m_int == other.m_int; }
        bool operator!=(const TestKey& other) const { return m_int != other.m_int; }
        int m_int;
    };

    template<typename K, typename V>
    struct PointerMap : public hkPointerMap < K, V > {};
    template<typename K, typename V>
    struct Map : public hkMap < K, V > { Map(){}  };
}

int maps_main()
{
    test_maps< MapTest::PointerMap, int>();
    test_maps< MapTest::Map, MapTest::TestKey>();
    test_maps< hkHashMap, int>();
    test_maps< hkHashMap, MapTest::TestKey>();
    test_nonPodMaps< MapTest::Map >();
    test_nonPodMaps< hkHashMap >();
    test_stringMaps< hkHashMap >();
    test_copying< hkHashMap >();
    return 0;
}

HK_TEST_REGISTER(maps_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__     );

template class hkMap<int, PodCountLiveObjects>;
template class hkMapBase<int, PodCountLiveObjects>;

template class hkMap<ObjKey, PodCountLiveObjects>;
template class hkMapBase<ObjKey, PodCountLiveObjects>;

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
