// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Algorithm/Hash/hkHash.h>

#include <Common/Base/System/Stopwatch/hkStopwatch.h>

static void algorithm()
{
    const int orig_x = 55;
    const int orig_y = 28;
    int x = orig_x;
    int y = orig_y;
    hkAlgorithm::swap(x,y);
    HK_TEST(orig_x==y);
    HK_TEST(orig_y==x);
}

static void array()
{
    int i;
    {
        hkArray<int> a;
        HK_TEST(a.getSize()==0);
        HK_TEST(a.isEmpty());
    }

    {
        hkArray<int> b(10);
        HK_TEST(b.getSize()==10);
        HK_TEST(!b.isEmpty());
        b.popBack();
        HK_TEST(b.getSize()==9);

        b.clear();
        HK_TEST(b.getSize()==0);
        HK_TEST(b.isEmpty());
        HK_TEST(b.getCapacity()!=0);
    }

    {
        hkArray<int> c(5, 99);
        HK_TEST(c.getSize()==5);
        for(i=0; i<c.getSize(); ++i)
        {
            HK_TEST(c[i]==99);
        }
        c.clearAndDeallocate();
        HK_TEST(c.getSize()==0);
        HK_TEST(c.getCapacity()==0);
    }

    {
        int p[] = { 0,1,2,3,4,5,6,7,8,9,10 };
        hkArray<int> d(p, 10, 10);
        d.removeAt(5);
        HK_TEST(d.getSize()==9);
        HK_TEST(d[5]!=5);
    }

    {
        hkArray<int> d(10);
        for(i=0; i<d.getSize(); ++i)
        {
            d[i] = i;
        }
        d.removeAt(5);
        HK_TEST(d.getSize()==9);
        HK_TEST(d[5]!=5);
        HK_TEST(d[4]==4);
        HK_TEST(d[6]==6);
    }

    {
        hkArray<int> d;
        d.setSize(10);
        for(i=0; i<d.getSize(); ++i)
        {
            d[i] = i;
        }
        d.removeAtAndCopy(5);
        HK_TEST(d.getSize()==9);
        HK_TEST(d[4]==4);
        HK_TEST(d[5]==6);
        HK_TEST(d[6]==7);
        HK_TEST(d.indexOf(4)==4);
        HK_TEST(d.indexOf(5)==-1);
        HK_TEST(d.indexOf(6)==5);
    }

    {
        hkArray<int> d(10);
        for(i=0; i<d.getSize(); ++i)
        {
            d[i] = i%5;
        }
        HK_TEST(d.indexOf(3)==3);
        HK_TEST(d.lastIndexOf(3)==8);
    }

    {
        hkArray<int> d;
        for(i=0; i<10; ++i)
        {
            d.pushBack(i);
        }
        for(i=0; i<10; ++i)
        {
            HK_TEST(d[i]==i);
        }
    }

    {
        hkArray<int> d;
        d.reserve(10);
        for(i=0; i<10; ++i)
        {
            d.pushBackUnchecked(i);
        }
        HK_TEST(d.getSize()==10);
        for(i=0; i<10; ++i)
        {
            HK_TEST(d[i]==i);
        }
    }

    {
        hkArray<int> d;
        d.reserveExactly(10);
        for(i=0; i<10; ++i)
        {
            d.pushBackUnchecked(i);
        }
        HK_TEST(d.getSize()==10);
        for(i=0; i<10; ++i)
        {
            HK_TEST(d[i]==i);
        }
    }

    {
        hkArray<int> d;
        d.reserveExactly(10);
        for(i=0; i<10; ++i)
        {
            d.pushBackUnchecked(i);
        }
        for(i=0; i<10; ++i)
        {
            HK_TEST(d[i]==i);
        }
    }

    {
        hkArray<int> d;
        d.reserveExactly(10);
        d.setSizeUnchecked(5);
        HK_TEST(d.getSize()==5);
        HK_TEST(d.getCapacity()>=10);
    }

    {
        hkArray<int> d;
        for(i=0; i<10; ++i)
        {
            d.pushBack(i);
        }
        d.insertAt(0,99);
        HK_TEST(d.getSize()==11);
        HK_TEST(d[0]==99);
        for(i=0; i<10; ++i)
        {
            HK_TEST(d[i+1] == i);
        }
    }

    {
        hkArray<int> d;
        hkArray<int> e;
        for(i=0; i<10; ++i)
        {
            d.pushBack(i);
            e.pushBack(-i);
        }
        d.insertAt(3,e.begin(), e.getSize());
        HK_TEST(d.getSize()==20);
        HK_TEST(d[2]==2);
        HK_TEST(d[3]==0);
        HK_TEST(d[4]==-1);
        HK_TEST(d[12]==-9);
        HK_TEST(d[13]==3);
        HK_TEST(d[14]==4);
        HK_TEST(d[19]==9);

        d.swap(e);
    }

    {
        hkInplaceArray<int, 10> a;
        HK_TEST(a.getCapacity()>=10);
        for(i=0; i<20; ++i)
        {
            a.pushBack(i);
        }
        HK_TEST(a.getSize()==20);
        HK_TEST(a.getCapacity()>=20);

    }
}

static void pseudorandom()
{
    hkPseudoRandomGenerator rand(1234);

    const int sizes[2] = {200, 70000};
    for (int i=0; i<2; i++)
    {
        const int N = sizes[i];
        hkArray<int> array1; array1.setSize(N);

        for(int j=0; j<N; j++)
        {
            array1[j] = j;
        }

        // Make sure that shuffle works for a big (>65K) size
        rand.shuffle( array1.begin(), array1.getSize() );
    }
}

static void hash()
{
    {
        static char data[] = "This is just a little test string for, you know, testing...";
        hkUint32 dataSize = sizeof( data ) - 1;
        hkUint32 seed = 0x020410FF;

        // Validation for CRC32 implementations.
        {
            hkUint32 expectedCrc32_noSeed = 0x8a3d71af;
            hkUint32 expectedCrc32_withSeed = 0xba6c0af9;

            HK_TEST( expectedCrc32_noSeed == hkHash::computeCrc32( data, dataSize ) );
            HK_TEST( expectedCrc32_withSeed == hkHash::appendCrc32( seed, data, dataSize ) );

            {
                hkUint32 crc32 = hkHash::computeCrc32( data, dataSize / 2 );
                HK_TEST( expectedCrc32_noSeed == hkHash::appendCrc32( crc32, &data[ dataSize / 2 ], dataSize - ( dataSize / 2 ) ) );
            }

            {
                hkUint32 crc32 = hkHash::appendCrc32( seed, data, dataSize / 2 );
                HK_TEST( expectedCrc32_withSeed == hkHash::appendCrc32( crc32, &data[ dataSize / 2 ], dataSize - ( dataSize / 2  ) ) );
            }
        }

        // Validation for MurmurHash3 implementation.
        {
            hkUint32 expectedMurmurHash3_noSeed = 0x99b0ab03;
            hkUint32 expectedMurmurHash3_withSeed = 0xeab0babf;

            HK_TEST( expectedMurmurHash3_noSeed == hkHash::computeMurmurHash3( data, dataSize ) );
            HK_TEST( expectedMurmurHash3_withSeed == hkHash::computeMurmurHash3( data, dataSize, seed ) );
        }
    }

#if 0   // Performance tests are not part of the standard test suite, but are kept in here for future reference.
    // Performance tests.
    {
        const hkUint32 numElements = 1024;
        const int numIterations = 1024;

        hkPseudoRandomGenerator rand(1234);
        hkArray< hkUint32 > data( numElements );
        hkUint32 dataSize = data.getSize() * sizeof( hkUint32 );
        for ( int i = 0; i < data.getSize(); ++i )
        {
            data[i] = rand.getRand32();
        }

        {
            hkStopwatch stopwatch;
            stopwatch.start();
            hkUint32 hash = 0x0;

            for ( int i = 0; i < numIterations; ++i )
            {
                hash = hkHash::computeCrc32( &data[0], dataSize );
            }

            stopwatch.stop();

            Log_Info( "hkHash::computeCrc32() - time needed: {} ms", stopwatch.getElapsedSeconds() * 1000.0f );
        }

        {
            hkStopwatch stopwatch;
            stopwatch.start();
            hkUint32 hash = 0x0;

            for ( int i = 0; i < numIterations; ++i )
            {
                hash = hkHash::computeMurmurHash3( &data[0], dataSize );
            }

            stopwatch.stop();

            Log_Info( "hkHash::computeMurmurHash3() - time needed: {} ms", stopwatch.getElapsedSeconds() * 1000.0f );
        }
    }
#endif


}

int htl_main()
{
    algorithm();
    array();
    pseudorandom();
    hash();
    return 0;
}

HK_TEST_REGISTER(htl_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__     );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
