// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Container/Hash/hkHashMap.h>
#include <Common/Base/Types/hkScopedPtr.h>

class hkReadersWriterLock;
class hkReadLock;

template<typename T>
class hkWeakPtr;

namespace hkWeakPtrDetail
{
    class Status;

    template<typename T>
    _Ret_maybenull_ Status* getStatus(const hkWeakPtr<T>& ptr);
}

template<typename T>
hkUint32 hkHashValue(const hkWeakPtr<T>& ptr);

/// hkWeakPtr is a pointer which can point to any hkReferencedObject without adding a reference. If the
/// object dies then the hkWeakPtr will resolve to null. It is modeled after std::weak_ptr:
///     http://en.cppreference.com/w/cpp/memory/weak_ptr
/// The canonical usage pattern is:
///    hkWeakPtr<Foo> ptr = /* create weak ptr from an existing hkRefPtr */;
///    if(hkRefPtr<Foo> obj = ptr.acquire())
///    {
///        // use obj safely
///    }
///
/// It's interface has been designed with special care to be safe and catch as many misuses as possible.
/// In particular a hkWeakPtr must be constructed from an hkRefPtr to ensure that the object stays alive
/// during construction. To avoid spurious reference count operations most methods take a const hkRefPtr&
/// and a special "IndirectRefPtr" otherwise which will create a hkRefPtr from other pointer types like T*.
///
/// It is reflected & serializable though note that usage through the reflection interface
/// is not safe because reflection can't guarantee that the object will stay alive.
///
/// hkWeakPtr can be used in hkHashSet or as key in hkHashMap.
template<typename T>
class hkWeakPtr
{
    public:

        HK_DECLARE_CLASS(hkWeakPtr, New, Reflect);

        HK_REFLECT_AS_POINTER(&hkWeakPtrDetail::WeakReferencedObjectPointerImpl::s_instance, T);
        HK_RECORD_ATTR(hk::AddFlags(hkReflect::Type::TYPE_POINTER_WEAK));

            /// Create a null pointer.
        hkWeakPtr();

            /// Create a weak ptr from another weak ptr.
        template<typename U>
        hkWeakPtr(const hkWeakPtr<U>& obj);

            /// Create a weak ptr from an existing hkRefPtr.
        template<typename U>
        hkWeakPtr(const hkRefPtr<U>& obj);

            /// Create a weak ptr from an existing hkViewPtr.
        template<typename U>
        hkWeakPtr(const hkViewPtr<U>& obj);

            /// Create a weak ptr from a T*.
        template<typename U>
        hkWeakPtr(_In_opt_ U* obj);

            /// Special casing for constructing a weak pointer with HK_NULL
        hkWeakPtr(_In_opt_ const void* obj);

        /// Copy obj into this weak ptr.
        template<typename U>
        void set(const hkWeakPtr<U>& obj);

            /// Set the weak ptr to reference obj using an existing hkRefPtr.
        template<typename U>
        void set(const hkRefPtr<U>& obj);

            /// Set the weak ptr to a referenced object within a view pointer
        template<typename U>
        void set(const hkViewPtr<U>& obj);

            /// Set the weak ptr to reference obj using a T*.
        template<typename U>
        void set(_In_opt_ U* obj);

            /// Special casing for setting a pointer to null with ptr.set(HK_NULL)
        void set (_In_opt_ const void* obj);

            /// Copy obj into this weak ptr.
        template<typename U>
        hkWeakPtr<T>& operator=(const hkWeakPtr<U>& rhs);

        /// Set the weak ptr to reference obj using an existing hkRefPtr.
        template<typename U>
        hkWeakPtr<T>& operator=(const hkRefPtr<U>& rhs);

        /// Set the weak ptr to reference obj using an existing hkViewPtr.
        template<typename U>
        hkWeakPtr<T>& operator=(const hkViewPtr<U>& rhs);

            /// Set the weak ptr to reference obj using a T*.
        template<typename U>
        hkWeakPtr<T>& operator=(_In_opt_ U* obj);

            /// Special casing for setting to null using ptr = HK_NULL
        hkWeakPtr<T>& operator=(_In_opt_ const void* obj );

            /// Returns a non-null pointer if the object is still alive, null otherwise.
            /// For safety purposes you must assign the return value to an hkRefPtr.
        hkRefPtr<T> acquire() const;

            /// Use with caution! If you are absolutely certain that the object will not die
            /// during this function call then this will return the object if it is still alive
            /// or null otherwise. Since the return value is a raw pointer it will not keep the
            /// object alive either so you have no guarantees on its lifetime.
        _Ret_maybenull_ T* unsafeAcquire() const;

            /// Returns true if the referenced object is dead, false otherwise.
            /// Note that you shouldn't use this function to check that an object is still alive
            /// because it may be destroyed in a separate thread creating a race condition in
            /// your code.
        bool isDead() const;

            /// Returns true if both pointers point to the same object even if it is dead. Null pointers compare equal.
        bool operator==(const hkWeakPtr<T>& other) const;

    private:

        template<typename U>
        friend class hkWeakPtr;

        template<typename U>
        friend hkUint32 hkHashValue(const hkWeakPtr<U>& ptr);

        template<typename U>
        friend hkWeakPtrDetail::Status* hkWeakPtrDetail::getStatus(const hkWeakPtr<U>& ptr);

        // The value of m_status must not change as long as the hkWeakPtr is "the same" to maintain hash stability
        // (see hkHashValue(hkWeakPtr).
        mutable hkRefPtr<hkWeakPtrDetail::Status> m_status
            HK_ATTR(hk::OpaqueType);
};

namespace hkWeakPtrDetail
{
    class HK_EXPORT_COMMON WeakReferencedObjectPointerImpl : hkReflect::Detail::PointerImpl
    {
        public:

            WeakReferencedObjectPointerImpl() {}

            virtual hkResult getValue(_In_ const void* self, _In_opt_ const hkReflect::PointerType* type, _Out_ hkReflect::Var* val) const HK_OVERRIDE;
            virtual hkResult setValue(_Inout_ void* self, _In_ const hkReflect::PointerType* type, const hkReflect::Var& var) const HK_OVERRIDE;

            static const WeakReferencedObjectPointerImpl s_instance;
    };

    class HK_EXPORT_COMMON Status : public hkReferencedObject
    {
        public:

            HK_DECLARE_CLASS(Status, New);

            Status() : m_ptr(HK_NULL) {}
            Status(hkReferencedObject* ptr) : m_ptr(ptr) {}

            HK_ALWAYS_INLINE _Ret_maybenull_ hkReferencedObject* getPtr() const { return m_ptr; }
            HK_ALWAYS_INLINE void setDead() { m_ptr = HK_NULL; }

        private:

            hkReferencedObject* m_ptr;
    };

    class HK_EXPORT_COMMON StatusRegistry : public hkReferencedObject
    {
        public:

            class HK_EXPORT_COMMON ReadLock
            {
                public:

                    ReadLock(hkReadersWriterLock& lock);
                    ~ReadLock();

                private:

                    hkReadersWriterLock& m_lock;
            };

            HK_DECLARE_CLASS(StatusRegistry, New, Singleton);

            StatusRegistry();
            ~StatusRegistry();

            hkRefPtr<Status> getOrCreateStatus(const hkRefPtr<hkReferencedObject>& obj);

            void setDead(_Inout_ hkReferencedObject* obj);
            hkReadersWriterLock& getLock() { return *m_lock; }

        private:

            typedef hkHashMap< hkReferencedObject*, hkRefPtr<Status> > StatusMap;
            StatusMap m_objectStatusMap;
            hkScopedPtr<hkReadersWriterLock> m_lock;
    };

    template<typename U>
    _Ret_maybenull_ Status* getStatus(const hkWeakPtr<U>& ptr)
    {
        return ptr.m_status;
    }
}

#include <Common/Base/Types/hkWeakPtr.inl>
#include <Common/Base/_Auto/TemplateTypes/hkWeakPtr_Types.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
