// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Types/hkWeakPtr.h>
#include <Common/Base/Thread/ReadersWriterLock/hkReadersWriterLock.h>
#include <Common/Base/Object/hkSingleton.h>

hkResult hkWeakPtrDetail::WeakReferencedObjectPointerImpl::getValue(_In_ const void* vself, _In_opt_ const hkReflect::PointerType*, _Out_ hkReflect::Var* val) const
{
    const hkWeakPtr<hkReferencedObject>* self = static_cast< const hkWeakPtr<hkReferencedObject>* >(vself);

    if(hkRefPtr<hkReferencedObject> acquiredObj = self->acquire())
    {
        // This is unsafe, the object could die but we can't return a reference without leaking using reflection.
        *val = acquiredObj;
        return HK_SUCCESS;
    }
    *val = hkReflect::Var();
    return HK_SUCCESS;
}

hkResult hkWeakPtrDetail::WeakReferencedObjectPointerImpl::setValue(_Inout_ void* vself, _In_ const hkReflect::PointerType* type, const hkReflect::Var& var) const
{
    if(!hkReflect::Detail::canSetPtr(type->getSubType(), var))
    {
        return HK_FAILURE;
    }

    hkWeakPtr<hkReferencedObject>* self = static_cast<hkWeakPtr<hkReferencedObject>*>(vself);

    if(hkReferencedObject* target = hkDynCast(var))
    {
        // refcount == 0 can happen during cloning, in that case we don't want to trigger the destruction
        // of the object by making its refcount go to 1 then 0 again through the temporary hkRefPtr.
        const bool zeroRefCount = target->getReferenceCount() == 0;

        if(zeroRefCount)
        {
            target->addReference(); // pretend we have a ref.
        }

        self->set(target);

        if(zeroRefCount)
        {
            target->setReferenceCount(0); // Do not use removeReference() as that would destroy the object.
        }
    }
    else
    {
        self->set(HK_NULL);
    }

    return HK_SUCCESS;
}

const hkWeakPtrDetail::WeakReferencedObjectPointerImpl hkWeakPtrDetail::WeakReferencedObjectPointerImpl::s_instance;

hkWeakPtrDetail::StatusRegistry::StatusRegistry()
    : m_lock(new hkReadersWriterLock)
{
}

hkWeakPtrDetail::StatusRegistry::~StatusRegistry()
{
    // We need to clear the status map first because destroying the Status objects will query the StatusRegistry.
    m_objectStatusMap.clear();
}

hkRefPtr<hkWeakPtrDetail::Status> hkWeakPtrDetail::StatusRegistry::getOrCreateStatus(const hkRefPtr<hkReferencedObject>& obj)
{
#if !defined(HK_BUILDING_WITH_ENGINE)
    HK_FATAL_ERROR(0x1ac5f180, "WeakPtrs not available");
#endif
    HK_ASSERT_NO_MSG(0x1c5180, obj);

    hkReferencedObject* rawObj = obj;

    // Usual case: the status exists so we can just return it without modifying anything (only needs a Read lock).
    {
        hkReadLock lock(*m_lock);

        StatusMap::Iterator it = m_objectStatusMap.findKey(rawObj);
        if(m_objectStatusMap.isValid(it))
        {
            return m_objectStatusMap.getValue(it);
        }
    }

    // Will happen only once per object: create the new status and add in the map (requires Write lock).
    {
        hkWriteLock lock(*m_lock);

        hkRefPtr<Status> status = hkRefNew<Status>(new Status(rawObj));
        m_objectStatusMap.insert(rawObj, status);
        return status;
    }
}

void hkWeakPtrDetail::StatusRegistry::setDead(_Inout_ hkReferencedObject* obj)
{
    // Early out if we're deleting this object
    if(this == obj)
    {
        return;
    }

    // Early out for objects with no status
    {
        hkReadLock lock(*m_lock);
        if(m_objectStatusMap.contains(obj) == false)
        {
            return;
        }
    }

    {
        hkWriteLock lock(*m_lock);

        StatusMap::Iterator it = m_objectStatusMap.findKey(obj);
        HK_ASSERT_NO_MSG(0x4da05083, m_objectStatusMap.isValid(it));

        hkRefPtr<Status> status = m_objectStatusMap.getValue(it);
        status->setDead();

        m_objectStatusMap.remove(it);
    }
}

hkWeakPtrDetail::StatusRegistry::ReadLock::ReadLock(hkReadersWriterLock& lock)
    : m_lock(lock)
{
    m_lock.enterRO();
}

hkWeakPtrDetail::StatusRegistry::ReadLock::~ReadLock()
{
    m_lock.leaveRO();
}

HK_SINGLETON_IMPLEMENTATION(hkWeakPtrDetail::StatusRegistry);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
