// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Types/Traits/hkTraitModifier.h>

// Enable or disable variadic template-based automatic generation of the type array
// for a list of arguments of a variadic function (see HK_VARARGS1).
// The variadic template variant supports any number of argument but needs proper compiler support.
// The fallback variant is limited to a given number of arguments (see how many are unrolled below).
#if !defined(__HAVOK_PARSER__) && ( ( defined(HK_COMPILER_MSVC) && (_MSC_VER >= 1800) ) || (__cplusplus >= 201103) )
    #define HK_HAS_VARIADIC_TEMPLATE 1
#endif

namespace hkReflect
{
    struct Var;
    class Type;
    template<typename T> struct ReflectionOf;

    namespace Detail
    {
        /// Base for IsReflected trait, internal use.
        /// Will define ::Type to FalseType if the type definition is not visible (but will compile).
        template<typename T>
        struct IsReflectedBase
        {
            typedef hkTrait::TypesAreDifferent< typename ReflectionOf<T>::Holder, void > Type;
            enum { result = Type::result };
        };
    }

    /// IsReflected trait. ::Type will be TrueType and ::result 1 if T is reflected, FalseType and 0 otherwise.
    /// The trait cannot detect reflected typedefs, thus IsReflected<TypedefType> will not check if the typedef is reflected
    /// but rather the underlying type.
    template<typename T>
    struct IsReflected : public Detail::IsReflectedBase<T>
    {
        // This trait must not be used on incomplete types.
        static_assert(sizeof(typename hkTrait::VoidValue<T>::Type) > 0, "T is not reflected");
    };
}

/// Type-safe replacement for C/C++ varargs (...)
///
/// There are two APIs for providing typesafe varargs:
/// - One is the "list" version which corresponds closely to the standard C++ "...". It
///   is the most efficient in terms of performance and codesize, but slightly less convenient to use.
/// - The other option is "vector" version which does more work at the calling site, but is more convenient
///   to use. This is essentially an array of hkReflect::Var. However because it depends on hkReflect::Var,
///   it cannot be used in e.g. HK_ASSERT implementation as hkReflect::Var has not been included yet.
///
/// The list style is implemented here, and included early to be used in HK_ASSERT implementation.
/// The vector style is implemented in hkVarArgsV.h, and included later after hkReflect::Var has been defined.
///
///
/// ====== To use "list" style varargs ======
///
/// 1. Replace "..." with "hkVarArgs::VaTypes, ..." in the function prototype.
///    By convention we add an "L" suffix to the function name, but this it not required.
///      printf(const char* fmt, ... ) -> printfL( const char* fmt, hkVarArgs::VaTypes argsTypes, ... )
///
/// 2. At the call site, wrap the arguments in HK_VARARGS1. This will automatically generate the type array
///    for the given arguments, and prepend it to the list of arguments.
///      printf("name:%s cost:%i", name, cost) -> printfL( HK_VARARGS1("name:%s cost:%i", name, cost) );
///
/// 3. In the callee, you can use HK_VARARGS_UNPACK to convert to the vector-style interface and access
///    the arguments as a hkVarArgs::Vector (array of hkReflect::Var).
///      printfL( const char* fmt, hkVarArgs::VaTypes argTypes, ... )
///      {
///         hkVarArgs::FixedArray<10> args;
///         HK_VARARGS_UNPACK( args, argTypes ); // unpack at most 10 arguments into 'args'
///         for( int i = 0; i < args.getSize(); ++i ) {
///             if( hkReflect::IntVar h = args[i] ) ...
///             else if( hkReflect::ArrayVar h = args[i] ) ...
///
/// /!\ Note that hkVarArgs::VaTypes must be passed by value in order for the C language varargs feature to work correctly.
///
///
/// ====== To use "vector" style varargs ======
///
/// 1. Replace "..." with "const hkVarArgs::Vector& args" in the function prototype. By convention, we add
///    a "V" suffix to the method name also.
///      printf(const char* fmt, ...) -> printfV(const char* fmt, const hkVarArgs::Vector& args)
///
/// 2. Use one of the HK_VARARGS_FUNCTION_* macros to provide templated overloads which forward to the original function.
///
/// 3. The call site looks just like before, e.g. printf("name:%s cost:%i", name, cost);
///    The forwarding template methods ensure we have the correct types.
///
namespace hkVarArgs
{
    namespace Detail
    {
        /// Get the address of an argument passed to a variadic function.
        /// This will fill the m_addr member of hkReflect::Var if using the vector-style varargs.
        /// Note that before being cast to const void* on return:
        /// - addrOf<U*>(t) yields a U**, the address of the pointer variable 't'
        /// - addrOf<U[]>(t) yields a U*, the address of the array 't' (i.e. 't' itself in C/C++)
        template<typename T>
        static HK_ALWAYS_INLINE const void* addrOf(const T& t) { return &t; }

        /// Undefined struct for types we don't know how to deduce a reflection of.
        /// This means the type was used in a hkVarArgs, so we need some form of
        /// reflection, but the type is not reflected and we were unable to find
        /// a fallback type.
        struct PleaseCastArgumentToAKnownType;

        /// Fallback type for a non-reflected type.
        template<typename T>
        struct FallbackType
        {
            /// We don't know how to deduce a type for this argument.
            /// Ask the user to cast to something we know about. e.g. unreflected enum to int.
            /// Specializations are below for handling pointers and arrays.
            typedef PleaseCastArgumentToAKnownType Type;
        };

        /// Fallback specialization for pointers.
        template<typename T>
        struct FallbackType< T* > { typedef const void* Type; };

        /// Return the type itself if reflected or a fallback type if not.
        template<typename T, bool IsReflected>
        struct ReflectedOrFallback { typedef T Type; };
        template<typename T>
        struct ReflectedOrFallback<T, false> { typedef typename FallbackType< T >::Type Type; };

        /// Find a hkReflect::Type for an argument. Use normal reflection for reflected things.
        /// Else a closely related type (e.g. const void* for non reflected pointers, int for enum etc)
        template<typename T>
        struct VarArgsType
        {
            typedef typename hkTrait::RemoveConstRef< typename hkTrait::ArrayToPointer<T>::Type >::Type SimpleT;
            typedef typename ReflectedOrFallback< SimpleT, hkReflect::IsReflected<SimpleT>::result >::Type Type;
        };

        /// Specialization for char[], as we don't want to get ReflectOf<char*>, which would get us a
        /// hkReflect::Detail::StaticStringImpl instead of a hkReflect::Detail::CharBufferImpl, which
        /// have different interpretation of their 'addr' argument (the former is const char**, the
        /// later is directly const char*). This is consistent with the fact that addrOf(const char*)
        /// returns const char**, while addrOf(const char[]) returns const char*. The former is the
        /// reflection of a pointer variable, the later of a "value" variable (which happens to be an array).
        template<hkUlong N>
        struct VarArgsType<char [N]>
        {
            typedef char Type[N];
        };
        template<hkUlong N>
        struct VarArgsType<const char [N]>
        {
            typedef char Type[N];
        };
        template<hkUlong N>
        struct VarArgsType<char (&)[N]>
        {
            typedef char Type[N];
        };
        template<hkUlong N>
        struct VarArgsType<const char (&)[N]>
        {
            typedef char Type[N];
        };

            /// Fallback specialization for arrays.
        template<typename T, int N>
        struct FallbackType< T[N] > { typedef typename VarArgsType<T*>::Type Type; };

        /// Array of types implicitly associated with a list of arguments.
        /// Provides local storage for N argument's types.
        /// This is essentially a hkReflect::Detail::FixedArrayStorage, without the dependency to hkArrayView.
        template <hkUlong N>
        struct VarArgsTypeArray
        {
            hkUlong m_numTypes;
            const hkReflect::Type* m_types[N];
        };

        /// Specialization for empty argument lists, used as a "base class" of above
        /// (see hkVarArgs::VaTypes).
        template<>
        struct HK_EXPORT_COMMON VarArgsTypeArray<0>
        {
            hkUlong m_numTypes;
        };

#if defined(HK_HAS_VARIADIC_TEMPLATE)
        /// Get a list of types for a variadic list of arguments.
        /// This builds a static-storage VarArgsTypesArray for the arguments.
        template<typename... Args>
        struct VarArgsTypesOf; // not defined

        template<typename... Args>
        struct VarArgsTypesOf<void, Args...>
        {
            static const VarArgsTypeArray<sizeof...(Args)> typesArray;
        };

        /// Array definition.
        template<typename... Args>
        const VarArgsTypeArray<sizeof...(Args)> VarArgsTypesOf<void, Args...>::typesArray = {
            sizeof...(Args),
            { reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<Args>::Holder::typeData )... }
        };
#else
        // Fallback for compilers without variadic templates.
        // Essentially the same thing as above, manually unrolled.

        // Generated from //depot/Other/Personal/jerome.humbert/varargs/generateVarArgs.py

        template <typename T0, typename T1 = void, typename T2 = void, typename T3 = void, typename T4 = void, typename T5 = void, typename T6 = void, typename T7 = void, typename T8 = void, typename T9 = void, typename T10 = void>
        struct VarArgsTypesOf; // not defined

        template <typename T1>
        struct VarArgsTypesOf<void, T1> {
            static const VarArgsTypeArray<1> typesArray;
        };
        template <typename T1, typename T2>
        struct VarArgsTypesOf<void, T1, T2> {
            static const VarArgsTypeArray<2> typesArray;
        };
        template <typename T1, typename T2, typename T3>
        struct VarArgsTypesOf<void, T1, T2, T3> {
            static const VarArgsTypeArray<3> typesArray;
        };
        template <typename T1, typename T2, typename T3, typename T4>
        struct VarArgsTypesOf<void, T1, T2, T3, T4> {
            static const VarArgsTypeArray<4> typesArray;
        };
        template <typename T1, typename T2, typename T3, typename T4, typename T5>
        struct VarArgsTypesOf<void, T1, T2, T3, T4, T5> {
            static const VarArgsTypeArray<5> typesArray;
        };
        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6>
        struct VarArgsTypesOf<void, T1, T2, T3, T4, T5, T6> {
            static const VarArgsTypeArray<6> typesArray;
        };
        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6, typename T7>
        struct VarArgsTypesOf<void, T1, T2, T3, T4, T5, T6, T7> {
            static const VarArgsTypeArray<7> typesArray;
        };
        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6, typename T7, typename T8>
        struct VarArgsTypesOf<void, T1, T2, T3, T4, T5, T6, T7, T8> {
            static const VarArgsTypeArray<8> typesArray;
        };
        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6, typename T7, typename T8, typename T9>
        struct VarArgsTypesOf<void, T1, T2, T3, T4, T5, T6, T7, T8, T9> {
            static const VarArgsTypeArray<9> typesArray;
        };
        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6, typename T7, typename T8, typename T9, typename T10>
        struct VarArgsTypesOf<void, T1, T2, T3, T4, T5, T6, T7, T8, T9, T10> {
            static const VarArgsTypeArray<10> typesArray;
        };

        template <typename T1>
        const VarArgsTypeArray<1> VarArgsTypesOf<void, T1>::typesArray = {
            1,
            {
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T1>::Holder::typeData )
            }
        };
        template <typename T1, typename T2>
        const VarArgsTypeArray<2> VarArgsTypesOf<void, T1, T2>::typesArray = {
            2,
            {
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T1>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T2>::Holder::typeData )
            }
        };
        template <typename T1, typename T2, typename T3>
        const VarArgsTypeArray<3> VarArgsTypesOf<void, T1, T2, T3>::typesArray = {
            3,
            {
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T1>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T2>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T3>::Holder::typeData )
            }
        };
        template <typename T1, typename T2, typename T3, typename T4>
        const VarArgsTypeArray<4> VarArgsTypesOf<void, T1, T2, T3, T4>::typesArray = {
            4,
            {
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T1>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T2>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T3>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T4>::Holder::typeData )
            }
        };
        template <typename T1, typename T2, typename T3, typename T4, typename T5>
        const VarArgsTypeArray<5> VarArgsTypesOf<void, T1, T2, T3, T4, T5>::typesArray = {
            5,
            {
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T1>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T2>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T3>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T4>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T5>::Holder::typeData )
            }
        };
        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6>
        const VarArgsTypeArray<6> VarArgsTypesOf<void, T1, T2, T3, T4, T5, T6>::typesArray = {
            6,
            {
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T1>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T2>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T3>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T4>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T5>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T6>::Holder::typeData )
            }
        };
        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6, typename T7>
        const VarArgsTypeArray<7> VarArgsTypesOf<void, T1, T2, T3, T4, T5, T6, T7>::typesArray = {
            7,
            {
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T1>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T2>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T3>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T4>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T5>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T6>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T7>::Holder::typeData )
            }
        };
        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6, typename T7, typename T8>
        const VarArgsTypeArray<8> VarArgsTypesOf<void, T1, T2, T3, T4, T5, T6, T7, T8>::typesArray = {
            8,
            {
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T1>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T2>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T3>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T4>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T5>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T6>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T7>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T8>::Holder::typeData )
            }
        };
        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6, typename T7, typename T8, typename T9>
        const VarArgsTypeArray<9> VarArgsTypesOf<void, T1, T2, T3, T4, T5, T6, T7, T8, T9>::typesArray = {
            9,
            {
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T1>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T2>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T3>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T4>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T5>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T6>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T7>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T8>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T9>::Holder::typeData )
            }
        };
        template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6, typename T7, typename T8, typename T9, typename T10>
        const VarArgsTypeArray<10> VarArgsTypesOf<void, T1, T2, T3, T4, T5, T6, T7, T8, T9, T10>::typesArray = {
            10,
            {
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T1>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T2>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T3>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T4>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T5>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T6>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T7>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T8>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T9>::Holder::typeData ),
                reinterpret_cast<const hkReflect::Type*>( &hkReflect::ReflectionOf<T10>::Holder::typeData )
            }
        };
#endif

        /// Specialization for an empty argument list.
        /// Since HK_VARARGS1() adds a void at the beginning of the argument's types,
        /// an empty argument list yields VarArgsTypesOf<void>.
        template<>
        struct HK_EXPORT_COMMON VarArgsTypesOf<void>
        {
            static const VarArgsTypeArray<0> typesArray;
        };
    }

    /// Collection of vararg types implicitly associated with a va_list of a C-style variadic function.
    /// Implemented as a pointer to a fixed-size array of hkReflect::Type, one
    /// per argument of the va_list. See VarArgsTypeArray.
    typedef const Detail::VarArgsTypeArray<0>* VaTypes;
}

#if defined(HK_COMPILER_SNC) || defined(HK_COMPILER_GHS) || defined(HK_PLATFORM_ANDROID) || defined(HK_PLATFORM_LINUX) || defined(HK_COMPILER_CLANG)
    #define HK_VARARGS_TYPEOF(I,D,X) ,typename hkVarArgs::Detail::VarArgsType< __typeof__(X) >::Type
#else
    #define HK_VARARGS_TYPEOF(I,D,X) ,typename hkVarArgs::Detail::VarArgsType< decltype(X) >::Type
#endif

#define HK_VARARGS_ADDROF(I,D,X) hkVarArgs::Detail::addrOf(X),


/// Macro to create a typesafe variadic parameter pack for use with "list" style varargs.
/// This expands to the first fixed argument (usually a format string), followed by
/// an address to the types of the parameters, then the address of each parameter.
/// Note that the argument list following the VaTypes array has N+1 arguments, the last one
/// being useless (0) and only there to prevent a compiler error if there's no argument
/// at all in the vararg list. This is harmless since we won't read it.
#if defined(HK_COMPILER_MSVC)
    // MSVC automatically elide trailing comma when a macro vararg list (__VA_ARGS__) is empty.
    // However its macro expansion rules with respect to variadic macros are different from
    // most other compilers.
    #define HK_VARARGS1(ARG0, ...) \
        ARG0, \
        reinterpret_cast<hkVarArgs::VaTypes>( \
            &hkVarArgs::Detail::VarArgsTypesOf< void HK_PP_FOR_EACH(HK_VARARGS_TYPEOF, _, __VA_ARGS__) >::typesArray \
        ), \
        HK_PP_FOR_EACH(HK_VARARGS_ADDROF, _, __VA_ARGS__) 0
#else
    // Non-MSVC compilers do not elide the trailing comma automatically when the varargs list is empty.
    // This can be forced using the named varargs ("args...") syntax.
    #define HK_VARARGS1(ARG0, args...) \
        ARG0, \
        reinterpret_cast<hkVarArgs::VaTypes>( \
            &hkVarArgs::Detail::VarArgsTypesOf< void HK_PP_FOR_EACH( HK_VARARGS_TYPEOF, _, ##args ) >::typesArray \
        ), \
        HK_PP_FOR_EACH( HK_VARARGS_ADDROF, _, ##args ) 0
#endif

// Use this when passing __VA_ARGS__ to HK_VARARGS1 from an outer macro.
// Works around non-standard MSVC preprocessor. Note the double parenthesis.
//   #define MY_MACRO(...) HK_VARARGS1_WRAP((__VA_ARGS__))
#define HK_VARARGS1_WRAP(tuple) HK_VARARGS1 tuple

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
