// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Types/hkSignalSlots.h>
#include <Common/Base/System/hkBaseSystem.h>

#define DEBUG_LOG_IDENTIFIER "base.signal"
#include <Common/Base/System/Log/hkLog.hxx>

#ifdef HK_ENABLE_SIGNAL_TRACKING

namespace hkSignalInternals
{
    struct  SpinLock
    {
        struct Scope { Scope(SpinLock& l) : m_lock(&l) { l.enter(); } ~Scope() { m_lock->leave(); } SpinLock* m_lock; };

        HK_INLINE                   SpinLock() : m_value(0) {}
        HK_INLINE                   ~SpinLock()             { HK_ASSERT(0xADAB24F0,m_value==0,"Unreleased lock"); }
        HK_INLINE void          enter()                 { for(;;) { if(!atomicAdd(&m_value,1)) break; else atomicAdd(&m_value,-1),spin(); }}
        HK_INLINE void          leave()                 { atomicAdd(&m_value,-1); }
        HK_INLINE void          spin() const            { do {} while(*(volatile hkUint32*)&m_value); }
        HK_INLINE static hkUint32   atomicAdd(_Inout_ volatile hkUint32* value, int increment)
        {
            return hkAtomic::exchangeAdd32((hkUint32*)value,increment);
        }
    private:
        hkUint32    m_value;
    };
}

using namespace hkSignalInternals;

//
hkSignal*       hkSignal::s_root;
static SpinLock hkSignal_lock;

//
void hkSignal::beginTrack(_Inout_ hkSignal* signal)
{
    SpinLock::Scope lock(hkSignal_lock);

    // Add this signal to the beginning of the double-linked list rooted at s_root.
    if ( s_root != HK_NULL ) s_root->m_links[0] = signal;
    signal->m_links[0]  =   HK_NULL;
    signal->m_links[1]  =   hkSignal::s_root;
    s_root              =   signal;
}

//
void hkSignal::endTrack(_Inout_ hkSignal* signal)
{
    SpinLock::Scope lock(hkSignal_lock);

    // Remove this signal from the double-linked list.
    if(signal->m_links[1]) signal->m_links[1]->m_links[0] = signal->m_links[0];
    if(signal->m_links[0]) signal->m_links[0]->m_links[1] = signal->m_links[1]; else hkSignal::s_root = signal->m_links[0];
}

#endif

//
int hkSignal::getNumSubscriptions() const
{
    int n=0;
    for(hkSlot* s = getSlots(); s; s=s->getNext())
    {
        n += s->hasNoSubscription() ? 0 : 1;
    }
    return n;
}

//
void hkSignal::unsubscribeAll(_Inout_ void* object)
{
    SlotList* prev = &m_slots;
    for(hkSlot* slot = getSlots(), *next = HK_NULL; slot; slot = next)
    {
        next = slot->getNext();
        if(slot->m_object == object)
        {
            if (m_slots.getInt())
            {
                // the slot list is in use, delay the deletion
                slot->unsubscribe();
                prev = &slot->m_next;
            }
            else
            {
                // delete the slot directly
                prev->setPtr(next);
                delete slot;
            }
        }
        else
        {
            prev = &slot->m_next;
        }
    }
}

//
hkBool hkSignal::unsubscribeInternal(_Inout_ void* object, _In_ const void* method, int length)
{
    SlotList* prev = &m_slots;
    for(hkSlot* slot = getSlots(); slot; slot = slot->getNext())
    {
        if(slot->m_object == object && slot->matchMethod(method, length))
        {
            if (m_slots.getInt())
            {
                // the slot list is in use, delay the deletion
                slot->unsubscribe();
            }
            else
            {
                // delete the slot directly
                prev->setPtr(slot->getNext());
                delete slot;
            }
            return true;
        }
        prev = &slot->m_next;
    }
    return false;
}

hkBool hkSignal::unsubscribe(ConnectionId id)
{
    return hkSignal::unsubscribeInternal(id, HK_NULL, 666);
}

//
_Ret_maybenull_
hkSlot* hkSignal::find(_Inout_ void* object, _In_ const void* method, int length)
{
    for(hkSlot* slot = getSlots(); slot; slot = slot->getNext())
    {
        if(slot->m_object == object && !slot->hasNoSubscription() && slot->matchMethod(method, length))
        {
            return slot;
        }
    }
    return HK_NULL;
}

//
void hkSignal::destroy()
{
    if(hkBaseSystem::isInitialized())
    {
        #ifdef HK_SIGNAL_DEBUG
        if(getNumSubscriptions())
        {
            HK_WARN_ALWAYS(0x2AC66B60, "" << getNumSubscriptions() << " slot(s) still connected on '" << getName() << "' destruction");
            printReport();
        }
        #endif

#if defined(HK_DEBUG)
        if (getSlots() && (m_slots.getInt() != 0))
        {
            //consider protecting the object holding the signal using HKT_PROTECT_THIS_FROM_DELETION before the fire
            HK_WARN_ALWAYS( 0x2BD66B60, "Destroying a signal while it's being fired - this can lead to problems." );
        }
#endif

        reset();
    }
}

//
void hkSignal::reset()
{
    if(getSlots())
    {
        if (m_slots.getInt() == 0)
        {
            // Clear the slot list.
            for (hkSlot *slot = getSlots(), *next = slot->getNext(); slot;)
            {
                delete slot;
                slot = next;
                next = next ? next->getNext() : HK_NULL;
            }
            m_slots.set(HK_NULL, 0);
        }
        else
        {
            // Just mark the slots as unsubscribed.
            for (hkSlot *slot = getSlots(); slot; slot = slot->getNext())
            {
                slot->unsubscribe();
            }
        }
    }
}

//
void hkSignal::printReport() const
{
    Log_Info( "<{}>({})", getName(), this );
    for ( const hkSlot* slot = getSlots(); slot; slot = slot->getNext() )
    {
        if ( !slot->hasNoSubscription() )
        {
            Log_Info( "\t[{}]({})", slot->getName(), slot );
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
