// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

template<typename TYPE>
template<typename U>
hkViewPtr<TYPE>::hkViewPtr(const hkRefPtr<U>& p) : m_ptr(p) {}

// ---------------------------------------------------------------------------------------------------------------------
// Static utilities
// ---------------------------------------------------------------------------------------------------------------------

/// Template function used to make the cast result type dependent so that we dont
/// need a definition of hkReferencedObject here.
template<typename TYPE>
HK_ALWAYS_INLINE _Ret_notnull_ const hkReferencedObject* hkPtrTypes::castToRefObj(_In_ const TYPE* ptr)
{
    return reinterpret_cast<const hkReferencedObject*>(ptr);
}

/// Add a reference to the pointed entity.
/// hkRefPtr will count references to the pointed object.
/// This function is used internally to add references to the pointed object.
template<typename TYPE>
void hkPtrTypes::s_addReferenceToPointed(const hkRefPtr<TYPE>& ptr)
{
    const TYPE* pointed = ptr.val();

    if(pointed)
    {
        // The pointer is cast to a hkReferencedObject* in order to perform this operation without
        // requiring the full definition of TYPE, hkRefPtr must only be used with hkReferencedObjects.
        castToRefObj(pointed)->addReference();
    }
}

/// Remove a reference from the pointed object.
/// See the documentation for s_addReferenceToPointed().
template<typename TYPE>
void hkPtrTypes::s_removeReferenceFromPointed(const hkRefPtr<TYPE>& ptr)
{
    const TYPE* pointed = ptr.val();

    if(pointed)
    {
        // The pointer is cast to a hkReferencedObject* in order to perform this operation without
        // requiring the full definition of TYPE, hkRefPtr must only be used with hkReferencedObjects.
        castToRefObj(pointed)->removeReference();
    }
}


namespace hkRefPtrReferencedObjectCheck
{
    hkBool::CompileTimeTrueType check(_In_ const hkReferencedObject*);
    hkBool::CompileTimeFalseType check(...);
}

// This check does not produce any code and will check that the type can be casted to
// hkReferencedObject implicitly.
#define HK_REFPTR_CHECK_REFERENCED_OBJECT() \
    static_assert( sizeof(hkRefPtrReferencedObjectCheck::check(reinterpret_cast<TYPE*>(0))) == sizeof(hkBool::CompileTimeTrueType), "hkRefPtr can only be used with hkReferencedObject-derived types");


// hkRefNew<> ----------------------------------------------------------------------------------------------------------

template <typename TYPE>
inline hkRefNew<TYPE>::hkRefNew(_In_opt_ TYPE* t)
    : m_ptr(t)
{
    HK_REFPTR_CHECK_REFERENCED_OBJECT();
}

template <typename TYPE>
inline hkRefNew<TYPE>::hkRefNew(const hkRefNew& ptr)
    : m_ptr(ptr.stealOwnership())
{
}

template <typename TYPE>
inline hkRefNew<TYPE>::~hkRefNew()
{
    HK_ASSERT_NO_MSG(0x35bfb497, m_ptr == HK_NULL);
}

template <typename TYPE>
inline _Ret_maybenull_ TYPE* hkRefNew<TYPE>::stealOwnership() const
{
    TYPE* t = m_ptr;
    m_ptr = HK_NULL;
    return t;
}

template <typename TYPE>
inline _Ret_maybenull_ TYPE* hkRefNew<TYPE>::val() const
{
    return m_ptr;
}

template <typename TYPE>
inline _Ret_maybenull_ TYPE* hkRefNew<TYPE>::operator->() const
{
    return m_ptr;
}

template <typename TYPE>
inline hkRefNew<TYPE>::operator hkBoolOperator::BoolType() const
{
    return hkBoolOperator::cast(m_ptr);
}

// hkRefLoan<> ---------------------------------------------------------------------------------------------------------

template <typename TYPE>
inline hkRefLoan<TYPE>::hkRefLoan(_In_ TYPE* t)
    : m_ptr(t)
{
    HK_REFPTR_CHECK_REFERENCED_OBJECT();
}

template <typename TYPE>
inline hkRefLoan<TYPE>::operator TYPE*() const
{
    return m_ptr;
}

// hkRefPtr<> ----------------------------------------------------------------------------------------------------------

template<typename TYPE>
template<typename RTYPE>
HK_ALWAYS_INLINE
void hkRefPtr<TYPE>::moveAssign(hkRefPtr<RTYPE>&& rp)
{
    // Temporary variable handles the self-move case
    auto tmp = rp.m_ptr;
    rp.m_ptr = nullptr;
    hkPtrTypes::s_removeReferenceFromPointed(*this);
    m_ptr = tmp;
}

template<typename TYPE>
inline hkRefPtr<TYPE>::hkRefPtr()
    : m_ptr(HK_NULL)
{}

template<typename TYPE>
inline hkRefPtr<TYPE>::hkRefPtr(nullptr_t)
    : m_ptr(nullptr)
{}

template<typename TYPE>
template<typename RTYPE>
inline hkRefPtr<TYPE>::hkRefPtr(const hkRefPtr<RTYPE>& rp)
    : m_ptr(rp.m_ptr)
{
    HK_REFPTR_CHECK_REFERENCED_OBJECT();
    hkPtrTypes::s_addReferenceToPointed(*this);
}

template<typename TYPE>
inline hkRefPtr<TYPE>::hkRefPtr(const hkRefPtr<TYPE>& rp)
    : m_ptr(rp.m_ptr)
{
    hkPtrTypes::s_addReferenceToPointed(*this);
}

template<typename TYPE>
template<typename RTYPE>
HK_ALWAYS_INLINE hkRefPtr<TYPE>::hkRefPtr(hkRefPtr<RTYPE>&& rp)
    : m_ptr(rp.m_ptr)
{
    rp.m_ptr = nullptr;
}

template<typename TYPE>
HK_ALWAYS_INLINE hkRefPtr<TYPE>::hkRefPtr(hkRefPtr<TYPE>&& rp)
    : m_ptr(rp.m_ptr)
{
    rp.m_ptr = nullptr;
}

template<typename TYPE>
inline hkRefPtr<TYPE>::hkRefPtr(hkRefLoan<TYPE> rp)
    : m_ptr(rp)
{
    hkPtrTypes::s_addReferenceToPointed(*this);
}

template<typename TYPE>
template<typename RTYPE>
inline hkRefPtr<TYPE>::hkRefPtr(hkRefNew<RTYPE> rp)
    : m_ptr(rp.stealOwnership())
{
    HK_REFPTR_CHECK_REFERENCED_OBJECT();
}

template<class TYPE>
template<typename RTYPE>
inline hkRefPtr<TYPE>::hkRefPtr(const hkAutoRemoveReferenceWrapper<RTYPE>& rp)
    : m_ptr(static_cast<TYPE*>(rp))
{
    HK_REFPTR_CHECK_REFERENCED_OBJECT();
    hkPtrTypes::s_addReferenceToPointed(*this);
}

template<typename TYPE>
inline hkRefPtr<TYPE>::hkRefPtr(_In_ TYPE* e)
    : m_ptr(e)
{
    HK_REFPTR_CHECK_REFERENCED_OBJECT();
    hkPtrTypes::s_addReferenceToPointed(*this);
}

template <typename TYPE>
template <typename RTYPE>
inline hkRefPtr<TYPE>::hkRefPtr(const hkViewPtr<RTYPE>& rp)
    : m_ptr(rp)
{
    HK_REFPTR_CHECK_REFERENCED_OBJECT();
    hkPtrTypes::s_addReferenceToPointed(*this);
}

template<typename TYPE>
inline hkRefPtr<TYPE>::~hkRefPtr()
{
    hkPtrTypes::s_removeReferenceFromPointed(*this);
}

template<typename TYPE>
template<typename RTYPE>
HK_ALWAYS_INLINE hkRefPtr<TYPE>& hkRefPtr<TYPE>::operator=(const hkRefPtr<RTYPE>& rp)
{
    HK_REFPTR_CHECK_REFERENCED_OBJECT();
    // add reference first to allow self-assignment
    hkPtrTypes::s_addReferenceToPointed(rp);
    // remove reference from self
    hkPtrTypes::s_removeReferenceFromPointed(*this);
    m_ptr = rp.m_ptr;
    return *this;
}

template<typename TYPE>
HK_ALWAYS_INLINE hkRefPtr<TYPE>& hkRefPtr<TYPE>::operator=(const hkRefPtr<TYPE>& rp)
{
    // add reference first to allow self-assignment
    hkPtrTypes::s_addReferenceToPointed(rp);
    // remove reference from self
    hkPtrTypes::s_removeReferenceFromPointed(*this);
    m_ptr = rp.m_ptr;
    return *this;
}

template<typename TYPE>
template<typename RTYPE>
HK_ALWAYS_INLINE hkRefPtr<TYPE>& hkRefPtr<TYPE>::operator=(hkRefPtr<RTYPE>&& rp)
{
    HK_REFPTR_CHECK_REFERENCED_OBJECT();
    moveAssign(hk::move(rp));
    return *this;
}

template<typename TYPE>
HK_ALWAYS_INLINE hkRefPtr<TYPE>& hkRefPtr<TYPE>::operator=(hkRefPtr<TYPE>&& rp)
{
    moveAssign(hk::move(rp));
    return *this;
}

template<typename TYPE>
inline void hkRefPtr<TYPE>::operator=(const hkRefLoan<TYPE>& e)
{
    operator=( static_cast<TYPE*>(e) );
}

template<typename TYPE>
template<typename RTYPE>
inline void hkRefPtr<TYPE>::operator=(const hkRefNew<RTYPE>& e)
{
    HK_REFPTR_CHECK_REFERENCED_OBJECT();
    // remove reference from self
    hkPtrTypes::s_removeReferenceFromPointed(*this);
    m_ptr = e.stealOwnership();
}

template<typename TYPE>
template<typename RTYPE>
inline void hkRefPtr<TYPE>::operator=(const hkAutoRemoveReferenceWrapper<RTYPE>& e)
{
    operator=( static_cast<RTYPE*>(e) );
}


template<typename TYPE>
inline void hkRefPtr<TYPE>::operator=(_In_opt_ TYPE* e)
{
    HK_REFPTR_CHECK_REFERENCED_OBJECT();
    // add reference first to allow self-assignment
    hkPtrTypes::s_addReferenceToPointed(*reinterpret_cast<const hkRefPtr<TYPE>*>(&e));
    // remove reference from self
    hkPtrTypes::s_removeReferenceFromPointed(*this);
    m_ptr = e;
}

template <typename TYPE>
template <typename RTYPE>
void hkRefPtr<TYPE>::operator=(const hkViewPtr<RTYPE>& rp)
{
    operator=(rp.val());
}

template<typename TYPE>
inline void hkRefPtr<TYPE>::setAndDontIncrementRefCount(_In_opt_ TYPE* e )
{
    HK_REFPTR_CHECK_REFERENCED_OBJECT();
    if (e != val())
    {
        // remove reference from self
        hkPtrTypes::s_removeReferenceFromPointed(*this);
        m_ptr = e;
    }
}

template<class TYPE>
inline void hkRefPtr<TYPE>::unsetAndDontDecrementRefCount()
{
    m_ptr = HK_NULL;
}

template<class TYPE>
HK_ALWAYS_INLINE void hkRefPtr<TYPE>::swap(hkRefPtr& rp)
{
    
    TYPE* tmp = m_ptr;
    m_ptr = rp.m_ptr;
    rp.m_ptr = tmp;
}

template <typename TYPE>
_Ret_maybenull_ TYPE* hkRefPtr<TYPE>::val() const
{
    return m_ptr;
}

template<class TYPE>
HK_ALWAYS_INLINE _Ret_notnull_ TYPE* hkRefPtr<TYPE>::operator->() const
{
    return m_ptr;
}

template<class TYPE>
inline _Ret_notnull_ hkRefPtr<TYPE>::operator TYPE*() const
{
    return val();
}

template <typename TYPE>
HK_ALWAYS_INLINE void swap(hkRefPtr<TYPE>& a, hkRefPtr<TYPE>& b)
{
    a.swap(b);
}


template <typename TYPE>
HK_ALWAYS_INLINE hkRefPtr<TYPE> hk::makeRef()
{
    return hkRefPtr<TYPE>(hkRefNew<TYPE>(new TYPE));
}

template <typename TYPE, typename T1>
HK_ALWAYS_INLINE hkRefPtr<TYPE> hk::makeRef(T1&& t1)
{
    return hkRefPtr<TYPE>(hkRefNew<TYPE>(new TYPE(hk::forward<T1>(t1))));
}

template <typename TYPE, typename T1, typename T2>
HK_ALWAYS_INLINE hkRefPtr<TYPE> hk::makeRef(T1&& t1, T2&& t2)
{
    return hkRefPtr<TYPE>(hkRefNew<TYPE>(new TYPE(hk::forward<T1>(t1), hk::forward<T2>(t2))));
}

template <typename TYPE, typename T1, typename T2, typename T3>
HK_ALWAYS_INLINE hkRefPtr<TYPE> hk::makeRef(T1&& t1, T2&& t2, T3&& t3)
{
    return hkRefPtr<TYPE>(hkRefNew<TYPE>(new TYPE(hk::forward<T1>(t1), hk::forward<T2>(t2), hk::forward<T3>(t3))));
}

template <typename TYPE, typename T1, typename T2, typename T3, typename T4>
HK_ALWAYS_INLINE hkRefPtr<TYPE> hk::makeRef(T1&& t1, T2&& t2, T3&& t3, T4&& t4)
{
    return hkRefPtr<TYPE>(hkRefNew<TYPE>(new TYPE(hk::forward<T1>(t1), hk::forward<T2>(t2), hk::forward<T3>(t3), hk::forward<T4>(t4))));
}

template <typename TYPE, typename T1, typename T2, typename T3, typename T4, typename T5>
HK_ALWAYS_INLINE hkRefPtr<TYPE> hk::makeRef(T1&& t1, T2&& t2, T3&& t3, T4&& t4, T5&& t5)
{
    return hkRefPtr<TYPE>(hkRefNew<TYPE>(new TYPE(hk::forward<T1>(t1), hk::forward<T2>(t2), hk::forward<T3>(t3), hk::forward<T4>(t4), hk::forward<T5>(t5))));
}

namespace hkDetail
{
    struct IndirectPtr
    {
        template<typename T>
        IndirectPtr(const hkRefPtr<T>& p)
            : m_addrAndType(hkReflect::exactObj(p.val()))
        {
        }

        template<typename T>
        IndirectPtr(const hkViewPtr<T>& p)
            : m_addrAndType(hkReflect::exactObj(p.val()))
        {
        }

        hkReflect::Detail::AddrAndType m_addrAndType;
    };
}

HK_ALWAYS_INLINE hkReflect::Detail::AutoCast hkDynCast(const hkDetail::IndirectPtr& from)
{
    return hkReflect::Detail::AutoCast(from.m_addrAndType.m_addr, from.m_addrAndType.m_type);
}

HK_ALWAYS_INLINE _Ret_maybenull_ void* hkDynCast(const hkDetail::IndirectPtr& from, _In_ const hkReflect::Type* toType)
{
    return hkReflect::upCast(from.m_addrAndType.m_addr, from.m_addrAndType.m_type, toType);
}

template<typename T, typename DEDUCED>
HK_ALWAYS_INLINE _Ret_maybenull_ T* hkPtrConstCast(const hkRefPtr<DEDUCED>& ptr)
{
    return const_cast<T*>(ptr.val());
}

template<typename T, typename DEDUCED>
HK_ALWAYS_INLINE _Ret_maybenull_ T* hkPtrStaticCast(const hkRefPtr<DEDUCED>& ptr)
{
    return static_cast<T*>(ptr.val());
}

template<typename T, typename DEDUCED>
HK_ALWAYS_INLINE _Ret_maybenull_ T* hkPtrReinterpretCast(const hkRefPtr<DEDUCED>& ptr)
{
    return reinterpret_cast<T*>(ptr.val());
}

template<typename T, typename DEDUCED>
HK_ALWAYS_INLINE  _Ret_maybenull_ T* hkPtrConstCast(const hkViewPtr<DEDUCED>& ptr)
{
    return const_cast<T*>(ptr.val());
}

template<typename T, typename DEDUCED>
HK_ALWAYS_INLINE _Ret_maybenull_ T* hkPtrStaticCast(const hkViewPtr<DEDUCED>& ptr)
{
    return static_cast<T*>(ptr.val());
}

template<typename T, typename DEDUCED>
HK_ALWAYS_INLINE _Ret_maybenull_ T* hkPtrReinterpretCast(const hkViewPtr<DEDUCED>& ptr)
{
    return reinterpret_cast<T*>(ptr.val());
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
