// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

template <class TYPE>
class hkRefPtr;

/// Wrapper for a non-owned pointer.
/// Implicitly marks the wrapped pointer as non-owned, i.e. setting it during deserialization will not
/// increment the reference count of any pointed hkReferencedObject.
template<typename TYPE>
class hkViewPtr
{
    public:
        HK_DECLARE_CLASS(hkViewPtr, NewExplicit, TrackAs(void*), Reflect);
        HK_REFLECT_AS_POINTER(&hkReflect::Detail::ViewPointerImpl::s_instance, TYPE);
        HK_RECORD_ATTR( hk::AddFlags(hkReflect::Type::TYPE_POINTER_WEAK) );
    typedef hkReflect::Detail::PointerPropertyImpl<hkViewPtr<TYPE>, TYPE> PropertyImpl;

        HK_ALWAYS_INLINE hkViewPtr() : m_ptr(HK_NULL) {}
        HK_ALWAYS_INLINE hkViewPtr(nullptr_t) : m_ptr(nullptr) {}
        HK_ALWAYS_INLINE hkViewPtr(_In_opt_ TYPE* t) : m_ptr(t) {}
        template<typename U>    HK_ALWAYS_INLINE hkViewPtr(_In_opt_ U* t) : m_ptr(t) {}
        template<typename U>    HK_ALWAYS_INLINE hkViewPtr(hkViewPtr<U> t) : m_ptr(t.val()) {}
        template<typename U>    HK_ALWAYS_INLINE hkViewPtr(const hkRefPtr<U>& t);

        template<typename U>
        HK_ALWAYS_INLINE void operator=(_In_opt_ U* u) { m_ptr = u; }
        HK_ALWAYS_INLINE void operator=(const hkViewPtr& v) { m_ptr = v.m_ptr; }
        HK_ALWAYS_INLINE void operator=(nullptr_t) { m_ptr = nullptr; }

        HK_ALWAYS_INLINE _Ret_maybenull_ operator TYPE*() const { return m_ptr; }   ///< resolve
        HK_ALWAYS_INLINE operator hkReflect::Var() const { return hkReflect::Var(m_ptr); }
        HK_ALWAYS_INLINE _Ret_notnull_ TYPE* val() const        { return m_ptr; }   ///< resolve
        HK_ALWAYS_INLINE _Ret_notnull_ TYPE* operator->() const { return m_ptr; }   ///< resolve

    private:
        TYPE* m_ptr;
};

namespace hkPtrTypes
{
    template<typename TYPE>
    inline _Ret_notnull_ const hkReferencedObject* castToRefObj(_In_ const TYPE* ptr);

    template<typename TYPE>
    inline void s_addReferenceToPointed(const hkRefPtr<TYPE>& ptr);
    template<typename TYPE>
    inline void s_removeReferenceFromPointed(const hkRefPtr<TYPE>& ptr);
}

    /// Intermediate value for pointer return values. To avoid a resource leak,
    /// it needs to have a reference removed when the object is no longer needed.
    /// Usually this type is used as an intermediate return value for functions and will
    /// be assigned directly to a hkRefPtr.
    /// Note that this indicates a newly added reference, not necessarily a new object, i.e.
    /// the returned object may be shared.
template <typename TYPE>
class hkRefNew
{
public:

    /// Implicit conversion from raw pointer.
    inline hkRefNew(_In_opt_ TYPE* t);
    /// Copy constructors, has move semantics: will steal the pointer.
    inline hkRefNew(const hkRefNew& ptr);
    /// destructor, will check that the pointer has been stolen.
    inline ~hkRefNew();
    /// Assume ownership of this reference.
    /// Assumes responsibility for calling removeReference in the future.
    inline _Ret_maybenull_ TYPE* stealOwnership() const;

    /// returns the underlying pointer
    inline _Ret_maybenull_ TYPE* val() const;

    inline _Ret_maybenull_ TYPE* operator->() const;

    // safe conversion to bool
    inline operator hkBoolOperator::BoolType() const;

private:

    hkRefNew(); ///< Not implemented
    void operator=(const hkRefNew& b); ///< Not implemented
    hkRefNew(const hkRefPtr<TYPE>& b); ///< Not implemented
    void operator=(const hkRefPtr<TYPE>& b); ///< Not implemented

    // mutable because hkRefNew has move semantics,
    // the copy constructor actually mutate it's parameter
    mutable TYPE* m_ptr;
};

    /// The contained object is owned by another object. Usually used as an intermediate
    /// value for function return types, it indicates that a reference has not been added to
    /// the returned object. It is still owned and managed by somebody else. If the owning object
    /// is modified, the reference may become invalid (dangling pointer).
    /// Assign to a hkRefPtr to prevent the reference being removed prematurely.
template <typename TYPE>
class hkRefLoan
{
public:

    /// Implicit conversion from raw pointer.
    inline hkRefLoan(_In_ TYPE* t);
    /// Implicit conversion to raw pointer.
    inline operator TYPE*() const;

private:

    hkRefLoan(); ///< Not implemented
    void operator=(const hkRefLoan& b); ///< Not implemented
    TYPE* m_ptr;
};

// forward declaration
template<typename TYPE>
struct hkAutoRemoveReferenceWrapper;

    /// A simple helper class to automatically add and remove references to objects.
template <class TYPE>
class hkRefPtr
{
public:

    typedef hkRefPtr<TYPE> ThisType;
    HK_DECLARE_CLASS(hkRefPtr, NewExplicit, TrackAs(void*), Reflect);
    HK_REFLECT_AS_POINTER(&hkReflect::Detail::ReferencedObjectPointerImpl::s_instance, TYPE);
    typedef hkReflect::Detail::PointerPropertyImpl<hkRefPtr<TYPE>, TYPE> PropertyImpl;

    /// Default constructor.
    /// Stored pointer is set to HK_NULL.
    inline hkRefPtr();

    /// Nullptr constructor.
    inline hkRefPtr(nullptr_t);

    /// Constructor that constructs an hkRefPtr<const T> or an hkRefPtr<T> from an hkRefPtr<T>.
    /// Increase reference count for object in 'rp' and set pointer to it.
    template<typename RTYPE>
    inline hkRefPtr(const hkRefPtr<RTYPE>& rp);

    /// Prevents the compiler from generating a non-templated copy constructor.
    inline hkRefPtr(const hkRefPtr<TYPE>& rp);

    /// Constructor. Transfers ownership from 'rp' without changing the refcount.
    template<typename RTYPE>
    hkRefPtr(hkRefPtr<RTYPE>&& rp);

    /// Move constructor. Transfers ownership from 'rp' without changing the refcount.
    /// (we don't want the compiler to generate a default move constructor)
    hkRefPtr(hkRefPtr<TYPE>&& rp);

    /// Add a reference count for object in 'rp' and set pointer to it.
    inline hkRefPtr(hkRefLoan<TYPE> rp);

    /// Steal (don't increment) the reference from 'rp'.
    template<typename RTYPE>
    inline hkRefPtr(hkRefNew<RTYPE> rp);

    /// Add a reference count for object in 'rp' and set pointer to it.
    template<typename RTYPE>
    inline hkRefPtr(const hkAutoRemoveReferenceWrapper<RTYPE>& rp);

    /// Constructor from pointer.
    /// Increase reference count for object 'e' and set the pointer to it.
    inline hkRefPtr(_In_ TYPE* e);

    /// Constructor from hkViewPtr.
    /// Increase reference count for object pointed by rp and set the pointer to it.
    template<typename RTYPE>
    inline hkRefPtr(const hkViewPtr<RTYPE>& rp);

    /// Destructor.
    /// Decrease reference count for stored object.
    inline ~hkRefPtr();

    /// Assignment operator that assigns to a hkRefPtr<const T> or a hkRefPtr<T> from an hkRefPtr<T>.
    /// Increase reference count for object in 'rp',
    /// decrease reference count for stored object and set pointer to object from 'rp'.
    template<typename RTYPE>
    hkRefPtr& operator=(const hkRefPtr<RTYPE>& rp);

    /// Prevents the compiler from generating a non-templated assignment operator.
    hkRefPtr& operator=(const hkRefPtr<TYPE>& rp);

    /// Move assignment operator. Transfers ownership from 'rp' without changing the refcount
    template<typename RTYPE>
    hkRefPtr& operator=(hkRefPtr<RTYPE>&& rp);

    /// Assignment operator.
    /// (we don't want the compiler to generate a default move assignment operator)
    hkRefPtr& operator=(hkRefPtr&& rp);

    /// Assignment operator.
    /// Increase reference count for object in 'rp',
    /// decrease reference count for stored object and set pointer to object from 'rp'.
    inline void operator=(const hkRefLoan<TYPE>& rp);

    /// Assignment operator.
    /// Steals (does not increment) the reference from 'rp'.
    /// Decrease the original reference count and set pointer to object from 'rp'.
    template<typename RTYPE>
    inline void operator=(const hkRefNew<RTYPE>& rp);

    /// Assignment operator.
    /// Increase reference count for object in 'rp',
    /// decrease reference count for stored object and set pointer to object from 'rp'.
    template<typename RTYPE>
    inline void operator=(const hkAutoRemoveReferenceWrapper<RTYPE>& rp);

    /// Assignment operator.
    /// Increase reference count for object 'e',
    /// decrease reference count for stored object and set pointer to 'e'.
    HK_ALWAYS_INLINE void operator=(_In_opt_ TYPE* e);

    /// Assignment operator.
    /// Increase reference count for object in 'rp',
    /// decrease reference count for stored object and set pointer to object from 'rp'.
    template<typename RTYPE>
    HK_INLINE void operator=(const hkViewPtr<RTYPE>& rp);


        /// Pointer to stored object.
    HK_ALWAYS_INLINE _Ret_notnull_ TYPE* operator->() const;

    /// Replace stored pointer with 'e' without incrementing reference count for 'e'.
    /// Reference count for previously stored object is decreased.
    inline void setAndDontIncrementRefCount(_In_opt_ TYPE* e);

    /// Unset stored pointer without decreasing the reference count.
    inline void unsetAndDontDecrementRefCount();

    /// Swaps the ownership of objects with 'rp' without incrementing reference counts for both.
    void swap(hkRefPtr& rp);

    HK_ALWAYS_INLINE _Ret_notnull_ operator TYPE* () const;

    /// Return pointer to stored object.
    HK_ALWAYS_INLINE _Ret_maybenull_ TYPE* val() const;


    HK_ALWAYS_INLINE operator hkReflect::Var() const { return hkReflect::Var(m_ptr); }

private:
    template<typename RTYPE>
    void moveAssign(hkRefPtr<RTYPE>&& rp);

    TYPE* m_ptr;

    template<typename Q>
    friend class hkRefPtr;
    friend class hkRefVariant;
};

/// Swaps two hkRefPtr without touching reference counters
template <typename TYPE>
void swap(hkRefPtr<TYPE>& a, hkRefPtr<TYPE>& b);

namespace hk
{
    /// Constructs a new object using provided arguments and returns an owning hkRefPtr
    
    template <typename TYPE>
    hkRefPtr<TYPE> makeRef();

    template <typename TYPE, typename T1>
    hkRefPtr<TYPE> makeRef(T1&& t1);

    template <typename TYPE, typename T1, typename T2>
    hkRefPtr<TYPE> makeRef(T1&& t1, T2&& t2);

    template <typename TYPE, typename T1, typename T2, typename T3>
    hkRefPtr<TYPE> makeRef(T1&& t1, T2&& t2, T3&& t3);

    template <typename TYPE, typename T1, typename T2, typename T3, typename T4>
    hkRefPtr<TYPE> makeRef(T1&& t1, T2&& t2, T3&& t3, T4&& t4);

    template <typename TYPE, typename T1, typename T2, typename T3, typename T4, typename T5>
    hkRefPtr<TYPE> makeRef(T1&& t1, T2&& t2, T3&& t3, T4&& t4, T5&& t5);

}  // namespace hk





#include <Common/Base/Types/hkRefPtr.inl>
#include <Common/Base/_Auto/TemplateTypes/hkRefPtr_Types.inl>

namespace hkReflect
{
    template <typename To, typename From>
    hkViewPtr< typename hkTrait::ConstIfConst<From, To>::type > isExactlyA(const hkViewPtr<From>& from)
    {
        return isExactlyA<To, From>(from.get());
    }

    /// Returns 'from' cast as 'To*' if the dynamic type of 'from' is exactly the reflection type of 'To', HK_NULL otherwise.
    template <typename To, typename From>
    hkRefPtr< typename hkTrait::ConstIfConst<From, To>::type > isExactlyA(const hkRefPtr<From>& from)
    {
        const To* res = isExactlyA<To>(from.val());
        hkRefPtr< typename hkTrait::ConstIfConst<From, To>::type > ret;
        if (res != HK_NULL)
        {
            ret = static_cast< typename hkTrait::ConstIfConst<From, To>::type* >(from.val());
        }
        return ret;
    }

    template< typename T >
    struct ReflectionOf< Detail::MarkAsOpaque< hkViewPtr<T> > > : public ReflectionOf < hkViewPtr< Detail::Opaque > >
    {
    };

    template< typename T >
    struct ReflectionOf< Detail::MarkAsOpaque< hkRefPtr<T> > > : public ReflectionOf < hkRefPtr< Detail::Opaque > >
    {
    };
}

namespace hkDetail { struct IndirectPtr; }

/// Type-deducing version of hkDynCast for hkRefPtr. We need a indirection instead of taking a
/// hkRefPtr<T> to avoid ambiguities on expressions like hkDynCast<U>(refptr).
HK_ALWAYS_INLINE hkReflect::Detail::AutoCast hkDynCast(const hkDetail::IndirectPtr& from);

/// Type-deducing version of hkDynCast for hkRefPtr. We need a indirection instead of taking a
/// hkRefPtr<T> to avoid ambiguities on expressions like hkDynCast<U>(refptr).
HK_ALWAYS_INLINE _Ret_maybenull_ void* hkDynCast(const hkDetail::IndirectPtr& from, _In_ const hkReflect::Type* toType);

template<typename T, typename DEDUCED>
HK_ALWAYS_INLINE _Ret_maybenull_ T* hkPtrConstCast(const hkRefPtr<DEDUCED>& ptr);

template<typename T, typename DEDUCED>
HK_ALWAYS_INLINE _Ret_maybenull_ T* hkPtrStaticCast(const hkRefPtr<DEDUCED>& ptr);

template<typename T, typename DEDUCED>
HK_ALWAYS_INLINE _Ret_maybenull_ T* hkPtrReinterpretCast(const hkRefPtr<DEDUCED>& ptr);

template<typename T, typename DEDUCED>
HK_ALWAYS_INLINE _Ret_maybenull_ T* hkPtrConstCast(const hkViewPtr<DEDUCED>& ptr);

template<typename T, typename DEDUCED>
HK_ALWAYS_INLINE _Ret_maybenull_ T* hkPtrStaticCast(const hkViewPtr<DEDUCED>& ptr);

template<typename T, typename DEDUCED>
HK_ALWAYS_INLINE _Ret_maybenull_ T* hkPtrReinterpretCast(const hkViewPtr<DEDUCED>& ptr);

namespace hkReflect
{
    /// Returns the data pointer cast to 'To*' if if the type of the pointed object is exactly 'To', HK_NULL otherwise.
    /// Advanced use, use hkDynCast for dynamic casts instead.
    template <typename To, typename From>
    hkRefPtr< typename hkTrait::ConstIfConst<From, To>::type > exactMatchDynCast(const hkRefPtr<From>& from)
    {
        hkRefPtr< typename hkTrait::ConstIfConst<From, To>::type > ret;
        if (const To* res = exactMatchDynCast<To>(from.val()))
        {
            ret = static_cast<typename hkTrait::ConstIfConst<From, To>::type*>(from.val());
        }
        return ret;
    }

    // Overloads for hkRefPtr.
    template <typename T>
    HK_ALWAYS_INLINE hkReflect::Detail::AddrAndType exactObj(const hkRefPtr<T>& p) { return exactObj(p.val()); }

    
    template <typename T>
    HK_ALWAYS_INLINE _Ret_notnull_ const hkReflect::Type* exactTypeOf(const hkRefPtr<T>& p) { return exactTypeOf(p.val()); }

    // Overloads for hkViewPtr.
    template <typename T>
    HK_ALWAYS_INLINE hkReflect::Detail::AddrAndType exactObj(hkViewPtr<T> p) { return exactObj(p.val()); }

    
    template <typename T>
    HK_ALWAYS_INLINE _Ret_notnull_ const hkReflect::Type* exactTypeOf(hkViewPtr<T> p) { return exactTypeOf(p.val()); }
}

namespace hkTrait
{
    template <typename T>
    struct RemoveConstPtr< hkViewPtr<const T> > { typedef hkViewPtr<T> Type; };

    template <typename T>
    struct RemoveConstPtr< hkRefPtr<const T> > { typedef hkRefPtr<T> Type; };

    template <typename T>
    struct RemovePointer< hkViewPtr<T> > { typedef T type; };

    template <typename T>
    struct RemovePointer< hkRefPtr<T> > { typedef T type; };

    template<typename T>
    struct CanBeReinterpreted< T*, const hkRefPtr<T> > : public TrueType {};

    template<typename T>
    struct CanBeReinterpreted< hkRefPtr<T>, T* const> : public TrueType {};

    template<typename T>
    struct CanBeReinterpreted< T*, const hkRefPtr<const T> > : public TrueType {};

    template<typename T>
    struct CanBeReinterpreted< hkRefPtr<T>, const T* const> : public TrueType {};
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
