// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// this: #include <Common/Base/Types/hkHandle.h>


/// The purpose of this class is to allow declaration of a unique type which is
/// used as index in another structure, or as key in a hashed structure. It is
/// normally used to replace an integer type (which is used as internal storage
/// anyway) with an hkHandle type. Implicit construction of the handle from the
/// storage type is not allowed. Usage of hkHandles obtained from a Havok
/// class is usually opaque (the user doesn't manipulate the handle value directly,
/// but normally it uses it to identify elements inside a Havok structure).
///
/// Declarations of hkHandle types should happen using the HK_DECLARE_HANDLE macro.
/// Handles with the same type and invalid value will be different types if they have
/// been given different names using the HK_DECLARE_HANDLE macro.
///
/// Use HK_DECLARE_HANDLE( name , type , invalid value ) macro for declarations.
template <typename TYPE, HK_REFLECT_RESOLVE(TYPE) INVALID_VALUE, typename DISCRIMINANT>
struct hkHandle
{
        HK_DECLARE_CLASS(hkHandle, NewOpaque, Reflect, Pod);
        HK_RECORD_ATTR( hk::IgnoreTemplateParams(DISCRIMINANT) );

        /// Underlying type.
        typedef TYPE    Type;

        /// Invalid value.
        enum { InvalidValue = INVALID_VALUE };

        /// Only allow for explicit or default construction.
        template <typename T>
        explicit HK_ALWAYS_INLINE HK_CONSTEXPR hkHandle(T v);

#if defined(HK_COMPILER_GHS) || defined(HK_PLATFORM_PSVITA)
        explicit HK_ALWAYS_INLINE HK_CONSTEXPR hkHandle(int v);
#endif

         HK_ALWAYS_INLINE            hkHandle() {}

        /// Assignment.
        HK_INLINE hkHandle& operator =(hkHandle other);

        /// Check for '=='.
        HK_ALWAYS_INLINE bool       operator ==(hkHandle other) const;

        /// Check for '!='.
        HK_ALWAYS_INLINE bool       operator !=(hkHandle other) const;

        /// Check for '<'.
        HK_ALWAYS_INLINE bool       operator <(hkHandle other) const;

        /// Check for '<='.
        HK_ALWAYS_INLINE bool       operator <=(hkHandle other) const;

        /// Check for '>'.
        HK_ALWAYS_INLINE bool       operator >(hkHandle other) const;

        /// Check for '>='.
        HK_ALWAYS_INLINE bool       operator >=(hkHandle other) const;

        /// Read-only access to the actual index value.
        HK_ALWAYS_INLINE    TYPE        value() const;

        HK_ALWAYS_INLINE    TYPE        valueUnchecked() const;

        /// Check if the ID is valid ( != INVALID_VALUE / != invalid() )
        HK_INLINE bool      isValid() const;

        /// Makes sure this handle is set to be invalid.
        HK_INLINE void setInvalid();

        /// Get an invalid ID ( equivalent to hkHandle(INVALID_VALUE) ).
        static HK_ALWAYS_INLINE HK_CONSTEXPR hkHandle invalid();

    protected:

        TYPE    m_value HK_ATTR(hk::Default(INVALID_VALUE));    ///< Actual handle value.
};

template<typename T, T InvalidVal, typename D>
hkUint32 hkHashValue(hkHandle<T, InvalidVal, D> h)
{
    return hkHash::hkHashValue(h.value());
}

#include <Common/Base/Types/hkHandle.inl>
#include <Common/Base/Reflect/Attributes/hkAttributes.h>
#include <Common/Base/_Auto/TemplateTypes/hkHandle_Types.inl>

#define HK_DECLARE_HANDLE(_name_ , _type_ , _nullvalue_) struct _name_##Discriminant; typedef hkHandle<_type_, _nullvalue_, _name_##Discriminant> _name_

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
