// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#ifndef HK_BASE_ENDIAN_H
#define HK_BASE_ENDIAN_H

#if defined(HK_COMPILER_MSVC) && !defined(__HAVOK_PARSER__)
    #include <intrin.h>
    #define HK_BSWAP16 _byteswap_ushort
    #define HK_BSWAP32 _byteswap_ulong
    #define HK_BSWAP64 _byteswap_uint64
#endif

namespace hkEndian
{
HK_INLINE hkUint8  swap(hkUint8  s) { return s; }
HK_INLINE hkUint16 swap(hkUint16 s)
{
    #ifdef HK_BSWAP16
        return HK_BSWAP16(s);
    #else
        return
            ((s & 0xff00) >> 8)|
            ((s & 0x00ff) << 8);
    #endif
}
HK_INLINE hkUint32 swap(hkUint32 s)
{
    #ifdef HK_BSWAP32
        return HK_BSWAP32(s);
    #else
        return
            ((s & 0xff000000) >> 24)|
            ((s & 0x00ff0000) >>  8)|
            ((s & 0x0000ff00) <<  8)|
            ((s & 0x000000ff) << 24);
    #endif
}
HK_INLINE hkUint64 swap(hkUint64 s)
{
    #ifdef HK_BSWAP64
        return HK_BSWAP64(s);
    #else
        return
            ((s & 0xff00000000000000ull) >> 56)|
            ((s & 0x00ff000000000000ull) >> 40)|
            ((s & 0x0000ff0000000000ull) >> 24)|
            ((s & 0x000000ff00000000ull) >>  8)|
            ((s & 0x00000000ff000000ull) <<  8)|
            ((s & 0x0000000000ff0000ull) << 24)|
            ((s & 0x000000000000ff00ull) << 40)|
            ((s & 0x00000000000000ffull) << 56);
    #endif
}

#undef HK_BSWAP16
#undef HK_BSWAP32
#undef HK_BSWAP64

template< typename T >
HK_INLINE T swap(T t)
{
    typedef typename hkTrait::MatchingIntType<sizeof(T),false>::IntType IntType;
    return hkBitCast<T>( swap( hkBitCast<IntType>(t) ) );
}

template<typename NativeType, bool LittleEndian>
class EndianType
{
    enum { SIZE = sizeof(NativeType) };
    typedef typename hkTrait::MatchingIntType<SIZE,false>::IntType Storage;
    typedef typename hkTrait::TraitBool<LittleEndian==HK_ENDIAN_LITTLE>::Type IsNativeEndian;
    HK_INLINE Storage _swap(Storage src, hkTrait::TrueType isNative) const { return src; }
    HK_INLINE Storage _swap(Storage src, hkTrait::FalseType isNative) const { return hkEndian::swap(src); }
    HK_INLINE Storage _swap(Storage src) const { return _swap(src, IsNativeEndian()); }

public:

    // Conversion is the following steps, either forward or backward.
    // 1. Convert i to NativeType (could be promotion or truncation)
    // 2. Bitcast NativeType to a same-size integer (Storage)
    // 3. Byteswap the Storage (this may be a no-op)

    EndianType() { }

    template<typename T>
    EndianType(T i) { *this = i; }

    template<typename T>
    void operator=(T i)
    {
        m_val = _swap( hkBitCast<Storage>( hkLosslessCast<NativeType>(i) ) );
    }

    bool operator==(EndianType i) const
    {
        return m_val == i.m_val;
    }

    template<typename T>
    inline operator T() const
    {
        Storage st = _swap(m_val);
        NativeType nt = hkBitCast<NativeType>(st);
        return hkLosslessCast<T>(nt);
    }

    typedef NativeType ReturnType;

    template<typename READER>
    static NativeType read(READER& reader)
    {
        return reader.template readPod<EndianType>(); // implicit conversion to native
    }

private:

    Storage m_val;
};

}

namespace hkTrait
{
    template<typename NT, bool LE> struct IsSigned< hkEndian::EndianType<NT,LE> > : public IsSigned<NT> {  };
}

#define HK_ENDIAN_DECLARE(NTYPE) \
    typedef hkEndian::EndianType<NTYPE, true> NTYPE##Le; \
    typedef hkEndian::EndianType<NTYPE, false> NTYPE##Be; \
    HK_REFLECTIONOF_SPECIALIZE(HK_EXPORT_COMMON, NTYPE##Le); \
    HK_REFLECTIONOF_SPECIALIZE_KIND(NTYPE##Le, IntType); \
    HK_REFLECTIONOF_SPECIALIZE(HK_EXPORT_COMMON, NTYPE##Be); \
    HK_REFLECTIONOF_SPECIALIZE_KIND(NTYPE##Be, IntType)

#define HK_ENDIAN_DECLARE2(NTYPE) \
    HK_ENDIAN_DECLARE(NTYPE); \
    namespace hkReflect { \
        template<bool LE> struct ReflectionOf< hkEndian::EndianType<hkReflect::Typedef::NTYPE ## _Tag, LE> > : \
            public ReflectionOf< hkEndian::EndianType<NTYPE, LE> > {}; \
    }

HK_ENDIAN_DECLARE2(hkInt8);
HK_ENDIAN_DECLARE2(hkUint8);
HK_ENDIAN_DECLARE2(hkInt16);
HK_ENDIAN_DECLARE2(hkUint16);
HK_ENDIAN_DECLARE2(hkInt32);
HK_ENDIAN_DECLARE2(hkUint32);
HK_ENDIAN_DECLARE2(hkInt64);
HK_ENDIAN_DECLARE2(hkUint64);
HK_ENDIAN_DECLARE(hkHalf16);
HK_ENDIAN_DECLARE(hkFloat32);
HK_ENDIAN_DECLARE(hkDouble64);

typedef hkDouble64Le hkFloat64Le;
typedef hkDouble64Be hkFloat64Be;

// Dummy types, used only for reflection.
#define HK_ENDIAN_BOOL_DECLARE(SIZE, ENDIAN, ENDIAN_FLAG) \
    class hkBool ## SIZE ## ENDIAN \
    { \
        public: \
            HK_DECLARE_CLASS( hkBool ## SIZE ## ENDIAN, New, Reflect ); \
            HK_REFLECT_AS_BOOL( &hkReflect::Detail::BoolImplN<hkUint ## SIZE>::s_instance, hkReflect::Format::OfBoolGeneric<ENDIAN_FLAG, SIZE>::Value ); \
            hkUint ## SIZE m_storage; \
    };

HK_ENDIAN_BOOL_DECLARE( 8, Le, 0 );
HK_ENDIAN_BOOL_DECLARE( 8, Be, 1 );
HK_ENDIAN_BOOL_DECLARE( 32, Le, 0 );
HK_ENDIAN_BOOL_DECLARE( 32, Be, 1 );

#undef HK_ENDIAN_BOOL_DECLARE
#endif //HK_BASE_ENDIAN_H

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
