// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#ifndef HKBASE_HKBASETYPES_H
#define HKBASE_HKBASETYPES_H

#include <Common/Base/Types/hkBaseDefs.h> 

typedef hkReal hkTime;



HK_INLINE hkInt32 HK_CALL hkAlign( hkInt32 unalignedVal, hkInt32 alignment )
{
    return (unalignedVal + alignment - 1) & (~(alignment - 1));
}

HK_INLINE hkInt32 HK_CALL hkPointerToInt32( _In_opt_ const void* ptr )
{
    return static_cast<int>( hkUlong(ptr) );
}

/// get the byte offset of B - A, as a full long.
HK_ALWAYS_INLINE hkLong HK_CALL hkGetByteOffset(_Maybevalid_ _Notnull_ const void* base, _Pre_ _Maybevalid_ _Notnull_ const void* pntr)
{
    return hkLong(pntr) - hkLong(base);
}

/// get the byte offset of B - A, as an int (64bit issues, so here for easy code checks)
HK_ALWAYS_INLINE int HK_CALL hkGetByteOffsetInt(_Maybevalid_ _Notnull_ const void* base, _Pre_ _Maybevalid_ _Notnull_ const void* pntr)
{
    return static_cast<int>( hkGetByteOffset( base, pntr ) );
}

/// get the byte offset of B - A, as a full 64bit hkUint64.
HK_ALWAYS_INLINE hkInt32 HK_CALL hkGetByteOffsetCpuPtr(_Maybevalid_ const void* base, _Maybevalid_ _Notnull_ const void* pntr)
{
    return hkInt32(hkLong((const char*)(pntr) - (const char*)(base)));
}

template <typename TYPE>
_Post_satisfies_(return == (const char *)(base)+offset)
HK_ALWAYS_INLINE TYPE* HK_CALL hkAddByteOffset(_Maybevalid_ TYPE* base, hkLong offset )
{
    return reinterpret_cast<TYPE*>( reinterpret_cast<hkUlong>(base) + offset );
}

template <typename TYPE>
HK_ALWAYS_INLINE TYPE HK_CALL hkAddByteOffsetCpuPtr( TYPE base, hkLong offset )
{
    return reinterpret_cast<TYPE>( reinterpret_cast<hkUlong>(base) + offset );
}

template <typename TYPE>
_Post_satisfies_(return == (const char *)(base) + offset)
HK_ALWAYS_INLINE const TYPE* HK_CALL hkAddByteOffsetConst(_Maybevalid_ const TYPE* base, hkLong offset )
{
    return reinterpret_cast<const TYPE*>( reinterpret_cast<hkUlong>(base) + offset );
}

template <typename TYPE>
HK_ALWAYS_INLINE TYPE HK_CALL hkAddByteOffsetCpuPtrConst( TYPE base, hkLong offset )
{
    return reinterpret_cast<TYPE>( reinterpret_cast<hkUlong>(base) + offset );
}

template <typename TYPE>
HK_ALWAYS_INLINE bool HK_CALL hkRangesOverlap( TYPE beginA, TYPE endA, TYPE beginB, TYPE endB )
{
    HK_ASSERT_NO_MSG(0x7934288b, endA >= beginA );
    HK_ASSERT_NO_MSG(0x3f371980, endB >= beginB );
    return beginA < endB && beginB < endA;
}

/// If you have a pair of values ( which can convert to hkUlong) and you have one of them, than this function
/// allows you to quickly get the other value of the pair.
template <typename TYPE>
HK_ALWAYS_INLINE TYPE HK_CALL hkSelectOther( const TYPE a, const TYPE pairA, const TYPE pairB )
{
    HK_ASSERT_NO_MSG( 0x1615fae1, ( pairA != pairB ) && (a == pairA || a == pairB) );
    return (TYPE)( hkUlong(a) ^ hkUlong(pairA) ^ hkUlong(pairB) );
}

/// If you have a pair of pointers and you have one pointer, than this function allows you to quickly get the other pointer of the pair.
template <typename TYPE>
HK_ALWAYS_INLINE _Ret_notnull_ TYPE* HK_CALL hkSelect(int select, _In_ TYPE* pairA, _In_ TYPE* pairB)
{
    //HK_ASSERT_NO_MSG( 0xf0345456, select == 0 || select == 1);
    hkUlong ua = hkUlong(pairA);
    hkUlong ub = hkUlong(pairB);
    return reinterpret_cast<TYPE*>( ua ^ ((ua^ub)&(-select)) );
}

    /// Cast from SRC to DST ensuring that converting back gives the original value.
template<typename DST, typename SRC>
HK_ALWAYS_INLINE DST hkLosslessCast(SRC src)
{
    DST dst = static_cast<DST>(src);
    if (!hkTrait::TypesAreEqual<SRC, DST>::result)
    {
        HK_ASSERT_NO_MSG(0x1615fae2, static_cast<SRC>(dst) == src);
    }
    return dst;
}

    /// Be explicit that bits are dropped in the conversion.
template<typename DST, typename SRC>
HK_ALWAYS_INLINE DST hkTruncateCast(SRC src)
{
    return static_cast<DST>(src);
}

    /// Reinterpret between same sized values.
template<typename DST, typename SRC>
HK_ALWAYS_INLINE DST hkBitCast(SRC src)
{
    HK_COMPILE_TIME_ASSERT(sizeof(DST)==sizeof(SRC));
    return *reinterpret_cast<DST*>(&src);
}

namespace hkTrait
{
    template <unsigned SIZE> struct UnsignedFor;
    template<> struct UnsignedFor<1> { typedef hkUint8 Type; };
    template<> struct UnsignedFor<2> { typedef hkUint16 Type; };
    template<> struct UnsignedFor<4> { typedef hkUint32 Type; };
    template<> struct UnsignedFor<8> { typedef hkUint64 Type; };
}

    /// Turn off bits of a value without sign-extension issues.
template <typename VTYPE, typename MTYPE>
HK_ALWAYS_INLINE VTYPE HK_CALL hkClearBits(VTYPE value, MTYPE mask)
{
    typedef typename hkTrait::UnsignedFor<sizeof(VTYPE)>::Type UVTYPE;
    typedef typename hkTrait::UnsignedFor<sizeof(MTYPE)>::Type UMTYPE;
    return VTYPE(UVTYPE(value) & ~UVTYPE(UMTYPE(mask)));
}

    /// Set bits of a value without sign-extension issues.
template <typename VTYPE, typename MTYPE>
HK_ALWAYS_INLINE VTYPE HK_CALL hkSetBits(VTYPE value, MTYPE mask)
{
    typedef typename hkTrait::UnsignedFor<sizeof(VTYPE)>::Type UVTYPE;
    typedef typename hkTrait::UnsignedFor<sizeof(MTYPE)>::Type UMTYPE;
    return VTYPE(UVTYPE(value) | UVTYPE(UMTYPE(mask)));
}

HK_INLINE hkUint32 hkNextPowerOf2(hkUint32 in)
{
    in -= 1;

    in |= in >> 16;
    in |= in >> 8;
    in |= in >> 4;
    in |= in >> 2;
    in |= in >> 1;

    return in + 1;
}

HK_INLINE hkUint32 hkBitmask( int width )
{
    // return (1<<width) - 1,
    // except it must work for width = 32
    // (bear in mind that shift by >=32 places is undefined)
    hkUint32 maskIfNot32 = (1 << (width & 31)) - 1; // ans if width != 32, else 0
    hkUint32 is32 = ((width & 32) >> 5); // 1 if width = 32, else 0
    return maskIfNot32 - is32; // the answer in the case width=32 is -1
}

HK_INLINE hkUint64 hkBitmask64( int width )
{
    // return (1<<width) - 1,
    // except it must work for width = 64
    // (bear in mind that shift by >=64 places is undefined)
    hkUint64 maskIfNot64 = ( hkUint64( 1 ) << ( width & 63 ) ) - 1; // ans if width != 64, else 0
    hkUint64 is64 = ( hkUint64( width & 64 ) >> 6 ); // 1 if width = 64, else 0
    return maskIfNot64 - is64; // the answer in the case width=64 is -1
}

// Returns the size of x in bytes.
// This macros uses sizeof() and is therefore not safe for hkPackInts.
#define hkSizeOf(A) int(sizeof(A))
// Returns the size of x in bits instead of bytes.
// This macros uses sizeof() and is therefore not safe for hkPackInts.
#define hkBitSizeOf( x ) ( sizeof(x) * 8 )

/// A floating point value stored in 16-bit. This is a non IEEE representation using
/// 1 sign bit, 8 exponent bits and 7 mantissa bits. This is basically a 32-bit float
/// truncated. It supports the same floating point range as 32-bit float,
/// but only with 6 bit precision (because the last bit is not rounded and renormalized).
class HK_EXPORT_COMMON hkHalf16
{
    public:

        HK_DECLARE_CLASS(hkHalf16, NewOpaque, Reflect, Pod);
        HK_REFLECT_AS_FLOAT(&hkReflect::Detail::FloatImplN<hkHalf16>::s_instance, hkReflect::Format::OfFloat<hkHalf16>::Value);
        HK_RECORD_ATTR( hk::Default(0) );
        HK_RECORD_ATTR(hk::IncludeInMgd(false));

#ifndef HK_DISABLE_MATH_CONSTRUCTORS
        HK_ALWAYS_INLINE hkHalf16() {}
        HK_INLINE explicit hkHalf16(const hkFloat32& f);
        HK_INLINE explicit hkHalf16(const hkDouble64& f);
#endif

        HK_ALWAYS_INLINE hkHalf16& operator=(const hkHalf16& other)
        {
            m_value = other.m_value;
            return *this;
        }

        HK_INLINE void store( _Out_ double* d ) const
        {
            float f; store(&f);
            *d = double(f);
        }

        template <bool ROUND>
        HK_INLINE void setReal(const hkReal& r);

        template <bool ROUND>
        HK_INLINE void set(const float& r);

        template <bool ROUND>
        HK_INLINE void set(const double& r);

#ifndef HK_DISABLE_MATH_CONSTRUCTORS

        HK_INLINE hkHalf16& operator=(const float& f)
        {
            set<false>(f);
            return *this;
        }
        HK_INLINE hkHalf16& operator=(const double& f)
        {
            set<false>(f);
            return *this;
        }

#endif

        HK_INLINE bool operator==(const hkHalf16& other) const
        {
            return (m_value == other.m_value); // Warning: bitwise compare!
        }

        HK_INLINE void store(_Out_ float* f ) const
        {
            hkUint32* asInt = (hkUint32*)f;
            asInt[0] = m_value << 16;
        }

        HK_ALWAYS_INLINE bool isZero()  const { return 0 == (0x7fff & m_value); }
        HK_ALWAYS_INLINE bool isOne()   const { return 0x3f80 == m_value; }
        HK_ALWAYS_INLINE bool isMax()   const { return m_value == 0x7f7f; }
        HK_ALWAYS_INLINE void setZero() { m_value = 0x0000; }
        HK_ALWAYS_INLINE void setOne()  { m_value = 0x3f80; }
        HK_ALWAYS_INLINE void setMax()  { m_value = 0x7f7f; }
        HK_INLINE static float getMaxValue(){ hkUint32 v = 0x7f7f0000; return *(float*)&v; }
        HK_INLINE void setNegativeOne() { m_value = -16512; /*0xbf80*/ }

        HK_ALWAYS_INLINE operator float() const
        {
            return getFloat32();
        }

        HK_ALWAYS_INLINE float getFloat32() const
        {
            union
            {
                int i;
                float f;
            } u;
            u.i = (m_value <<16);
            return u.f;
        }

        HK_ALWAYS_INLINE double getDouble() const
        {
            return double(getFloat32());
        }

        HK_ALWAYS_INLINE hkReal getReal() const
        {
            return hkReal(getFloat32());
        }

        HK_ALWAYS_INLINE hkInt16 getStorage() const
        {
            return m_value;
        }

        HK_ALWAYS_INLINE void setStorage( hkInt16 val )
        {
            m_value = val;
        }

    private:

        hkInt16 m_value;
};

template <bool ROUND>
HK_INLINE void hkHalf16::set(const float& r)
{
#if defined HK_ARCH_PPC // PPC don't like LHS, if r is in memory, we should use a shortcut.
    m_value = *((hkInt16*)&r);
#else
    union { float f; int i; } u;
    u.f = hkFloat32(r);
    m_value = hkInt16(u.i>>16);
#endif
}

template <>
HK_INLINE void hkHalf16::set<true>(const float& r)
{
    union { float f; int i; } u;
    u.f = hkFloat32(r) * (1.0f + 1.0f/256.f);
    m_value = hkInt16(u.i>>16);
}


template <bool ROUND>
HK_INLINE void hkHalf16::set(const double& r)
{
#if defined HK_ARCH_PPC // PPC don't like LHS, if r is in memory, we should use a shortcut.
    m_value = *((hkInt16*)&r);
#else
    union { float f; int i; } u;
    u.f = hkFloat32(r);
    m_value = hkInt16(u.i>>16);
#endif
}

template <>
HK_INLINE void hkHalf16::set<true>(const double& r)
{
    union { float f; int i; } u;
    u.f = hkFloat32(r) * (1.0f + 1.0f/256.f);
    m_value = hkInt16(u.i>>16);
}

template <bool ROUND>
HK_INLINE void hkHalf16::setReal(const hkReal& r)
{
    set<ROUND>(r);
}

#ifndef HK_DISABLE_MATH_CONSTRUCTORS
HK_INLINE hkHalf16::hkHalf16(const hkFloat32& f)
{
    setReal<true>(f);
}
HK_INLINE hkHalf16::hkHalf16(const hkDouble64& f)
{
    setReal<true>(hkFloat32(f));
}
#endif

namespace hkReflect
{
    namespace Detail
    {
        template<> struct AggregateOf<hkHalf16,void>
        {
            struct Type
            {
                Type(hkFloat32 f)
                {
                    union
                    {
                        hkFloat32 f;
                        hkUint32 i;
                    } u;
                    u.f = f;
                    m_topHalf = u.i >> 16;
                }
                hkUint16 m_topHalf;
            };
        };
    }
    namespace Format
    {
        template<> struct OfFloat<hkHalf16> { enum { Value = Format::OfFloatGeneric<HK_ENDIAN_BIG,1,1,8,23-16>::Value }; };
    }
}

class HK_EXPORT_COMMON hkHalf32
{
    public:

        HK_DECLARE_CLASS(hkHalf32, NewOpaque, Reflect, Pod);
        HK_REFLECT_AS_FLOAT(&hkReflect::Detail::FloatImplN<hkFloat32>::s_instance, hkReflect::Format::OfFloat<hkFloat32>::Value);
        HK_RECORD_ATTR( hk::Default(0) );

#ifndef HK_DISABLE_MATH_CONSTRUCTORS
        HK_ALWAYS_INLINE hkHalf32() {}
        HK_INLINE explicit hkHalf32(const hkFloat32& f);
        HK_INLINE explicit hkHalf32(const hkDouble64& f);
#endif

        HK_INLINE hkHalf32& operator=(const hkHalf32& other)
        {
            m_value = other.m_value;
            return *this;
        }

        HK_INLINE void store(_Out_ float* f ) const
        {
            *f = m_value;
        }

        HK_INLINE void store(_Out_ double* d ) const
        {
            *d = double(m_value);
        }

        template <bool ROUND>
        HK_INLINE void setReal(const hkReal& r);

        template <bool ROUND>
        HK_INLINE void set(const float& r);

        template <bool ROUND>
        HK_INLINE void set(const double& r);

#if !defined( HK_DISABLE_MATH_CONSTRUCTORS )
        HK_INLINE hkHalf32& operator=(const float& f)
        {
            m_value = f;
            return *this;
        }

        HK_INLINE hkHalf32& operator=(const double& f)
        {
            m_value = float(f);
            return *this;
        }
#endif

        HK_INLINE bool operator==(const hkHalf32& other) const
        {
            const hkUint32* a = (const hkUint32*)&m_value;
            const hkUint32* b = (const hkUint32*)&other.m_value;
            return (*a == *b); // Warning: bitwise compare!
        }

        #define HK_HALF_DOUBLE_FLOAT_MAX 3.40282e+38f

        HK_ALWAYS_INLINE bool isZero() const { return m_value == 0.0f; }
        HK_ALWAYS_INLINE bool isOne()   const { return m_value == 1.0f; }
        HK_ALWAYS_INLINE bool isMax()  const { return m_value == HK_HALF_DOUBLE_FLOAT_MAX; }
        HK_ALWAYS_INLINE void setZero() { m_value = 0.0f; }
        HK_ALWAYS_INLINE void setOne()  { m_value = 1.0f; }
        HK_ALWAYS_INLINE void setMax()  { m_value = HK_HALF_DOUBLE_FLOAT_MAX; }
        HK_ALWAYS_INLINE void setNegativeOne() { m_value = -1.0f; }
        HK_INLINE static float getMaxValue(){ return HK_HALF_DOUBLE_FLOAT_MAX; }

        HK_ALWAYS_INLINE operator float() const
        {
            return m_value;
        }

        HK_ALWAYS_INLINE float getFloat32() const
        {
            return m_value;
        }

        HK_ALWAYS_INLINE double getDouble() const
        {
            return double(m_value);
        }

        HK_ALWAYS_INLINE hkReal getReal() const
        {
            return hkReal(m_value);
        }

        hkFloat32 getStorage() const
        {
            return m_value;
        }

        void setStorage( hkFloat32 val )
        {
            m_value = val;
        }

    public:

        hkFloat32 m_value;
};

template <bool ROUND>
HK_INLINE void hkHalf32::set(const float& r)
{
    m_value = r;
}

template <bool ROUND>
HK_INLINE void hkHalf32::set(const double& r)
{
    HK_ASSERT_NO_MSG( 0xf04fef45, r <= 3.40282e+38f );
    const hkUint64 mask = 0xFFFFFFFF00000000ull;
    hkUint64 t = ((const hkUint64*)&r)[0];
    hkUint64 val = t & mask;
    hkDouble64* f = (hkDouble64*)&val;
    m_value = hkFloat32(*f);
}

template <>
HK_INLINE void hkHalf32::set<true>(const float& r)
{
    m_value = hkFloat32(r);
}

template <>
HK_INLINE void hkHalf32::set<true>(const double& r)
{
    HK_ASSERT_NO_MSG( 0xf04fef45, r <= 3.40282e+38f );
    m_value = hkFloat32(r);
}

template <bool ROUND>
HK_INLINE void hkHalf32::setReal(const hkReal& r)
{
    set<ROUND>(r);
}

#ifndef HK_DISABLE_MATH_CONSTRUCTORS
HK_INLINE hkHalf32::hkHalf32(const hkFloat32& f)
{
    setReal<true>(f);
}

HK_INLINE hkHalf32::hkHalf32(const hkDouble64& f)
{
    setReal<true>(hkFloat32(f));
}
#endif

namespace hkReflect
{
    namespace Detail
    {
        template<> struct AggregateOf<hkHalf32,void>
        {
            typedef hkFloat32 Type;
        };
    }
    namespace Format
    {
        template<> struct OfFloat<hkHalf32> { enum { Value = OfFloat<hkFloat32>::Value }; };
    }
}


/// A floating point value stored in 16-bit according to the IEEE 754-2008
/// format 'binary16' which has 1 sign bit, 5 exponent bits and 10 mantissa bits.
/// This format is compatible with the widely used GPU 'half' variable format.
/// The supported value range is 6.1035156e-5 to 65504.0 (both pos/neg) with
/// 10 bit precision.
class HK_EXPORT_COMMON hkFloat16
{
public:

    HK_DECLARE_CLASS(hkFloat16, NewOpaque, Reflect, Pod, Version(1));

    HK_INLINE hkFloat16& operator=(const hkFloat16& other)
    {
        m_value = other.m_value;
        return *this;
    }

    HK_INLINE void setZero()
    {
        m_value = 0x0000;
    }

    HK_INLINE void setOne()
    {
        m_value = 0x3c00;
    }

    template <bool ROUND>
    void setReal(hkReal r);

    hkFloat32 getFloat32() const;

    HK_ALWAYS_INLINE hkReal getReal() const
    {
        return hkReal(getFloat32());
    }

    HK_ALWAYS_INLINE hkUint16 getBits() const
    {
        return m_value;
    }

    HK_INLINE void setBits(hkUint16 b)
    {
        m_value = b;
    }

#ifndef HK_DISABLE_MATH_CONSTRUCTORS
    HK_ALWAYS_INLINE hkFloat16() {}
    HK_INLINE explicit hkFloat16(const hkFloat32& f);
#endif

private:
    hkUint16 m_value;
};



#ifndef HK_DISABLE_MATH_CONSTRUCTORS
hkFloat16::hkFloat16(const hkFloat32& f)
{
    setReal<true>(f);
}
#endif


// numerical limits for float16:
namespace hkTrait
{
    template<> struct NumericLimits<hkFloat16>
    {
        static hkFloat16 minValue() { hkFloat16 val; val.setBits(0xFBFF); return val; }
        static hkFloat16 maxValue() { hkFloat16 val; val.setBits(0x7BFF); return val; }
    };
}


//#define HK_BOOL32_IS_STRICT
enum hkBool32FalseType { hkFalse32 = 0 };

#if defined(HK_BOOL32_IS_STRICT)
class hkBool32;
#else
#   if defined(HK_COMPILER_MSVC) || defined(HK_COMPILER_INTEL)
#       pragma warning( disable : 4800 ) // forcing value to bool 'true' or 'false' (performance warning)
#   endif
    /// Bool used for inner loops.
    ///
    /// False is zero, true is _any_ non-zero value. Advice: simply check for != false.
    /// \warning The following operations on hkBool32 will not work as expected: ==, >, <, ~, ^, &
    /// These will work: ! , &&, ||, |
typedef hkUint32 hkBool32;
#endif

typedef hkUint64 hkBool64;
#if defined(HK_ARCH_SUPPORTS_INT64)
typedef hkUint64 hkBoolLL;
#else
typedef hkUint32 hkBoolLL;
#endif

    /// A wrapper to store a hkBool in one byte, regardless of compiler options.
class HK_EXPORT_COMMON hkBool
{
    public:

        HK_DECLARE_CLASS(hkBool, NewOpaque, Reflect, Pod);
        HK_REFLECT_AS_BOOL(&hkReflect::Detail::BoolImplN<hkUint8>::s_instance, hkReflect::Format::OfBool<hkUint8>::Value);
        HK_RECORD_ATTR(hk::IncludeInMgd(false));

            // used in compile time asserts
        typedef char CompileTimeTrueType;
        typedef int CompileTimeFalseType;

        HK_ALWAYS_INLINE hkBool()
        {
        }

        HK_ALWAYS_INLINE hkBool(bool b)
        {
            m_bool = static_cast<char>(b);
        }


#if defined(HK_BOOL32_IS_STRICT)
        HK_INLINE hkBool(int  b){ m_bool = b!=0; }
        HK_INLINE hkBool(hkUint32  b){ m_bool = b!=0; }
        HK_INLINE hkBool(const hkBool32& b);
        HK_INLINE hkBool& operator=(const hkBool32& e);
#endif

        HK_ALWAYS_INLINE operator bool() const
        {
            return m_bool != 0;
        }

        HK_INLINE hkBool& operator=(bool e)
        {
            m_bool = static_cast<char>(e);
            return *this;
        }


        HK_INLINE hkBool operator==(bool e) const
        {
            return (m_bool != 0) == e;
        }

        HK_INLINE hkBool operator!=(bool e) const
        {
            return (m_bool != 0) != e;
        }

    private:
#if defined(HK_BOOL32_IS_STRICT)
        friend class hkBool32;
#endif
        char m_bool;
};


#if defined(HK_BOOL32_IS_STRICT)
class hkBool32
{
    // operators that work as expected
public:

    HK_INLINE hkBool32() {}

    HK_INLINE hkBool32(const hkBool& b)  { m_bool = b.m_bool; }

    HK_INLINE hkBool32(hkUint32 i) : m_bool(i) {}

    HK_INLINE hkBool32(const hkBool32& rhs): m_bool(rhs.m_bool) {}

    HK_INLINE operator bool() const { return m_bool != 0; }

    HK_INLINE operator hkBool() const { return hkBool(m_bool != 0); }

    //HK_INLINE hkBool32& operator =(const hkBool32& b) { m_bool = b.m_bool; return *this; }

    HK_INLINE bool operator !() const { return !m_bool; }

    HK_INLINE bool operator &&(const hkBool32& b) const { return m_bool && b.m_bool; }
    HK_INLINE bool operator &&(const hkBool& b) const { return m_bool && b; }
    HK_INLINE bool operator &&(const bool b) const { return (m_bool!=0) && b; }

    HK_INLINE bool operator ||(const hkBool32& b) const { return m_bool || b.m_bool; }
    HK_INLINE bool operator ||(const hkBool& b) const { return m_bool || b; }
    HK_INLINE bool operator ||(const bool b) const { return (m_bool!=0) || b; }

    //  HK_INLINE   bool operator !=(const bool b) const{ if ( b!= false) {HK_BREAKPOINT(0xf0345456);}; return m_bool != 0; }
    //  HK_INLINE   bool operator ==(const bool b) const{ if ( b!= false) {HK_BREAKPOINT(0xf0345456);}; return m_bool == 0; }
    //  HK_INLINE   bool operator !=(const int b) const{ if ( b!= 0) {HK_BREAKPOINT(0xf0345456);}; return m_bool != 0; }
    HK_INLINE   bool operator ==(hkBool32FalseType) const{ return m_bool == 0; }
    HK_INLINE   bool operator !=(hkBool32FalseType) const{ return m_bool != 0; }

    HK_INLINE hkBool32 operator |(const hkBool32& b) const { return m_bool | b.m_bool; }

    HK_INLINE hkBool32& operator |=(const hkBool32& b) { m_bool |= b.m_bool; return *this; }

    // operators that don't work
private:

    bool operator ==(const hkBool32& b) const;
    bool operator !=(const hkBool32& b) const;
    bool operator <=(const hkBool32& b) const;
    bool operator >=(const hkBool32& b) const;
    bool operator < (const hkBool32& b) const;
    bool operator > (const hkBool32& b) const;

    hkBool32 operator ~() const;
    hkBool32 operator &(const hkBool32& b) const;
    hkBool32 operator ^(const hkBool32& b) const;

    hkBool32& operator &=(const hkBool32& b);
    hkBool32& operator ^=(const hkBool32& b);

public:

    hkUint32 m_bool;
};
#endif

#if defined(HK_BOOL32_IS_STRICT)
    HK_INLINE hkBool::hkBool(const hkBool32& b)
    {
        m_bool = bool(b);
    }

    HK_INLINE hkBool& hkBool::operator=(const hkBool32& b)
    {
        m_bool = bool(b);
        return *this;
    }
#endif


// A lookup table for converting unsigned char to float
// useful for avoiding LHS

extern "C"
{
    extern const hkFloat32 hkUInt8ToFloat32[256];
}


/// A return type for functions.
struct HK_EXPORT_COMMON hkResult
{
    HK_DECLARE_CLASS(hkResult, New, Reflect, Pod, Version(1));
    HK_RECORD_ATTR(hk::IncludeInMgd(false));
    HK_RECORD_ATTR(hk::ToString(&hkResult::toString));
    //+hk.MemoryTracker(ignore=True)

    HK_ALWAYS_INLINE bool isSuccess() const
    {
        return m_code >= 0;
    }

    HK_ALWAYS_INLINE bool isFailure() const
    {
        return m_code < 0;
    }

    HK_INLINE bool isExactly(hkResult r) const
    {
        return m_code == r.m_code;
    }

    static HK_INLINE hkResult HK_CALL fromErrorCode(int code) { hkResult r; r.m_code = code; return r; }

    // Converts hkResult to string, contents in given stringbuf are replaced.
    void toString(hkStringBuf& sb) const;

    // Reflection based toString, for use in hkLog. Uses toString member function.
    static void toString(const hkReflect::Var& var, hkStringBuf& sb, const hkStringView& extra);

    int m_code;
};

// Using COM error code structure: highest bit set to 1 indicates error,
// bits 26-16 set to 4 indicates facility is FACILITY_ITF which allows for per interface
// definition of error codes (so codes from 2 different interfaces do not have to be unique).
// Docs recommend using FACILITY_ITF code (bits 15-0) in range 0x0200-0xFFFF, as that is outside
// the range of COM-defined FACILITY_ITF codes.
// HK_E_<> codes re-use low bits from corresponding system error codes, as defined in WinError.h.
//
// When adding a new hkResult code, update the hkResultToString table in hkBaseTypes.cpp accordingly.
//
#if !defined(_MSC_VER) || (_MSC_VER > 1700)
#   define HK_RESULT_FROM_ERROR_CODE(x) hkResult{(int)(x)}
#else
#   define HK_RESULT_FROM_ERROR_CODE(x) hkResult::fromErrorCode((int)(x))
#endif
#define  HK_SUCCESS                HK_RESULT_FROM_ERROR_CODE(0)          // Success.
#define  HK_FAILURE                HK_RESULT_FROM_ERROR_CODE(0x80040200) // Generic failure.
#define  HK_E_FILE_NOT_FOUND       HK_RESULT_FROM_ERROR_CODE(0x80040202) // File not found.
#define  HK_E_OUTOFMEMORY          HK_RESULT_FROM_ERROR_CODE(0x8004020E) // Out of memory, cannot perform operation.
#define  HK_E_WRITE_PROTECT        HK_RESULT_FROM_ERROR_CODE(0x80040213) // Cannot access write-protected entity.
#define  HK_E_HANDLE_EOF           HK_RESULT_FROM_ERROR_CODE(0x80040226) // End of file.
#define  HK_E_NOT_SUPPORTED        HK_RESULT_FROM_ERROR_CODE(0x80040232) // Operation not supported.
#define  HK_E_INVALID_PARAMETER    HK_RESULT_FROM_ERROR_CODE(0x80040257) // Invalid operation parameter.
#define  HK_E_NOT_IMPLEMENTED      HK_RESULT_FROM_ERROR_CODE(0x80040278) // Operation not implemented.
#define  HK_E_ALREADY_EXISTS       HK_RESULT_FROM_ERROR_CODE(0x800402b7) // Cannot create or modify object of operation because it already exists.
#define  HK_E_UNEXPECTED           HK_RESULT_FROM_ERROR_CODE(0x8004FFFF) // Unexpected internal state.

// undef & redefine these locally if you want
#define HK_DETAIL_RESULT_SYNTHETIC_FAIL(hRes) false
#define HK_DETAIL_RESULT_LOG(hRes,sExpr,...)



/// Note "fmt" is a hkStringBuf.format string i.e. contains "{}", not printf style "%s".
#define HK_RETURN_RES_IF(hRet, bExpr, /* [const char* fmt, fmtArgs...]] */ ...) \
    do { hkResult _hRet = hRet;  if( (bExpr)            || HK_DETAIL_RESULT_SYNTHETIC_FAIL(_hRet) ) { HK_DETAIL_RESULT_LOG(_hRet, #bExpr, __VA_ARGS__); return _hRet; } } while(0, 0)
#define HK_RETURN_IF_FAILED( hExpr, /* [const char* fmt, [fmtArgs...]]*/ ...) \
    do { hkResult _hRet = hExpr; if( _hRet.isFailure()  || HK_DETAIL_RESULT_SYNTHETIC_FAIL(_hRet) ) { HK_DETAIL_RESULT_LOG(_hRet, #hExpr, __VA_ARGS__); return _hRet; } } while(0, 0)
#define HK_RETURN_RES_IF_NULL(hRet, pExpr, /*[const char* fmt, [fmtArgs...]]*/ ...) \
    do { hkResult _hRet = hRet;  if( ((pExpr)==nullptr) || HK_DETAIL_RESULT_SYNTHETIC_FAIL(_hRet) ) { HK_DETAIL_RESULT_LOG(_hRet, #pExpr, __VA_ARGS__); return _hRet; } } while(0, 0)


    /// A wrapper to store an enum with explicit size.
template<typename ENUM, typename STORAGE>
class hkEnum
{
    public:
        typedef STORAGE StorageType;
        typedef ENUM EnumType;

        HK_DECLARE_CLASS(hkEnum, NewOpaque, Reflect, Pod);
        HK_REFLECT_AS_INT( &hkReflect::Detail::IntImplN< HK_REFLECT_RESOLVE( STORAGE ) >::s_instance, hkReflect::Format::OfInt< HK_REFLECT_RESOLVE(STORAGE) >::Value );
        HK_RECORD_ATTR( hk::AddAttribute(hk::Presets, &hkReflect::Detail::PresetsOf<ENUM>::presets) );
        HK_RECORD_ATTR(hk::Validate, hk::HasCustomValidation);

        typedef STORAGE ReflectDefaultType;

        HK_ALWAYS_INLINE hkEnum()
        {
        }

        HK_ALWAYS_INLINE hkEnum(ENUM e)
        {
            m_storage = static_cast<STORAGE>(e);
        }

        HK_ALWAYS_INLINE operator ENUM() const
        {
            return static_cast<ENUM>(m_storage);
        }
        HK_ALWAYS_INLINE void operator=(ENUM e)
        {
            m_storage = static_cast<STORAGE>(e);
        }
        HK_ALWAYS_INLINE hkBool operator==(ENUM e) const
        {
            return m_storage == static_cast<STORAGE>(e);
        }
        HK_ALWAYS_INLINE hkBool operator!=(ENUM e) const
        {
            return m_storage != static_cast<STORAGE>(e);
        }

        HK_ALWAYS_INLINE STORAGE val() const
        {
            return m_storage;
        }

        HK_ALWAYS_INLINE ENUM asEnum() const
        {
            return static_cast<ENUM>(m_storage);
        }

    private:

        STORAGE m_storage;
};

template<typename ENUM, typename STORAGE>
bool hkEnum<ENUM, STORAGE>::reflectValidate() const
{
    return hkReflect::Detail::isValidForPresets(hkReflect::Detail::PresetsOf<ENUM>::presets, this->val());
}

    /// A wrapper to store bitfield with an with explicit size.
template<typename BITS, typename STORAGE>
class hkFlags
{
    public:
        typedef STORAGE StorageType;
        typedef BITS BitsType;

        HK_DECLARE_CLASS(hkFlags, NewOpaque, Reflect, Pod);
        HK_REFLECT_AS_INT( &hkReflect::Detail::IntImplN< HK_REFLECT_RESOLVE( STORAGE ) >::s_instance, hkReflect::Format::OfInt< HK_REFLECT_RESOLVE(STORAGE) >::Value );
        HK_RECORD_ATTR(hk::AddAttribute(hk::Presets, &hkReflect::Detail::PresetsOf<BITS>::presets));
        HK_RECORD_ATTR(hk::Validate, hk::HasCustomValidation);

        typedef STORAGE ReflectDefaultType;

        HK_ALWAYS_INLINE hkFlags()                  {}
        HK_ALWAYS_INLINE hkFlags(STORAGE s)         {   m_storage = s;  }

        HK_ALWAYS_INLINE void clear()               {   m_storage = 0;      }
        HK_ALWAYS_INLINE void clear(STORAGE mask)   {   m_storage &= ~mask; }
        HK_ALWAYS_INLINE void clear(hkFlags mask)   {   m_storage &= ~mask.m_storage;   }
        HK_ALWAYS_INLINE void setAll( STORAGE s )   {   m_storage = s;  }

        HK_ALWAYS_INLINE void orWith( hkFlags s )   {   m_storage |= s.m_storage; }
        HK_ALWAYS_INLINE void orWith( STORAGE s )   {   m_storage |= s; }
        HK_ALWAYS_INLINE void xorWith( STORAGE s )  {   m_storage ^= s; }
        HK_ALWAYS_INLINE void andWith( STORAGE s )  {   m_storage &= s; }
        HK_ALWAYS_INLINE void andWith( hkFlags s )  {   m_storage &= s.m_storage;   }
        HK_ALWAYS_INLINE void andNotWith(STORAGE s) {   m_storage &= ~s; }

        HK_ALWAYS_INLINE void setWithMask( STORAGE s, STORAGE mask )    {   m_storage = (m_storage & ~mask) | (s & mask);   }
        HK_ALWAYS_INLINE STORAGE get() const                            {   return m_storage;           }
        HK_ALWAYS_INLINE STORAGE get( STORAGE mask ) const              {   return m_storage & mask;    }
        HK_ALWAYS_INLINE hkBool32 anyIsSet( STORAGE mask ) const        {   return (m_storage & mask);  }
        HK_ALWAYS_INLINE hkBool32 noneIsSet( STORAGE mask ) const       {   return (m_storage & mask) == 0; }
        HK_ALWAYS_INLINE bool allAreSet( STORAGE mask ) const           {   return (m_storage & mask) == mask;  }

        HK_ALWAYS_INLINE bool operator==( const hkFlags& f ) const      {   return f.m_storage == m_storage;    }
        HK_ALWAYS_INLINE hkBool32 operator!=( const hkFlags& f ) const  {   return f.m_storage ^ m_storage;     }

    protected:

        STORAGE m_storage;
};


template<typename STORAGE>
struct hkFlagsEx : public hkFlags<void, STORAGE> {};

#define HK_DECLARE_FLAGS_EX_CLASS( NAME, BITS ) \
    HK_DECLARE_POD_TYPE(); \
    HK_INLINE NAME() {} \
    HK_INLINE NAME( const StorageType& other ) { *( StorageType* )this = other; } \
    HK_INLINE NAME& operator=( const StorageType& other ) { *( StorageType* )this = other; return *this; } \
    HK_INLINE NAME( const BITS& bits ) { this->operator=( bits ); } \
    HK_INLINE void operator=( const BITS& bits ) \
    { \
        *( StorageType* )this = StorageType( bits ); HK_ASSERT_DEV( 0x22440867, ( hkUlong( bits ) == hkUlong( StorageType( bits ) ) ) || ( hkLong( bits ) == hkLong( StorageType( bits ) ) ), "Storage cannot hold enum" ); \
    } \
    HK_ALWAYS_INLINE operator StorageType() const { return *( StorageType* )this; } \
    HK_ALWAYS_INLINE bool operator==( BITS b ) const { return StorageType( b ) == m_storage; } \
    HK_ALWAYS_INLINE hkBool32 operator!=( BITS b ) const { return StorageType( b ) ^ m_storage; } \
    HK_MUST_END_WITH_SEMICOLON
// template<typename STORAGE>
// bool hkFlags<void, STORAGE>::reflectValidate() const
// {
//     return true;
// }

template<typename BITS, typename STORAGE>
bool hkFlags<BITS, STORAGE>::reflectValidate() const
{
    // Not yet implemented
//    const hk::Presets& presets = hkReflect::getPresetsOf<BITS>();
//    return presets.isValidForFlags(this->val());
    return true;
}

    /// Utility class for a bool-like type which doesn't have implicit conversions.
    /// Use "operator hkBoolOperator::BoolType" to instead of an "operator bool" to
    /// avoid accidental conversions to int for example.
struct HK_EXPORT_COMMON hkBoolOperator
{
    // A bogus type that can't be converted to anything useful but bool.
    typedef void (hkBoolOperator::*BoolType)(void***);
    void dummy(void***) {}

    template<typename T>
    static inline BoolType cast(const T& t) { return t!=0 ? &hkBoolOperator::dummy : HK_NULL; }
};

    /// Copyable, reflected wrapper for a reference.
template <typename T>
struct hkReferenceWrapper
{
    public:
        HK_DECLARE_CLASS( hkReferenceWrapper, New, Reflect, BypassCtorTrivial );
        HK_REFLECT_AS_POINTER( &hkReflect::Detail::ReferenceImpl::s_instance, T );
        HK_RECORD_ATTR( hk::AddFlags( Type::TYPE_POINTER_WEAK ) );

        HK_ALWAYS_INLINE hkReferenceWrapper( T& ref ) : m_val( &ref ) {}

        HK_ALWAYS_INLINE operator T&() const { return *m_val; }
        HK_ALWAYS_INLINE T& get() const { return *m_val; }

    protected:
        T* m_val;
};

    /// Multiply two non-negative signed integers.
    /// Put the possibly truncated result into *ret.
    /// Returns 0 on success, non-zero on failure.
    /// Failures: a or b is negative.
    /// (a*b) is greater than the upperBound.
HK_ALWAYS_INLINE _Must_inspect_result_ hkResult hkCheckMulSizes(
    hkInt32 a,
    hkInt32 b,
    _Out_ hkInt32* ret,
    hkInt32 upperBound = 2147483647/*INT_MAX*/)
{
    hkInt64 r = hkInt64(a) * hkInt64(b);
    *ret = hkInt32(r);
    return (a < 0 || r < 0 || r > upperBound) ? HK_FAILURE : HK_SUCCESS;
}

/// A const char* reflected as a hkReflect::StringType (rather than a pointer to const char).
/// These are never serialized; use a hkStringPtr for string members which should be serialized.
/// The semantics implied by using a hkCString is that the lifetime of the C string is
/// temporary (e.g. a function taking a hkCString must not store a reference to it).


// Todo: These attributes are not being applied
typedef _Null_terminated_ const char* hkCString;
    //HK_ATTR( hk::Reflect ) 

#define hkVAR_UNROLL(x,c)   HK_MULTILINE_MACRO_BEGIN \
{   \
    static_assert( (x) <= 4, "VAR_UNROLL_X_OUT_OF_RANGE" ); \
{ const int hkI = 0%(x); if ( 0 < (x) ){c;} }   \
{ const int hkI = 1%(x); if ( 1 < (x) ){c;} }   \
{ const int hkI = 2%(x); if ( 2 < (x) ){c;} }   \
{ const int hkI = 3%(x); if ( 3 < (x) ){c;} }   \
}   \
    HK_MULTILINE_MACRO_END

// Declare N bytes of reflected, unserialized, padding
#define HK_PADDING_BYTES( M, N ) hkUint8 M[N] HK_ATTR( hk::Serialize(false) );

struct HK_EXPORT_COMMON hkCountOfBadArgCheck
{
    //+hk.MemoryTracker(ignore=True)
    class ArgIsNotAnArray;
    template<typename T> static ArgIsNotAnArray isArrayType(const T*, const T* const*);
    static int isArrayType(const void*, const void*);
};

/// Returns the number of elements in the C array.
#define HK_COUNT_OF(x) ( \
    0 * hkSizeOf( reinterpret_cast<const ::hkCountOfBadArgCheck*>(x) ) + \
    0 * hkSizeOf( ::hkCountOfBadArgCheck::isArrayType((x), &(x)) ) + \
    hkSizeOf(x) / hkSizeOf((x)[0]) )

#endif // HKBASE_HKBASETYPES_H

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
