// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Algorithm/PseudoRandom/hkMT19937RandomGenerator.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
class hkStringBuf;

#include <Common/Base/Math/Vector/hkIntVector.h>

/// Havok UUID class.
/// A Universally Unique IDentifier (UUID) guarantees uniqueness across space and time and can be used to identify
/// different entities. There are several variants of UUIDs defined by different standards:
/// - NCS backward compatibility
/// - RFC-4122
/// - Microsoft Corporation backward compatibility (used for COM)
/// Havok UUIDs adhere to the RFC-4122 standard in terms of generation algorithms.
/// A generic UUID obtained from a different source can still be stored in an hkUuid.
class HK_EXPORT_COMMON hkUuid
{
    //+version(1)

    public:

        HK_DECLARE_CLASS(hkUuid, New, Pod, Reflect);

        /// Constructor
        hkUuid();

        /// Constructor
        hkUuid(hkUint32 d0, hkUint32 d1, hkUint32 d2, hkUint32 d3);

        /// Returns the Nil UUID according to RFC-4122 (a UUID whose all bits are zero).
        static const hkUuid& getNil();

        /// Returns true if the string passed in represents a valid UUID string.
        static bool isValidUuidString(_In_z_ const char* uuidString);

        /// Returns a random UUID
        static hkUuid random();

        /// Creates a UUID from its string representation (see setFromFormattedString).
        static hkUuid fromFormattedString(_In_z_ const char* uuidString);

        /// Creates a UUID by combining two UUIDs (see setCombination);
        static hkUuid combine(const hkUuid& uuid1, const hkUuid& uuid2);

        // Implicit default constructor is available for usage.
        // Implicit copy constructor is available for usage.
        // Implicit copy assignment operator is available for usage.

        /// Sets the value as a new, random Uuid. This random generation ensures that
        /// the probability of a collision is basically too low to be considered possible.
        /// In other words, the generated hkUuid value can be considered unique.
        void setRandom();

        /// Sets the value to the nil UUID (all bits to zero).
        void setNil();

        /// Sets the value of the hkUuid from a given string in the standard format:
        /// XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX.
        /// This function ignores the casing of hexadecimal digits in the string.
        /// If the given string is invalid, HK_FAILURE is returned and this UUID is not changed.
        hkResult setFromFormattedString(_In_z_ const char* uuidString);

        /// Sets the UUID as a combination of a few different UUIDs passed in. If the arguments are exactly
        /// the same, the result is guaranteed to be the same, it is a deterministic process.
        void setCombination(const hkUuid& uuid1, const hkUuid& uuid2);

        /// Comparison operator (equality).
        inline hkBool32 operator==(const hkUuid& rhs) const;

        /// Comparison operator (inequality).
        inline hkBool32 operator!=(const hkUuid& rhs) const;

        /// Comparison operator (binary less than)
        inline hkBool32 operator<(const hkUuid& rhs) const;

        /// Returns true if this UUID is the nil UUID.
        inline bool isNil() const;

        /// Prints a human readable representation of the UUID in the given string buffer,
        /// the returned string will be in the standard format XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX.
        /// The returned pointer references the contents of the string buffer.
        const char* toString(hkStringBuf& sbuf) const;

        /// Gets the data
        template <int I> HK_INLINE hkUint32 getData() const;
        HK_INLINE hkUint32 getData(int I) const;

        /// Sets the data
        template <int I> HK_INLINE void setData(hkUint32 v);
        HK_INLINE void setData(int I, hkUint32 v);

    protected:

        /// Used in NewBase.
        friend struct hkUuidUtil;

        static void reflectionToString(const hkReflect::Var& var, hkStringBuf& sb, const hkStringView& extra);
        HK_RECORD_ATTR(hk::ToString(&hkUuid::reflectionToString));

        hkUint32 m_data[4];
};

// A UUID as a serializable property
class HK_EXPORT_COMMON hkUuidObject : public hkReferencedObject
{
    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
        HK_DECLARE_REFLECTION();

    public:

        /// Constructor
        hkUuidObject(const hkUuid& uuidIn = hkUuid::getNil());

        /// Copy constructor
        hkUuidObject(const hkUuidObject& other);

    public:

        /// The UUID
        hkUuid m_uuid;
};

/// Pseudo-random number generator singleton.
class HK_EXPORT_COMMON hkUuidPseudoRandomGenerator : public hkReferencedObject
{
    public:
        HK_DECLARE_CLASS(hkUuidPseudoRandomGenerator, New, Singleton);

        hkUuidPseudoRandomGenerator();

        void getRand128(hkUint32& outValA, hkUint32& outValB, hkUint32& outValC, hkUint32& outValD);

        /// Sets the seed
        void setSeed(hkUint32 newSeed);

    protected:
        // Mersenne Twister 19937 generator
        // Generates high quality uniformly distributed 32 bit random numbers.
        hkMT19937RandomGenerator m_randomGenerator;
        hkCriticalSection m_criticalSection;
};

HK_ALWAYS_INLINE hkUint32 hkHashValue(const hkUuid& uuid)
{
    // We take just a part of the Uuid, collisions will be more likely to happen but
    // will work well as a hash.
    return uuid.getData<0>();
}

// Make sure that the size of an hkUuid is 128 bits (16 bytes).
HK_COMPILE_TIME_ASSERT(sizeof(hkUuid) == 16);

#include <Common/Base/Types/Uuid/hkUuid.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
