// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Types/Uuid/hkUuid.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Fwd/hkctime.h>
#include <Common/Base/Fwd/hkcctype.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
#include <Common/Base/Object/hkSingleton.h>

#if defined(HK_PLATFORM_PSVITA)
using std::isxdigit;
#endif

#if defined(HK_PLATFORM_PSVITA)
using std::time;
using std::clock;
#endif

// ---------------------------------------------------------------------------------------------------------------------
// Pseudo-random number generator singleton used for hkUuid creation.
// ---------------------------------------------------------------------------------------------------------------------

hkUuidPseudoRandomGenerator::hkUuidPseudoRandomGenerator()
:   m_criticalSection(5000)
{
    const hkUint32 currentTime = static_cast<unsigned long>(time(0));
    const hkUint32 currentClock = static_cast<unsigned long>(clock());
    const hkUint32 currentTickCounter = static_cast<unsigned long>(hkStopwatch::getTickCounter());
    const hkUint32 seed = currentTime ^ currentClock ^ currentTickCounter;

    m_randomGenerator.initialize(seed);
}

void hkUuidPseudoRandomGenerator::getRand128(hkUint32& outValA, hkUint32& outValB, hkUint32& outValC, hkUint32& outValD)
{
    hkCriticalSectionLock generatorLock(&m_criticalSection);
    outValA = m_randomGenerator.getRand();
    outValB = m_randomGenerator.getRand();
    outValC = m_randomGenerator.getRand();
    outValD = m_randomGenerator.getRand();
}

//
//  Sets the seed

void hkUuidPseudoRandomGenerator::setSeed(hkUint32 newSeed)
{
    hkCriticalSectionLock generatorLock(&m_criticalSection);
    m_randomGenerator.initialize(newSeed);
}

HK_SINGLETON_IMPLEMENTATION(hkUuidPseudoRandomGenerator);

// ---------------------------------------------------------------------------------------------------------------------
// Actual hkUuid implementation.
// ---------------------------------------------------------------------------------------------------------------------

static const hkUuid s_nilUuid(0, 0, 0, 0);

const hkUuid& hkUuid::getNil()
{
    return s_nilUuid;
}

void hkUuid::setNil()
{
    m_data[0] = 0;
    m_data[1] = 0;
    m_data[2] = 0;
    m_data[3] = 0;
}

hkUuid hkUuid::fromFormattedString(_In_z_ const char* uuidString)
{
    hkUuid u;
    u.setFromFormattedString(uuidString);
    return u;
}

hkUuid hkUuid::combine(const hkUuid& u1, const hkUuid& u2)
{
    hkUuid u;
    u.setCombination(u1, u2);
    return u;
}

#define CHECK_DASH_AT(index) if(uuidString[index] != '-') { return false; }
#define CHECK_DIGIT_RANGE(start, end) for(int index=start; index<=end; index++) { if(!isxdigit(uuidString[index])) { return false; } }
bool hkUuid::isValidUuidString(_In_z_ const char* uuidString)
{
    const int stringLength = hkString::strLen(uuidString);
    if(stringLength != 36)
    {
        return false;
    }

    CHECK_DIGIT_RANGE(0, 7);
    CHECK_DASH_AT(8);
    CHECK_DIGIT_RANGE(9, 12);
    CHECK_DASH_AT(13);
    CHECK_DIGIT_RANGE(14, 17);
    CHECK_DASH_AT(18);
    CHECK_DIGIT_RANGE(19, 22);
    CHECK_DASH_AT(23);
    CHECK_DIGIT_RANGE(24, 35);

    return true;
}
#undef CHECK_DASH_AT
#undef CHECK_DIGIT_RANGE

hkUuid hkUuid::random()
{
    hkUuid r;
    r.setRandom();
    return r;
}

union hkUuidImpl
{
    struct
    {
        hkUint32 m_timeLow;
        hkUint16 m_timeMid;
        hkUint16 m_timeHiAndVersion;
        hkUint8 m_clockSeqHiAndReserved;
        hkUint8 m_clockSeqLow;
        hkUint8 m_node[6];
    } m_uuid;

    hkUint32 m_val[4];
};

//
//  Constructor

hkUuid::hkUuid()
{
    m_data[0] = m_data[1] = m_data[2] = m_data[3] = 0;
}

hkUuid::hkUuid(hkUint32 d0, hkUint32 d1, hkUint32 d2, hkUint32 d3)
{
    setData<0>(d0); setData<1>(d1);
    setData<2>(d2); setData<3>(d3);
}

void hkUuid::setRandom()
{
    hkUint32 rand1, rand2, rand3, rand4;
    hkUuidPseudoRandomGenerator::getInstance().getRand128(rand1, rand2, rand3, rand4);

    hkUuidImpl uuid;
    uuid.m_uuid.m_timeLow = rand1;
    uuid.m_uuid.m_timeMid = static_cast<hkUint16>(rand2>>16);
    uuid.m_uuid.m_timeHiAndVersion = static_cast<hkUint16>(rand2);
    uuid.m_uuid.m_clockSeqHiAndReserved = static_cast<hkUint8>(rand3>>24);
    uuid.m_uuid.m_clockSeqLow = static_cast<hkUint8>(rand3>>16);
    uuid.m_uuid.m_node[0] = static_cast<hkUint8>(rand3>>8);
    uuid.m_uuid.m_node[1] = static_cast<hkUint8>(rand3);
    uuid.m_uuid.m_node[2] = static_cast<hkUint8>(rand4>>24);
    uuid.m_uuid.m_node[3] = static_cast<hkUint8>(rand4>>16);
    uuid.m_uuid.m_node[4] = static_cast<hkUint8>(rand4>>8);
    uuid.m_uuid.m_node[5] = static_cast<hkUint8>(rand4);

    // Write the variant id (which will identify RFC-4122) -> variant = 10X.
    uuid.m_uuid.m_clockSeqHiAndReserved |= 0x80; // most significant bit of the octet is 1
    uuid.m_uuid.m_clockSeqHiAndReserved &= 0xbf; // second most significant bit of the octer is 0

    // Write the version id (which will identify pseudo-randomly generated) -> version = 0100.
    uuid.m_uuid.m_timeHiAndVersion |= 0x4000;
    uuid.m_uuid.m_timeHiAndVersion &= 0x4fff;

    hkString::memCpy4(m_data, uuid.m_val, 4);
}

hkResult hkUuid::setFromFormattedString(_In_z_ const char* uuidString)
{
    if(!hkUuid::isValidUuidString(uuidString))
    {
        return HK_FAILURE;
    }

    char uuidStringCopy[36+6+1]; // 6 extra characters for expansion, 1 character for final '\0'.
    hkString::memCpy4(&uuidStringCopy[0], uuidString, 9); // copy the string in words of 4 bytes.

    // Walk backwards and expand the string in the buffer, isolating the parts we need to convert.
    // Basically we transform:
    // XXXXXXXX-XXXX-XXXX-XXXX-XXXXXXXXXXXX| (37 characters)
    // into:
    // XXXXXXXX|XXXX|XXXX|XX|XX|XX|XX|XX|XX|XX|XX| (43 characters)
    // Where | is the string terminator '\0'.

    // Copy m_node octet 6.
    uuidStringCopy[42] = '\0';
    uuidStringCopy[41] = uuidStringCopy[35];
    uuidStringCopy[40] = uuidStringCopy[34];

    // Copy m_node octet 5.
    uuidStringCopy[39] = '\0';
    uuidStringCopy[38] = uuidStringCopy[33];
    uuidStringCopy[37] = uuidStringCopy[32];

    // Copy m_node octet 4.
    uuidStringCopy[36] = '\0';
    uuidStringCopy[35] = uuidStringCopy[31];
    uuidStringCopy[34] = uuidStringCopy[30];

    // Copy m_node octet 3.
    uuidStringCopy[33] = '\0';
    uuidStringCopy[32] = uuidStringCopy[29];
    uuidStringCopy[31] = uuidStringCopy[28];

    // Copy m_node octet 2.
    uuidStringCopy[30] = '\0';
    uuidStringCopy[29] = uuidStringCopy[27];
    uuidStringCopy[28] = uuidStringCopy[26];

    // Copy m_node octet 1.
    uuidStringCopy[27] = '\0';
    uuidStringCopy[26] = uuidStringCopy[25];
    uuidStringCopy[25] = uuidStringCopy[24];

    // Copy m_clockSeqLow.
    uuidStringCopy[24] = '\0';
    uuidStringCopy[23] = uuidStringCopy[22];
    uuidStringCopy[22] = uuidStringCopy[21];

    // Finish m_clockSeqHiAndReserved.
    uuidStringCopy[21] = '\0';

    // Finish m_timeHiAndVersion.
    uuidStringCopy[18] = '\0';

    // Finish m_timeMid.
    uuidStringCopy[13] = '\0';

    // Finish m_timeLow.
    uuidStringCopy[8] = '\0';

    // Perform the conversion element by element.
    hkUuidImpl uuid;
    hkUint64 timeLow = hkString::atoull(&uuidStringCopy[0], 16);
    uuid.m_uuid.m_timeLow = static_cast<hkUint32>(timeLow);
    int timeMid = hkString::atoi(&uuidStringCopy[9], 16);
    uuid.m_uuid.m_timeMid = static_cast<hkUint16>(timeMid);
    int timeHiAndVersion = hkString::atoi(&uuidStringCopy[14], 16);
    uuid.m_uuid.m_timeHiAndVersion = static_cast<hkUint16>(timeHiAndVersion);
    int clockSeqHiAndReserved = hkString::atoi(&uuidStringCopy[19], 16);
    uuid.m_uuid.m_clockSeqHiAndReserved = static_cast<hkUint8>(clockSeqHiAndReserved);
    int clockSeqLow = hkString::atoi(&uuidStringCopy[22], 16);
    uuid.m_uuid.m_clockSeqLow = static_cast<hkUint8>(clockSeqLow);
    int node0 = hkString::atoi(&uuidStringCopy[25], 16);
    uuid.m_uuid.m_node[0] = static_cast<hkUint8>(node0);
    int node1 = hkString::atoi(&uuidStringCopy[28], 16);
    uuid.m_uuid.m_node[1] = static_cast<hkUint8>(node1);
    int node2 = hkString::atoi(&uuidStringCopy[31], 16);
    uuid.m_uuid.m_node[2] = static_cast<hkUint8>(node2);
    int node3 = hkString::atoi(&uuidStringCopy[34], 16);
    uuid.m_uuid.m_node[3] = static_cast<hkUint8>(node3);
    int node4 = hkString::atoi(&uuidStringCopy[37], 16);
    uuid.m_uuid.m_node[4] = static_cast<hkUint8>(node4);
    int node5 = hkString::atoi(&uuidStringCopy[40], 16);
    uuid.m_uuid.m_node[5] = static_cast<hkUint8>(node5);

    hkString::memCpy4(m_data, uuid.m_val, 4);
    return HK_SUCCESS;
}

void hkUuid::setCombination(const hkUuid& srcUuid1, const hkUuid& srcUuid2)
{
    HK_ASSERT_NO_MSG(0x4e4fc205, srcUuid1 != srcUuid2);
    HK_ASSERT_NO_MSG(0x6953bbf4, srcUuid1 != getNil());
    HK_ASSERT_NO_MSG(0x680ee458, srcUuid2 != getNil());

    hkUuidImpl uuid;
    hkUuidImpl uuid1;   hkString::memCpy4(uuid1.m_val, srcUuid1.m_data, 4);
    hkUuidImpl uuid2;   hkString::memCpy4(uuid2.m_val, srcUuid2.m_data, 4);

    uuid.m_uuid.m_timeLow = uuid1.m_uuid.m_timeLow ^ uuid2.m_uuid.m_timeLow;
    uuid.m_uuid.m_timeMid = uuid1.m_uuid.m_timeMid ^ uuid2.m_uuid.m_timeMid;
    uuid.m_uuid.m_timeHiAndVersion = uuid1.m_uuid.m_timeHiAndVersion ^ uuid2.m_uuid.m_timeHiAndVersion;
    uuid.m_uuid.m_clockSeqHiAndReserved = uuid1.m_uuid.m_clockSeqHiAndReserved ^ uuid2.m_uuid.m_clockSeqHiAndReserved;
    uuid.m_uuid.m_clockSeqLow = uuid1.m_uuid.m_clockSeqLow ^ uuid2.m_uuid.m_clockSeqLow;
    uuid.m_uuid.m_node[0] = uuid1.m_uuid.m_node[0] ^ uuid2.m_uuid.m_node[0];
    uuid.m_uuid.m_node[1] = uuid1.m_uuid.m_node[1] ^ uuid2.m_uuid.m_node[1];
    uuid.m_uuid.m_node[2] = uuid1.m_uuid.m_node[2] ^ uuid2.m_uuid.m_node[2];
    uuid.m_uuid.m_node[3] = uuid1.m_uuid.m_node[3] ^ uuid2.m_uuid.m_node[3];
    uuid.m_uuid.m_node[4] = uuid1.m_uuid.m_node[4] ^ uuid2.m_uuid.m_node[4];
    uuid.m_uuid.m_node[5] = uuid1.m_uuid.m_node[5] ^ uuid2.m_uuid.m_node[5];

    // Write the variant id (which will identify RFC-4122) -> variant = 10X.
    uuid.m_uuid.m_clockSeqHiAndReserved |= 0x80; // most significant bit of the octet is 1
    uuid.m_uuid.m_clockSeqHiAndReserved &= 0xbf; // second most significant bit of the octer is 0

    // Write the version id (which will identify pseudo-randomly generated) -> version = 0100.
    uuid.m_uuid.m_timeHiAndVersion |= 0x4000;
    uuid.m_uuid.m_timeHiAndVersion &= 0x4fff;

    hkString::memCpy4(m_data, uuid.m_val, 4);
}


const char* hkUuid::toString(hkStringBuf& sbuf) const
{
    hkUuidImpl uuid;
    hkString::memCpy4(uuid.m_val, m_data, 4);

    // Print the Uuid
    sbuf.printf( "%.8x-%.4x-%.4x-%.2x%.2x-%.2x%.2x%.2x%.2x%.2x%.2x",
        uuid.m_uuid.m_timeLow, uuid.m_uuid.m_timeMid, uuid.m_uuid.m_timeHiAndVersion,
        uuid.m_uuid.m_clockSeqHiAndReserved, uuid.m_uuid.m_clockSeqLow,
        uuid.m_uuid.m_node[0], uuid.m_uuid.m_node[1], uuid.m_uuid.m_node[2], uuid.m_uuid.m_node[3], uuid.m_uuid.m_node[4], uuid.m_uuid.m_node[5] );
    return sbuf.cString();
}

void hkUuid::reflectionToString(const hkReflect::Var& var, hkStringBuf& sb, const hkStringView& extra)
{
    static_cast<hkUuid*>(var.getAddress())->toString(sb);
}

//
//  Constructor

hkUuidObject::hkUuidObject(const hkUuid& uuidIn)
:   hkReferencedObject()
,   m_uuid(uuidIn)
{}

//
//  Copy constructor

hkUuidObject::hkUuidObject(const hkUuidObject& other)
:   hkReferencedObject()
,   m_uuid(other.m_uuid)
{}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
