// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <float.h>

namespace hkTrait
{
    /// The integral type that corresponds to the size and sign.
    template<int size, bool isSigned> struct MatchingIntType { };
    template<> struct MatchingIntType<1, true> { typedef hkInt8 IntType; };
    template<> struct MatchingIntType<1, false> { typedef hkUint8 IntType; };
    template<> struct MatchingIntType<2, true> { typedef hkInt16 IntType; };
    template<> struct MatchingIntType<2, false> { typedef hkUint16 IntType; };
    template<> struct MatchingIntType<4, true> { typedef hkInt32 IntType; };
    template<> struct MatchingIntType<4, false> { typedef hkUint32 IntType; };
    template<> struct MatchingIntType<8, true> { typedef hkInt64 IntType; };
    template<> struct MatchingIntType<8, false> { typedef hkUint64 IntType; };

    template<typename T>
    struct CorrespondingIntType
    {
        typedef typename MatchingIntType<sizeof(T), hkTrait::IsSigned<T>::result>::IntType Type;
    };


    /// Obtain the minimum of two integers at compile-time.
    /// Could extend this to generic types.
    template<int VAL1, int VAL2>
    struct IntMin
    {
        enum { Result = If< VAL1 < VAL2
            , Constant<VAL1>
            , Constant<VAL2> >::Type::Result };
    };


    /// Get the highest power of two which evenly divides N.
    /// The used algorithm is:
    /// int getAlignmentRecursive(int size)
    /// {
    ///     if(size%2 == 0)
    ///         return 2 * getAlignment(size/2);
    ///     else
    ///         return 1;
    /// }
    template<int N>
    struct GreatestPowerOfTwoDivisor;

    template<>
    struct GreatestPowerOfTwoDivisor<0> { enum { Result = 1 }; };

    template<int N>
    struct GreatestPowerOfTwoDivisor
    {
        enum { Result = If< (N>1) && (N%2==0)
            , Constant< 2*GreatestPowerOfTwoDivisor<N/2>::Result >
            , Constant<1> >::Type::Result };
    };

    namespace Detail
    {
        template<int SIZE, typename Signed>
        struct NumericLimitsValues {};

        template<>
        struct NumericLimitsValues < 1, hkTrait::TrueType >
        {
            static hkInt8 minValue() { return -128; }
            static hkInt8 maxValue() { return 127; }
        };

        template<>
        struct NumericLimitsValues < 1, hkTrait::FalseType >
        {
            static hkUint8 minValue() { return 0; }
            static hkUint8 maxValue() { return 255; }
        };

        template<>
        struct NumericLimitsValues < 2, hkTrait::TrueType >
        {
            static hkInt16 minValue() { return -0x8000; }
            static hkInt16 maxValue() { return  0x7fff; }
        };

        template<>
        struct NumericLimitsValues < 2, hkTrait::FalseType >
        {
            static hkUint16 minValue() { return 0x0000; }
            static hkUint16 maxValue() { return 0xffff; }
        };

        template<>
        struct NumericLimitsValues < 4, hkTrait::TrueType >
        {
            static hkInt32 minValue() { return -0x7fffffff - 1; }
            static hkInt32 maxValue() { return  0x7fffffff; }
        };

        template<>
        struct NumericLimitsValues < 4, hkTrait::FalseType >
        {
            static hkUint32 minValue() { return 0x00000000U; }
            static hkUint32 maxValue() { return 0xffffffffU; }
        };

        template<>
        struct NumericLimitsValues < 8, hkTrait::TrueType >
        {
            static hkInt64 minValue() { return -0x7fffffffffffffffLL - 1LL; }
            static hkInt64 maxValue() { return  0x7fffffffffffffffLL; }
        };

        template<>
        struct NumericLimitsValues < 8, hkTrait::FalseType >
        {
            static hkUint64 minValue() { return 0x0000000000000000ULL; }
            static hkUint64 maxValue() { return 0xffffffffffffffffULL; }
        };

        template<typename T>
        struct NumericLimitsHelper : public NumericLimitsValues<sizeof(T), typename IsSigned<T>::Type> {};
    }

    template<typename T>
    struct NumericLimits {};

    template<> struct NumericLimits<char> : public Detail::NumericLimitsHelper<char> {};
    template<> struct NumericLimits<signed char> : public Detail::NumericLimitsHelper<signed char> {};
    template<> struct NumericLimits<unsigned char> : public Detail::NumericLimitsHelper<unsigned char> {};
    template<> struct NumericLimits<signed short> : public Detail::NumericLimitsHelper<signed short> {};
    template<> struct NumericLimits<unsigned short> : public Detail::NumericLimitsHelper<unsigned short> {};
    template<> struct NumericLimits<signed int> : public Detail::NumericLimitsHelper<signed int> {};
    template<> struct NumericLimits<unsigned int> : public Detail::NumericLimitsHelper<unsigned int> {};
    template<> struct NumericLimits<signed long> : public Detail::NumericLimitsHelper<signed long> {};
    template<> struct NumericLimits<unsigned long> : public Detail::NumericLimitsHelper<unsigned long> {};
    template<> struct NumericLimits<signed long long> : public Detail::NumericLimitsHelper<signed long long> {};
    template<> struct NumericLimits<unsigned long long> : public Detail::NumericLimitsHelper<unsigned long long> {};

    // float based limits:
//  template<> struct NumericLimits<hkFloat16> // can be found in hkBaseTypes.h

    template<> struct NumericLimits<hkFloat32>
    {
        static hkFloat32 minValue() { return -FLT_MAX; }
        static hkFloat32 maxValue() { return FLT_MAX; }
    };
    template<> struct NumericLimits<hkDouble64>
    {
        static hkDouble64 minValue() { return -DBL_MAX; }
        static hkDouble64 maxValue() { return DBL_MAX; }
    };

}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
