// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#define HK_REF_PROPERTY_BASE_DESTRUCTION 0xf000
#define HK_REF_PROPERTY_BASE_PHYSICS 0xf100

    /// Implements a collection of properties uniquely identified by a key
class HK_EXPORT_COMMON hkRefCountedProperties : public hkReferencedObject
{
    //+version(2)

    public:

        HK_DECLARE_CLASS(hkRefCountedProperties, New, Reflect, BypassCtor);

        typedef hkUint16 PropertyKey;

            /// Entry in the array of properties
        struct Entry
        {
            public:

                enum
                {
                    INVALID_PROPERTY_KEY    = (PropertyKey)-1,  ///< Invalid property key
                };

            public:

                HK_DECLARE_CLASS(Entry, New, Reflect, BypassCtor);

                Entry()
                    : m_object(HK_NULL)
                    , m_key(INVALID_PROPERTY_KEY)
                    , m_flags(0)
                {}

                    /// Returns the key
                HK_INLINE PropertyKey getKey() const            { return m_key; }

                    /// Sets the key
                HK_INLINE void setKey(PropertyKey newKey)       { m_key = newKey; }

                    /// Returns the flags
                HK_INLINE hkUint16 getFlags() const         { return m_flags; }

                    /// Sets the flags
                HK_INLINE void setFlags(hkUint16 newFlags)  { m_flags = newFlags; }

            public:

                    /// The object
                hkRefPtr<hkReferencedObject> m_object;

                    /// The key
                PropertyKey m_key;

                    /// The flags
                hkUint16 m_flags;
        };

    public:

            /// Constructor
        hkRefCountedProperties();

            /// Copy constructor
        hkRefCountedProperties(const hkRefCountedProperties& other);

            /// Destructor
        ~hkRefCountedProperties();

            /// Adds a property to the collection. If a property is already installed for the given key, it will be replaced
        HK_INLINE void addProperty(PropertyKey propertyKey, _Inout_ hkReferencedObject* propertyObject)
        {
            addPropertyInternal( propertyKey, propertyObject, REFERENCE_COUNT_INCREMENT );
        }

            /// Removes all properties
        HK_INLINE void removeAllProperties()
        {
            m_entries.clear();
        }

            /// Removes a property from the collection
        void removeProperty(PropertyKey propertyKey);

            /// Replaces the property at the given key with the given object
        void replaceProperty(PropertyKey propertyKey, _In_ hkReferencedObject* newPropertyObject);

            /// Locates and returns the property at the given key. If no property was found, it will return null.
        _Ret_maybenull_ hkReferencedObject* accessProperty(PropertyKey propertyKey) const;

            /// Returns an existing object or creates a new one if none exists
        template<class TYPE>
        _Ret_notnull_ TYPE* installProperty( int key );

            /// Returns the number of keys
        HK_INLINE int getNumKeys() const
        {
            return m_entries.getSize();
        }

            /// Returns the i-th key
        HK_INLINE PropertyKey getKey(int idx) const
        {
            return m_entries[idx].m_key;
        }

    protected:

            // How to handle reference counts
        enum ReferenceCountHandling
        {
            REFERENCE_COUNT_INCREMENT,
            REFERENCE_COUNT_IGNORE,
        };

        void addPropertyInternal(PropertyKey propertyKey, hkReferencedObject* propertyObject, ReferenceCountHandling referenceHandling);

    protected:

            /// The array of entries
        hkArray<Entry> m_entries;
};


template<class TYPE>
_Ret_notnull_ TYPE* hkRefCountedProperties::installProperty(int key)
{
    TYPE* object = reinterpret_cast<TYPE*>(accessProperty( (PropertyKey)key ));
    if ( !object )
    {
        object = new TYPE;
        addPropertyInternal( (PropertyKey)key, object, REFERENCE_COUNT_IGNORE );
    }
    return object;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
