// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM        : ALL
// PRODUCT        : COMMON
// VISIBILITY    : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Types/Properties/hkPropertyRegistry.h>

#include <Common/Base/Container/String/hkInternedStringRegistry.h>
#include <Common/Base/Object/hkSingleton.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Algorithm/Hash/hkHash.h>

#if !defined(HK_DYNAMIC_DLL)
#include <Common/Base/Types/Properties/hkPropertyRegistry.hxx>
#endif


hkPropertyId hkPropertyId::make(_In_ const hkReflect::Type* type, _In_z_ const char* name, hkPropertyFlags::Enum flags)
{
    return hkPropertyRegistry::getInstance().getPropertyId(type, name, flags);
}

_Ret_notnull_
hkPropertyId::BuildingData* hkPropertyId::buildingData()
{
    if (m_desc.getInt() & DESC_BUILDING)
    {
        HK_ASSERT_NO_MSG(0x66fcfad3, m_desc.getPtr());
        return const_cast<BuildingData*>(reinterpret_cast<const BuildingData*>(m_desc.getPtr()));
    }
    else
    {
        m_desc.setInt(m_desc.getInt() | DESC_BUILDING);
        HK_ASSERT_NO_MSG(0x75952426, !m_desc.getPtr());
        BuildingData* data = new BuildingData();
        m_desc.setPtr(reinterpret_cast<hkPropertyDesc*>(data));
        return data;
    }
}

void hkPropertyId::tryBuildDesc(_Inout_ BuildingData* data)
{
    HK_ASSERT_NO_MSG(0x5346ec33, m_desc.getInt() & DESC_BUILDING);

    if (data->type && data->nameIsSet && data->flagsAreSet) // building data is complete
    {
        m_desc.setInt(m_desc.getInt() & ~DESC_BUILDING);
        m_desc.setPtr(hkPropertyId::make(data->type, data->name, data->flags).desc());
        delete data;
    }
}

void hkPropertyId::afterReflectNew()
{
    if (m_desc.getInt() & DESC_BUILDING)
    {
        BuildingData* data = buildingData();
        HK_ASSERT_NO_MSG(0x55f21a54, data->type);
        data->nameIsSet = true;
        data->flagsAreSet = true;

        tryBuildDesc(buildingData());
        HK_ASSERT_NO_MSG(0x5d6ea9ca, !(m_desc.getInt() & DESC_BUILDING));
    }
}

bool hkPropertyId::setType(_In_ const hkReflect::Type* type)
{
    HK_ASSERT_NO_MSG(0x6c7322eb, type);
    BuildingData* data = buildingData();
    HK_ASSERT_NO_MSG(0x364a1b93, !data->type);
    data->type = type;

    tryBuildDesc(data);
    return true;
}

bool hkPropertyId::setName(_In_z_ const char* name)
{
    BuildingData* data = buildingData();
    HK_ASSERT_NO_MSG(0x317e4b9, !data->name);
    data->name = name;
    data->nameIsSet = true;

    tryBuildDesc(data);
    return true;
}

bool hkPropertyId::setFlags(hkPropertyFlags::Enum flags)
{
    BuildingData* data = buildingData();
    HK_ASSERT_NO_MSG(0x618ff596, data->flags == 0);
    data->flags = flags;
    data->flagsAreSet = true;

    tryBuildDesc(data);
    return true;
}

static hkPropertyFlags::Enum flagsFromType(_In_ const hkReflect::Type* type)
{
    hkPropertyFlags::Enum res = hkPropertyFlags::DEFAULT;
    while (type)
    {
        if (!type->isSerializable())
        {
            res = (hkPropertyFlags::Enum)(res | hkPropertyFlags::NO_SERIALIZE);
        }
        if (type->getValueAttribute<hk::Untracked>(false))
        {
            res = (hkPropertyFlags::Enum)(res | hkPropertyFlags::NO_TRACK);
        }

        if (const hkReflect::PointerType* ptrType = type->asPointer())
        {
            type = ptrType->getSubType().get();
        }
        else
        {
            break;
        }
    }
    return res;
}

//
//    Computes the type code from the type

static hkUint32 computeTypeCode(_In_ const hkReflect::Type* typeIn)
{
    if ( typeIn )
    {
        const hkReflect::Detail::InheritanceInfo* inheritance = hkReflect::TypeDetail::getInheritance(typeIn);
        return (inheritance->getPreOrder() << 16) | inheritance->getRightMost();
    }

    return hkUint32(-1);
}

hkUint32 hkHashValue(const hkPropertyRegistry::Key& key)
{
    return hkHash::combineHashValues(hkHash::hkHashValue(key.typeCode), hkHash::hkHashValue(key.name));
}

bool operator==(const hkPropertyRegistry::Key& lhs, const hkPropertyRegistry::Key& rhs)
{
    return hkStringView(lhs.name) == hkStringView(rhs.name) && lhs.typeCode == rhs.typeCode;
}

namespace
{
    hkResult createPropertyReg(_Inout_ hkRefPtr<hkPropertyRegistry>* ptr)
    {
        if (hkReflect::MutableTypeReg* reg = hkReflect::getTypeReg())
        {
            *ptr = hkRefNew<hkPropertyRegistry>(new hkPropertyRegistry());
            return HK_SUCCESS;
        }

        return HK_FAILURE;
    }

    void HK_CALL onTypeRegChanges(
        const hkReflect::MutableTypeReg::TypesChangedFuncArgs& args,
        void* ptr)
    {
        hkPropertyRegistry* registry = reinterpret_cast<hkPropertyRegistry*>(ptr);
        registry->addTypes(args.m_added);
        registry->removeTypes(args.m_removed);
        registry->replaceTypes(args.m_replaced);
        registry->rebuildIdFromKey();
    }
}

HK_SINGLETON_MANUAL_IMPLEMENTATION(hkPropertyRegistry, &createPropertyReg, &hkSingletonUtil::quit<hkPropertyRegistry>);

hkPropertyRegistry::hkPropertyRegistry()
{
    m_typeRegSubscription = hkReflect::getTypeReg()->subscribeForChange(&onTypeRegChanges, this);
}

void hkPropertyRegistry::registerStaticProperties(_Inout_ hkStaticPropertyDescBase* head)
{
    hkCriticalSectionLock lock(&m_cs);

    hkReflect::MutableTypeReg* typeReg = hkReflect::getTypeReg();
    HK_ASSERT_NO_MSG(0x57202212, typeReg);

    for (hkStaticPropertyDescBase* desc = head; desc; desc = desc->next)
    {
        HK_ON_DEBUG(Key key; key.name = desc->name; key.typeCode = computeTypeCode(desc->type));
        HK_ASSERT_NO_MSG(0x59803966, !m_propertyIdFromKey.isValid(m_propertyIdFromKey.find(key)));

        if (desc->flags.get() == hkPropertyFlags::UNSET)
        {
            // If the type is already registered, get the flags now, if not, we will get the flags on type added callback
            if (const hkReflect::Type* type = typeReg->typeFromType(desc->type))
            {
                desc->flags = flagsFromType(type);
            }
        }

        m_properties.pushBack(PropertyInfo(desc, false));
    }
    rebuildIdFromKey();
}

void hkPropertyRegistry::unregisterStaticProperties(_Inout_ hkStaticPropertyDescBase* head)
{
    hkCriticalSectionLock lock(&m_cs);

    for (hkStaticPropertyDescBase* desc = head; desc; desc = desc->next)
    {
        const int index = m_properties.indexOf(PropertyInfo(desc, false));
        HK_ASSERT_NO_MSG(0x249ea53a, index != -1);
        if (index == -1)
        {
            HK_WARN_ALWAYS(0xabba803d, "Could not find static property to unregister - " << (desc->name ? desc->name : "empty!") << ", ignoring.");
        }
        else
        {
            m_properties.removeAt(index);
        }
    }
    rebuildIdFromKey();
}

hkPropertyRegistry::~hkPropertyRegistry()
{
    for (int i = 0; i < m_properties.getSize(); ++i)
    {
        PropertyInfo& info = m_properties[i];
        if (info.m_1)
        {
            if (info.m_0->name)
            {
                hkString::strFree(const_cast<char*>(info.m_0->name));
            }
            delete info.m_0;
        }
    }
}

hkPropertyId hkPropertyRegistry::getPropertyId(_In_ const hkReflect::Type* type, _In_z_ const char* name, hkPropertyFlags::Enum flags)
{
    hkCriticalSectionLock lock(&m_cs);
    Key key = { name, computeTypeCode(type) };
    if (hkPropertyDesc* ptr = m_propertyIdFromKey.getWithDefault(key, HK_NULL))
    {
        return hkPropertyId(ptr);
    }

    return createPropertyId(type, name, flags);
}

hkPropertyId hkPropertyRegistry::createPropertyId(_In_ const hkReflect::Type* type, _In_z_ const char* name, hkPropertyFlags::Enum flags)
{
    hkCriticalSectionLock lock(&m_cs);

    Key key = { name, computeTypeCode(type) };
    key.name = name ? hkString::strDup(name) : HK_NULL;
    hkPropertyDesc* ptr = new hkPropertyDesc(type, key.name, (flags == hkPropertyFlags::UNSET) ? flagsFromType(type) : flags);
    m_properties.pushBack(PropertyInfo(ptr, true));
    m_propertyIdFromKey.insert(key, ptr);

    return hkPropertyId(ptr);
}

void hkPropertyRegistry::setPropertyFlags(_In_ const hkPropertyDesc* desc, hkPropertyFlags::Enum flags)
{
    hkCriticalSectionLock lock(&m_cs);

    if ((desc->flags.get() != (hkUint32)hkPropertyFlags::UNSET) && ((hkUint32)flags != desc->flags.get()))
    {
        // This can fire if a property was first de-serialized with some flags
        HK_WARN(0x49c7e9a2, "Mismatch between flags in registered property and flags to set. The newly set flags will override the registered ones.");
    }

    if (desc->flags == (hkUint32)hkPropertyFlags::UNSET)
    {
        const_cast<hkPropertyDesc*>(desc)->flags = (hkUint32)flags;
    }
}

void hkPropertyRegistry::addTypes(const hkArrayView<const hkReflect::Type* const>& typesAdded)
{
    hkCriticalSectionLock lock(&m_cs);

    for (int i = 0; i < m_properties.getSize(); ++i)
    {
        hkPropertyDesc* desc = m_properties[i].m_0;
        // Update flags on all desc that are still unset and whose type has been added
        if (m_properties[i].m_0->flags.get() == hkPropertyFlags::UNSET && typesAdded.indexOf(desc->type) != -1)
        {
            desc->flags = flagsFromType(desc->type);
        }
    }
}

void hkPropertyRegistry::removeTypes(const hkArrayView<const hkReflect::Type* const>& typesRemoved)
{
    hkCriticalSectionLock lock(&m_cs);

    for (int i = 0; i < m_properties.getSize(); ++i)
    {
        hkPropertyDesc* desc = m_properties[i].m_0;
        // only remove allocated properties, the static ones should be handled by
        // static deregistration
        if (m_properties[i].m_1 && typesRemoved.indexOf(desc->type) != -1)
        {
            desc->type = HK_NULL;
        }
    }
}

void hkPropertyRegistry::replaceTypes(hkArrayView<hkReflect::MutableTypeReg::TypeReplacement> typeReplacements)
{
    hkCriticalSectionLock lock(&m_cs);

    for(PropertyInfo& propertyInfo : m_properties)
    {
        hkPropertyDesc* desc = propertyInfo.m_0;
        // only replace allocated properties, the static ones can never be
        // replaced.
        if(propertyInfo.m_1)
        {
            for(auto typeReplacement : typeReplacements)
            {
                if(desc->type == typeReplacement.m_old)
                {
                    desc->type = typeReplacement.m_new;
                }
            }
        }
    }
}

void hkPropertyRegistry::rebuildIdFromKey()
{
    hkCriticalSectionLock lock(&m_cs);

    m_propertyIdFromKey.clear();

    for (int i = 0; i < m_properties.getSize(); ++i)
    {
        hkPropertyDesc* desc = m_properties[i].m_0;
        if (desc->type == HK_NULL) // ignore types that have been removed
        {
            continue;
        }

        Key key = { desc->name, computeTypeCode(desc->type)    };
        m_propertyIdFromKey.insert(key, desc);
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
