// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Types/Properties/hkPropertyBag.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Types/Properties/hkDefaultPropertyBag.h>
#include <Common/Base/Memory/Tracker/hkMemoryTrackerSnapshot.h>

// Adding a new property can cause reallocation. This breaks concurrent readers.
// A mitigation is to allocate all properties upfront so at least their address doesn't change
// and the hash map entries don't change, this hack will assert if the invariant is violated in the lifetime of an application.
namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    struct hkPropertyBagHack : public hkPropertyBag
    {
        void lock()
        {
            ensureBag().m_locked = true;
        }
    };
}

extern void HK_EXPORT_COMMON hkPropertyBag_lock(hkPropertyBag& bag);
void hkPropertyBag_lock(hkPropertyBag& bag)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE
    static_cast<hkPropertyBagHack&>(bag).lock();
}


hkPropertyBag::~hkPropertyBag()
{
    clear();
}

hkPropertyBag::hkPropertyBag(const hkPropertyBag& other)
    : m_bag(HK_NULL)
{
    if (other.m_bag)
    {
        ensureBag() = *other.m_bag;
    }
}


hkPropertyBag& hkPropertyBag::operator=(const hkPropertyBag& other)
{
    if (this == &other)
    {
        return *this;
    }
    if (other.m_bag)
    {
        ensureBag() = *other.m_bag;
    }
    else
    {
        clear();
    }
    return *this;
}

hkDefaultPropertyBag& hkPropertyBag::ensureBag()
{
    if (m_bag == HK_NULL)
    {
        m_bag = new hkDefaultPropertyBag();
    }
    return *m_bag;
}


void hkPropertyBag::clear()
{
    if (m_bag)
    {
        delete m_bag;
        m_bag = HK_NULL;
    }
}

bool hkPropertyBag::isEmpty() const
{
    return m_bag ? m_bag->isEmpty() : true;
}

int hkPropertyBag::getSize() const
{
    return m_bag ? m_bag->getSize() : 0;
}

bool hkPropertyBag::hasProperty(hkPropertyId id) const
{
    return m_bag ? m_bag->hasProperty(id) : false;
}

hkResult hkPropertyBag::addChecked(hkPropertyId id, const hkReflect::Var& value)
{
    return ensureBag().addChecked(id, value);
}

hkResult hkPropertyBag::removeChecked(hkPropertyId id)
{
    return m_bag ? m_bag->removeChecked(id) : HK_FAILURE;
}

_Ret_maybenull_
void* hkPropertyBag::getValueAddressChecked(_In_ const hkReflect::Type* type, hkPropertyId id) const
{
    return m_bag ? m_bag->getValueAddressChecked(type, id) : HK_NULL;
}

hkPropertyBag::Entry hkPropertyBag::entryAtIndex(int i) const
{
    HK_ASSERT_NO_MSG(0x55c62988, m_bag);
    auto& item = m_bag->atIndex(i);
    Entry res = {item.m_0, item.m_1.var()};
    return res;
}

void hkPropertyBag::deepCopy(const hkPropertyBag& other)
{
    if (other.m_bag == HK_NULL)
    {
        clear();
        return;
    }
    else if (m_bag == HK_NULL || m_bag == other.m_bag)
    {
        m_bag = new hkDefaultPropertyBag();
    }

    for (auto& entry : other.m_bag->m_propertyMap.viewItems())
    {
        if (!entry.m_0.desc()->isCloneable())
        {
            // nothing to do here
        }
        else if (hkReflect::PointerVar pvar = entry.m_1.var())
        {
            hkReflect::Var clone = entry.m_1.var().clone();
            HK_VERIFY(0x7965c366, addChecked(entry.m_0, clone).isSuccess(), "failed to clone pointer property");
            clone.destroy();
        }
        else
        {
            HK_VERIFY(0x5fbc3da4, addChecked(entry.m_0, entry.m_1.var()).isSuccess(), "failed to update property");
        }
    }
}

struct hkPropertyBag::Impl : public hkReflect::Detail::ArrayImpl
{
    Impl() {}

    virtual hkResult inplaceFixup(_Inout_ void* self, _In_ const hkReflect::Type* selfType, _Inout_ void* target, _In_ const hkReflect::Type* targetType, int count) const HK_OVERRIDE
    {
        hkPropertyBag& bag = *static_cast<hkPropertyBag*>(self);

        hkReflect::Var storedBagVar(target, targetType);
        hkPropertyBag* storedBag = storedBagVar.dynCast<hkPropertyBag>();
        if (storedBag == HK_NULL)
        {
            return HK_FAILURE;
        }

        bag = *storedBag;

        return HK_SUCCESS;
    }

    virtual hkResult getValue(_Inout_ const void* arrAddr, _In_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const HK_OVERRIDE
    {
        const hkReflect::Type * const propertyBagType = hkReflect::getType<hkDefaultPropertyBag>();
        const hkPropertyBag& bag = *static_cast<const hkPropertyBag*>(arrAddr);
        *val = hkReflect::ArrayValue(bag.m_bag, (bag.m_bag ? 1 : 0), propertyBagType);
        return HK_SUCCESS;
    }

    virtual hkResult setNumElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int nelem) const HK_OVERRIDE
    {
        if (nelem != 0 && nelem != 1)
        {
            return HK_FAILURE;
        }

        hkPropertyBag& bag = *static_cast<hkPropertyBag*>(arrAddr);
        if ((bag.m_bag == HK_NULL) != (nelem == 0))
        {
            if (nelem == 0)
            {
                bag.clear();
            }
            else
            {
                bag.ensureBag();
            }
        }
        return HK_SUCCESS;
    }

    hkResult spliceInto(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int index, int numToDel, const hkReflect::ArrayValue& toInsert) const HK_OVERRIDE
    {
        hkPropertyBag& bag = *static_cast<hkPropertyBag*>(arrAddr);

        if (numToDel > 1)
        {
            // Assert?
            return HK_FAILURE;
        }

        if (bag.m_bag && ((numToDel == 1) || (numToDel == -1)))
        {
            bag.clear();
        }

        if (toInsert.getCount() == 0)
        {
            return HK_SUCCESS;
        }

        if (bag.m_bag!= HK_NULL)
        {
            return HK_FAILURE;
        }

        bag.m_bag = toInsert[0].clone().dynCast<hkDefaultPropertyBag>();

        return (bag.m_bag != HK_NULL) ? HK_SUCCESS : HK_FAILURE;
    }

    virtual AllocResult allocateElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, hkReflect::QualType elemType, int n) const HK_OVERRIDE
    {
        hkPropertyBag& bag = *static_cast<hkPropertyBag*>(arrAddr);
        if (bag.m_bag)
        {
            return ALLOC_NOT_EMPTY;
        }
        else if (!elemType->extendsOrEquals<hkDefaultPropertyBag>())
        {
            return ALLOC_FAILURE;
        }
        else if (n > 1)
        {
            return ALLOC_FAILURE;
        }
        else if (n == 1)
        {
            bag.m_bag = elemType->newInstance().dynCast<hkDefaultPropertyBag>();
            return ALLOC_SUCCESS;
        }
        else
        {
            // Allocating 0 elements?
            return ALLOC_SUCCESS;
        }
    }

    /// Free internal allocations.
    virtual void clearAllocs(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* type) const HK_OVERRIDE
    {
        hkPropertyBag& bag = *static_cast<hkPropertyBag*>(arrAddr);

        // Free but *do not* destruct (would cause double deletes on pointed elements).
        if (bag.m_bag)
        {
            hkReflect::Detail::deallocateDefault(bag.m_bag, hkReflect::exactTypeOf(bag.m_bag));
        }
    }
};

void hkPropertyBag::trackerHandler(const hkReflect::Var& var, class hkMemoryTrackerSnapshot& snapshot)
{
    const hkPropertyBag* bag = hkDynCast(var);
    HK_ASSERT_NO_MSG(0x159ad4a6, bag);
    if (bag->m_bag)
    {
        snapshot.addLink(bag->m_bag);
    }
}

void hkPropertyBag::hackAfterReflectNew()
{
    if (m_bag)
    {
        m_bag->hackAfterReflectNew();
    }
}

const hkPropertyBag::Impl hkPropertyBag::s_impl;

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
