// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Types/hkPtrAndInt.h>

struct HK_EXPORT_COMMON hkPropertyId;

struct HK_EXPORT_COMMON hkPropertyFlags
{
    HK_DECLARE_CLASS(hkPropertyFlags, New, Reflect);

    enum Enum
    {
        DEFAULT = 0,

        /// Excludes the property from being serialized
        NO_SERIALIZE = 1 << 0,

        /// Excludes the property from being tracked by the object tracker
        NO_TRACK = 1 << 1,

        /// Excludes the property from being copied in clone and copy operations
        NO_CLONE = 1 << 2,

        /// Get the flags that are already stored with the property
        UNSET = -1
    };
};

/// Property description and identifier
struct HK_EXPORT_COMMON hkPropertyDesc
{
    public:
        HK_DECLARE_CLASS(hkPropertyDesc, New, Reflect, BypassCtor);
        HK_RECORD_ATTR(hk::ExcludeFromVersionCheck);

        bool isSerialized() const { return flags.noneIsSet(hkPropertyFlags::NO_SERIALIZE); }
        bool isTracked() const { return flags.noneIsSet(hkPropertyFlags::NO_TRACK); }
        bool isCloneable() const { return flags.noneIsSet(hkPropertyFlags::NO_CLONE); }

        typedef hkFlags<hkPropertyFlags::Enum, hkUint32> Flags;

        const hkReflect::Type* type;
        const char* name;
        Flags flags;

        HK_INLINE hkPropertyId id() const;

    protected:
        hkPropertyDesc() : type(HK_NULL), name(HK_NULL), flags(hkPropertyFlags::DEFAULT) {}
        hkPropertyDesc(_In_ const hkReflect::Type* t, _In_z_ const char* n)
            : type(t), name(n), flags(hkPropertyFlags::DEFAULT) {}
        hkPropertyDesc(_In_ const hkReflect::Type* t, _In_z_ const char* n, hkPropertyFlags::Enum f)
            : type(t), name(n), flags(f) {}

        friend class hkPropertyRegistry;
};

class hkStaticPropertyDescBase : public hkPropertyDesc
{
    public:
        static hkStaticPropertyDescBase* head;
    protected:
        friend class hkPropertyRegistry;
        hkStaticPropertyDescBase(_In_ const hkReflect::Type* type, _In_z_ const char* name, hkPropertyFlags::Enum flags)
            : hkPropertyDesc(type, name, flags), next(head) { head = this; }

        hkStaticPropertyDescBase* next;
};

/// Static property description, should always be a global variable
template<typename T>
class hkStaticPropertyDesc : public hkStaticPropertyDescBase
{
    public:
        typedef T Type;

        HK_ALWAYS_INLINE hkStaticPropertyDesc(_In_z_ const char* name, hkPropertyFlags::Enum flags = hkPropertyFlags::UNSET)
            : hkStaticPropertyDescBase(hkReflect::getType<Type>(), name, flags) {}
};

struct HK_EXPORT_COMMON hkPropertyId
{
    public:
        HK_DECLARE_CLASS(hkPropertyId, New, Reflect);
        HK_RECORD_ATTR(hk::ExcludeFromVersionCheck);

        hkPropertyId() : m_desc(HK_NULL) {}

        static hkPropertyId make(_In_ const hkReflect::Type* type, _In_z_ const char* name, hkPropertyFlags::Enum flags = hkPropertyFlags::UNSET);

        template<typename T>
        static hkPropertyId make(_In_z_ const char* name, hkPropertyFlags::Enum flags = hkPropertyFlags::UNSET) { return make(hkReflect::getType<T>(), name, flags); }

        bool operator==(const hkPropertyId& other) const { return m_desc == other.m_desc; }
        bool operator!=(const hkPropertyId& other) const { return m_desc != other.m_desc; }

        _Ret_notnull_
        const hkPropertyDesc* desc() const { return m_desc.getPtr(); }

        friend hkUint32 hkHashValue(hkPropertyId id)
        {
            return hkHash::hkHashValue((hkUlong)id.desc());
        }

    private:
        hkPropertyId(_In_ const hkPropertyDesc* desc) : m_desc(desc) {}
        friend struct hkPropertyDesc;
        friend class hkPropertyRegistry;

        enum DescPointerFlags
        {
            DESC_DEFAULT = 0,
            DESC_BUILDING = 1 << 0
        };

        typedef hkPtrAndInt<const hkPropertyDesc, unsigned, 1> PropertyDescPtr;
        PropertyDescPtr m_desc HK_ATTR(hk::Serialize(false));

        struct BuildingData
        {
            HK_DECLARE_CLASS(BuildingData, New);
            const hkReflect::Type* type;
            const char* name;
            hkPropertyFlags::Enum flags;
            bool flagsAreSet;
            bool nameIsSet;
        };

        _Ret_notnull_ BuildingData* buildingData();
        void tryBuildDesc(_Inout_ BuildingData* data);
        void afterReflectNew();

        // Serialization trick: when properties are set, we asume we are being deserialized, and store
        // the properties in BuildingData pointed by the m_desc pointers, then on afterReflectNew we
        // resolve the actual desc
        _Ret_maybenull_ const hkReflect::Type* getType() const
        {
            HK_ASSERT(0x503b8dbf, !(m_desc.getInt() & DESC_BUILDING), "Cannot query the type property while building the desc");
            return m_desc.getPtr() ? m_desc.getPtr()->type : HK_NULL;
        }
        bool setType(_In_ const hkReflect::Type* type);
        HK_PROPERTY(type, getType, setType) HK_ATTR(hk::Serialize);

        _Ret_maybenull_z_
        const char* getName() const
        {
            HK_ASSERT(0x3638806b, !(m_desc.getInt() & DESC_BUILDING), "Cannot query the name property while building the desc");
            return m_desc.getPtr() ? m_desc.getPtr()->name : HK_NULL;
        }

        bool setName(_In_z_ const char* name);
        HK_PROPERTY(name, getName, setName) HK_ATTR(hk::Serialize);

        hkPropertyFlags::Enum getFlags() const
        {
            HK_ASSERT(0x2014c722, !(m_desc.getInt() & DESC_BUILDING), "Cannot query the type property while building the desc");
            return m_desc.getPtr() ? (hkPropertyFlags::Enum)m_desc.getPtr()->flags.get() : hkPropertyFlags::DEFAULT;
        }
        bool setFlags(hkPropertyFlags::Enum type);
        HK_PROPERTY(flags, getFlags, setFlags) HK_ATTR(hk::Serialize);

};

HK_INLINE hkPropertyId hkPropertyDesc::id() const  { return hkPropertyId(this); }

HK_REFLECT_ENUM(HK_EXPORT_COMMON, hkPropertyFlags::Enum);

namespace hkHashMapDetail
{
    template<typename T> struct TolerantIndexRebuild;

    /// Disable warnings on hash index rebuilds in hkHashBase::afterReflectNew as
    /// we generate these in advance in a few cases for property bags to make them
    /// usable before afterReflectNew is called on their owner.
    template<> struct TolerantIndexRebuild<hkPropertyId> : hkTrait::TrueType {};
}

    /// Curiously Recursive Template Pattern for propertybag interface.
    /// Each method has two overloads, one
    /// This class has overloads for properties which are statically declared (hkStaticPropertyDesc<T>)
    /// and dynamically created (hkPropertyId). In general, statically declared properties are preferred
    /// since they give additional compile time checks.
template<typename IMPL>
struct hkPropertyInterface
{
        /// Adds a copy of T uniquely associated with the given id.
    template<typename T>
    hkResult add(const hkStaticPropertyDesc<T>& desc, const typename hkStaticPropertyDesc<T>::Type& value)
    {
        return static_cast<IMPL*>(this)->addChecked(desc.id(), hkReflect::Var(&value));
    }

        /// Adds a property (copied by value) uniquely associated with the given id.
    hkResult add(hkPropertyId id, const hkReflect::Var& value)
    {
        return static_cast<IMPL*>(this)->addChecked(id, value);
    }

        /// Removes the given property associated with the given id.
    template<typename T>
    hkResult remove(const hkStaticPropertyDesc<T>& desc)
    {
        return static_cast<IMPL*>(this)->removeChecked(desc.id());
    }

        /// Removes the given property associated with the given id.
    hkResult remove(hkPropertyId id)
    {
        return static_cast<IMPL*>(this)->removeChecked(id);
    }

    hkReflect::Var getVar(hkPropertyId id)
    {
        const hkReflect::Type* type = id.desc()->type;
        if (void* valAddr = static_cast<const IMPL*>(this)->getValueAddressChecked(type, id))
        {
            return hkReflect::Var(valAddr, type);
        }
        return hkReflect::Var();
    }

        /// Get a pointer to the property or nullptr if not present
    template<typename T>
    _Ret_maybenull_ T* getIfExists(hkPropertyId id)
    {
        if (void* valAddr = static_cast<const IMPL*>(this)->getValueAddressChecked(hkReflect::getType<T>(), id))
        {
            return static_cast<T*>(valAddr);
        }
        return nullptr;
    }

        /// Get a pointer to the property or nullptr if not present
    template<typename T>
    _Ret_maybenull_ T* getIfExists(const hkStaticPropertyDesc<T>& desc)
    {
        return getIfExists<T>(desc.id());
    }

        /// Get a pointer to the property or nullptr if not present
    template<typename T>
    _Ret_maybenull_ const T* getIfExists(hkPropertyId id) const
    {
        return const_cast<hkPropertyInterface*>(this)->getIfExists<T>(id);
    }

        /// Get a pointer to the property or nullptr if not present
    template<typename T>
    _Ret_maybenull_ const T* getIfExists(const hkStaticPropertyDesc<T>& desc) const
    {
        return getIfExists<T>(desc.id());
    }

        /// Get a ref to the property, assert if the property does not exists
    template<typename T>
    T& get(hkPropertyId id)
    {
        T* res = getIfExists<T>(id);
        HK_ASSERT_NO_MSG(0x5cf732e1, res);
        return *res;
    }

        /// Get a ref to the property, assert if the property does not exists
    template<typename T>
    T& get(const hkStaticPropertyDesc<T>& desc)
    {
        return get<T>(desc.id());
    }

        /// Get a const ref to the property, assert if the property does not exists
    template<typename T>
    const T& get(hkPropertyId id) const
    {
        return const_cast<hkPropertyInterface*>(this)->get<T>(id);
    }

        /// Get a ref to the property, assert if the property does not exists
    template<typename T>
    const T& get(const hkStaticPropertyDesc<T>& desc) const
    {
        return get<T>(desc.id());
    }

    
#ifdef HK_COMPILER_CLANG
#   pragma clang diagnostic push
#   pragma clang diagnostic ignored "-Wreturn-stack-address"
#endif

        /// Get a copy of the given property value or a default if not present.
    template<typename T>
    const T& getWithDefault(hkPropertyId id, const T& def = T()) const
    {
        if (void* valAddr = static_cast<const IMPL*>(this)->getValueAddressChecked(hkReflect::getType<T>(), id))
        {
            return *static_cast<T*>(valAddr);
        }
        return def;
    }

#ifdef HK_COMPILER_CLANG
#   pragma clang diagnostic pop
#endif

        /// Get a copy of the given property value or a default if not present.
    template<typename T>
    const T& getWithDefault(const hkStaticPropertyDesc<T>& desc, const T& def = T()) const
    {
        return getWithDefault<T>(desc.id(), def);
    }

        /// Access the value of a property. Creates the property if it doesn't exist.
    template<typename T>
    T& getOrCreate(const hkStaticPropertyDesc<T>& desc, const T& def = T())
    {
        while (1)
        {
            if (void* valAddr = static_cast<const IMPL*>(this)->getValueAddressChecked(hkReflect::getType<T>(), desc.id()))
            {
                return *static_cast<T*>(valAddr);
            }
            else
            {
                HK_ON_DEBUG(hkResult r = )add(desc, def);
                HK_ASSERT_NO_MSG(0x2233ee50, r.isSuccess());
            }
        }
    }

        /// Access the value of a property. Creates the property if it doesn't exist.
        /// If the type is not correct, null may be returned.
    template<typename T>
    T& getOrCreate(hkPropertyId id, const T& def = T())
    {
        while (1)
        {
            if (void* valAddr = static_cast<const IMPL*>(this)->getValueAddressChecked(hkReflect::getType<T>(), id))
            {
                return *static_cast<T*>(valAddr);
            }
            else if( add(id, &def).isSuccess() )
            {
                // loop & read the address
            }
            else
            {
                HK_ASSERT(0x7c6cf815, false, "Incorrect type used for getOrCreate");
            }
        }
    }

        /// Get or insert a pointer property. Additionally, if the pointer is null, initialize it with a new object.
    template<typename OBJ>
    _Ret_notnull_ OBJ* getOrAllocate(const hkStaticPropertyDesc<OBJ*>& desc)
    {
        OBJ*& ptr = getOrCreate(desc);
        if( ptr == nullptr )
        {
            ptr = new OBJ();
        }
        return ptr;
    }

        /// Get the pointer property or allocate a new one. May return null if the type is incorrect.
    template<typename PTR_TYPE>
    PTR_TYPE& getOrAllocate(hkPropertyId id)
    {
        PTR_TYPE& ptr = getOrCreate(id, PTR_TYPE());
        if( ptr == nullptr )
        {
            typedef typename hkTrait::RemovePointer<PTR_TYPE>::type Obj;
            ptr = new Obj();
        }
        return ptr;
    }
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
