// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// this: #include <Common/Base/Types/Geometry/hkGeometry.h>

namespace hkAsyncHeartbeat
{
    class Heartbeat;
}

/// Stores a triangle based mesh.
/// Also see #include <Common/GeometryUtilities/Misc/hkGeometryUtils.h>
struct hkGeometry : public hkReferencedObject
{
public:
    //+version(1)
    HK_DECLARE_CLASS( hkGeometry, New );
    HK_DETAIL_DECLARE_REFLECT_EXPORT( hkGeometry, HK_EXPORT_COMMON );

    /// A hkTriangle contains indices into an array of vertices and a material index.
    struct HK_EXPORT_COMMON Triangle
    {
        HK_DECLARE_CLASS( Triangle, New, Reflect, Pod );

        void set( int a, int b, int c, int material = -1 ) { m_a = a; m_b = b; m_c = c;  m_material = material; }
        int getVertex( int i ) const { return ( &m_a )[i]; }

        int m_a;
        int m_b;
        int m_c;
        int m_material;
    };

public:

    // Constructor
    HK_INLINE hkGeometry() {}

    // Destructor
    HK_INLINE ~hkGeometry() {}

    /// Copy constructor. Required since hkArray's copy constructor is not public.
    HK_EXPORT_COMMON hkGeometry( const hkGeometry& other );

    /// Clear content.
    HK_EXPORT_COMMON void clear();

    /// Check if data is numerically valid.
    HK_EXPORT_COMMON hkBool isValid() const;

    /// Returns a read-only reference to a triangle vertex.
    HK_INLINE const hkVector4& getVertex( int triangleIndex, int vertexIndex ) const { HK_ASSERT( 0xB9C6C634, vertexIndex >= 0 && vertexIndex < 3, "Index out-of-range" ); return m_vertices[( &m_triangles[triangleIndex].m_a )[vertexIndex]]; }

    /// Returns a read-write reference to a triangle vertex.
    HK_INLINE hkVector4& getVertex( int triangleIndex, int vertexIndex ) { HK_ASSERT( 0xB9C6C634, vertexIndex >= 0 && vertexIndex < 3, "Index out-of-range" ); return m_vertices[( &m_triangles[triangleIndex].m_a )[vertexIndex]]; }

    /// Copies the triangle vertices in a preallocated buffer passed as parameter.
    HK_INLINE void getTriangle( int triangleIndex, _Inout_updates_all_( 3 ) hkVector4* HK_RESTRICT vertices ) const;

    /// Append another geometry (vertex welding not performed).
    HK_INLINE hkResult appendGeometry( const hkGeometry& geometry, const hkTransform& transform );

    /// Append another geometry with optional transform (vertex welding not performed).
    HK_EXPORT_COMMON hkResult appendGeometry( const hkGeometry& geometry, _In_opt_ const hkMatrix4* transform = HK_NULL );

    /// Append a vertex and return its index
    HK_EXPORT_COMMON int appendVertex( hkVector4Parameter v );

    /// Append a vertex and return the triangle index
    HK_EXPORT_COMMON int appendTriangle( hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, int material = -1 );

    /// hkAsyncHeartbeat::Heartbeat aware overloads of the above
    /// appendGeometry functions.
    HK_INLINE hkResult appendGeometry( const hkGeometry& geometry, hkAsyncHeartbeat::Heartbeat& heartbeat );
    HK_INLINE hkResult appendGeometry( const hkGeometry& geometry, const hkTransform& transform, hkAsyncHeartbeat::Heartbeat& heartbeat );
    HK_EXPORT_COMMON hkResult appendGeometry( const hkGeometry& geometry, const hkMatrix4* transform, hkAsyncHeartbeat::Heartbeat& heartbeat );

public:

    /// Array of vertices that the triangles can index into.
    hkArray<hkVector4> m_vertices;

    /// Array of triangles. The triangles are quadruples of ints with the first 3 being indices into the m_vertices array.
    hkArray<struct Triangle> m_triangles;
};

#include <Common/Base/Types/Geometry/Geometry/hkGeometry.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
