// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

class hkLocalFrameCollector;
class hkLocalFrameGroup;

    /// An abstract local frame that can be part of a hierarchy of local frames.
class HK_EXPORT_COMMON hkLocalFrame : public hkReferencedObject
{
    public:

        HK_DECLARE_REFLECTION();
        HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE );

            /// Default constructor does nothing.
        hkLocalFrame() {}

            /// Get the transform from the coordinates of this local frame to the coordinates
            /// at the root of the local frame hierarchy.
        void getTransformToRoot( hkTransform& transform ) const;

            /// Get the position of this local frame in the coordinates
            /// at the root of the local frame hierarchy.
        void getPositionInRoot( hkVector4& position ) const;

            /// Return the transform from the coordinates of this frame to
            /// the coordinates of the parent frame.
        virtual void getLocalTransform( hkTransform& transform ) const = 0;

            /// Sets the local space transform
        virtual void setLocalTransform( const hkTransform& transform ) = 0;

            /// Return the position of this frame in the coordinates of the parent frame.
            /// The default implementation calls getLocalTransform() and then pulls the
            /// position from the transform.
        virtual void getLocalPosition( hkVector4& position ) const;

            /// Get all of the frames near a given point.
        virtual void getNearbyFrames( const hkVector4& target, hkReal maxDistance, hkLocalFrameCollector& collector ) const = 0;

            /// Get the name of the character (can be HK_NULL).
        virtual _Ret_z_ const char* getName() const = 0;

            /// Get the local frame that is this frame's parent.
        virtual _Ret_maybenull_ const hkLocalFrame* getParentFrame() const = 0;

            /// Set the local frame that is this frame's parent.
        virtual void setParentFrame( _In_opt_ const hkLocalFrame* parentFrame ) = 0;

            /// Get the number of children of this local frame
        virtual int getNumChildFrames() const = 0;

            /// Get the i'th child of this hkLocalFrame
        virtual _Ret_notnull_ hkLocalFrame* getChildFrame( int i ) const = 0;

            /// Get the group to which this frame belongs.
        virtual _Ret_maybenull_ const hkLocalFrameGroup* getGroup() const = 0;

            /// Set the group to which this frame belongs.
        virtual void setGroup( _In_ const hkLocalFrameGroup* localFrameGroup ) = 0;

            /// Gets all children in the local frame and its descendants.
        virtual void getDescendants( hkArrayBase<const hkLocalFrame*>& descendants, hkMemoryAllocator& a) const;

            /// Gets all children in the local frame and its descendants.
        HK_INLINE void getDescendants(hkArray<const hkLocalFrame*>& descendants ) const
        {
            getDescendants(descendants, hkContainerHeapAllocator().get(&descendants));
        }
};

    /// An abstract class for collecting local frames.
class HK_EXPORT_COMMON hkLocalFrameCollector : public hkReferencedObject
{
    public:

        HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE );

            /// Add a local frame to the collector.
        virtual void addFrame( _In_ const hkLocalFrame* frame, hkReal distance ) = 0;
};

    /// All the local frames with the same group name share a hkLocalFrameGroup.
class HK_EXPORT_COMMON hkLocalFrameGroup : public hkReferencedObject
{
    public:

        HK_DECLARE_REFLECTION();
        HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE );

            /// Default constructor
        hkLocalFrameGroup( _In_opt_z_ const char* name=HK_NULL ) : m_name(name) {}

            /// Get the name of the group (cannot be HK_NULL).
        _Ret_z_ const char* getName() const { return m_name; }

    private:

            /// The name of this group.
        hkStringPtr m_name;
};


    /// A local frame that stores a transform, name, and parent and children pointers.
class HK_EXPORT_COMMON hkSimpleLocalFrame : public hkLocalFrame
{
    public:
        HK_DECLARE_CLASS(hkSimpleLocalFrame, New, Reflect, NonCopyable);
        HK_RECORD_ATTR( hk::Version(1) );

            /// Default constructor does nothing.
        hkSimpleLocalFrame() : m_parentFrame(HK_NULL), m_group(HK_NULL), m_name(HK_NULL) {}

            /// Destructor removes references to the children.
        ~hkSimpleLocalFrame();

            // hkLocalFrame implementation
        virtual void getLocalTransform( hkTransform& transform ) const;

            /// Sets the local space transform
        virtual void setLocalTransform( const hkTransform& transform );

            // hkLocalFrame implementation
        virtual void getLocalPosition( hkVector4& position ) const;

            // hkLocalFrame implementation
        virtual void getNearbyFrames( const hkVector4& target, hkReal maxDistance, hkLocalFrameCollector& collector ) const;

            // hkLocalFrame implementation
        virtual _Ret_z_ const char* getName() const { return m_name; }

            // hkLocalFrame implementation
        virtual _Ret_maybenull_ const hkLocalFrame* getParentFrame() const { return m_parentFrame; }

            // hkLocalFrame implementation
        virtual void setParentFrame(_In_opt_ const hkLocalFrame* parentFrame) { m_parentFrame = parentFrame; }

            // hkLocalFrame implementation
        virtual int getNumChildFrames() const;

            // hkLocalFrame implementation
        virtual _Ret_notnull_ hkLocalFrame* getChildFrame( int i ) const;

            // hkLocalFrame implementation
        virtual _Ret_maybenull_ const hkLocalFrameGroup* getGroup() const { return m_group; }

            // hkLocalFrame implementation
        virtual void setGroup(_In_opt_ const hkLocalFrameGroup* group );

            /// Transforms points from this coordinate frame to the coordinate system of the parent frame.
        hkTransform m_transform;

            /// The children of this frame. This frame owns the children, so a reference will be removed
            /// when this frame is destroyed. If you build one of these objects in code, make sure
            /// to add a reference to each child.
        typedef class hkLocalFrame* ChildPointer HK_ATTR(hk::NonNull);
        hkArray<ChildPointer> m_children;

            /// The parent frame. This frame does not own the parent, so we don't do any reference counting.
        hkViewPtr<const hkLocalFrame> m_parentFrame;

            /// The group that this local frame belongs to (can be HK_NULL).
        const hkLocalFrameGroup* m_group;

            /// The name of this frame.
        hkStringPtr m_name;
};

HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkSimpleLocalFrame::ChildPointer, hkSimpleLocalFrame_ChildPointer_Tag);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
