// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <Common/Base/hkBaseHeartbeat.h>
#include <Common/Base/Algorithm/Heartbeat/hkHeartbeatAlgos.h>

/// Copy constructor. Required since hkArray's copy constructor is not public.
hkGeometry::hkGeometry (const hkGeometry& other)
: hkReferencedObject(other)
{
    m_vertices = other.m_vertices;
    m_triangles = other.m_triangles;
}

/// Clear content.
void hkGeometry::clear()
{
    m_triangles.clear();
    m_vertices.clear();
}

/// Check if data is numerically valid
hkBool hkGeometry::isValid() const
{
    // Vertices
    const unsigned int numVertices = m_vertices.getSize();
    for (unsigned int i = 0; i < numVertices; ++i)
    {
        if (!m_vertices[i].isOk<3>())
        {
            return false;
        }
    }

    // Check that the triangle indices are in the range [0, numVertices - 1]
    for (int i = 0; i < m_triangles.getSize(); ++i)
    {
        const Triangle& triangle = m_triangles[i];
        if (static_cast<unsigned int>(triangle.m_a) >= numVertices ||
            static_cast<unsigned int>(triangle.m_b) >= numVertices ||
            static_cast<unsigned int>(triangle.m_c) >= numVertices)
        {
            return false;
        }
    }
    return true;
}

hkResult hkGeometry::appendGeometry(const hkGeometry& geometry, _In_opt_ const hkMatrix4* transform)
{
    const int baseVertex = m_vertices.getSize();
    const int baseTriangle = m_triangles.getSize();

    hkResult vertRes = m_vertices.reserve( baseVertex + geometry.m_vertices.getSize() );
    if (vertRes.isFailure())
        return HK_FAILURE;
    hkResult triRes = m_triangles.reserve( baseTriangle + geometry.m_triangles.getSize() );
    if (triRes.isFailure())
        return HK_FAILURE;

    m_vertices.append(geometry.m_vertices);
    m_triangles.append(geometry.m_triangles);

    if(transform)
    {
        for (int i = baseVertex; i < m_vertices.getSize(); ++i)
        {
            hkVector4   v; transform->transformPosition(m_vertices[i], v);
            m_vertices[i].setXYZ(v);
        }
    }

    if(baseVertex > 0)
    {
        for (int i = baseTriangle; i < m_triangles.getSize(); ++i)
        {
            m_triangles[i].m_a  +=  baseVertex;
            m_triangles[i].m_b  +=  baseVertex;
            m_triangles[i].m_c  +=  baseVertex;
        }
    }

    return HK_SUCCESS;
}

hkResult hkGeometry::appendGeometry(const hkGeometry& geometry, const hkMatrix4* transform, hkAsyncHeartbeat::Heartbeat& heartbeat)
{
    hkAsyncHeartbeat::Scope heartbeatScope(heartbeat, 0xe525bdd0,
        hkAsyncHeartbeat::HeartbeatFile::BASE, "hkGeometry::appendGeometry");

    const int baseVertex = m_vertices.getSize();
    const int baseTriangle = m_triangles.getSize();

    HK_RETURN_IF_FAILED( m_vertices.reserve( baseVertex + geometry.m_vertices.getSize() ) );
    HK_RETURN_IF_FAILED( m_triangles.reserve( baseTriangle + geometry.m_triangles.getSize() ) );

    heartbeat.tick<0x83941c6c>();
    if(transform)
    {
        hkVector4* newVerts = m_vertices.expandByUnchecked(geometry.m_vertices.getSize());
        heartbeat.forRange<0x4ee1bded>(0, geometry.m_vertices.getSize(),
            [&](int i)
            {
                transform->transformPosition(geometry.m_vertices[i], newVerts[i]);
            });
    }
    else
    {
        hkVector4* newVerts = m_vertices.expandByUnchecked(geometry.m_vertices.getSize());
        heartbeat.tick<0x39404477>();
        hkAlgorithm::memCpy(newVerts, geometry.m_vertices.begin(), geometry.m_vertices.getSize() * sizeof(hkVector4), heartbeat);
    }

    if(baseVertex > 0)
    {
        Triangle* newTriangles = m_triangles.expandByUnchecked(geometry.m_triangles.getSize());
        heartbeat.forRange<0x7d9ee2ec>(0, geometry.m_triangles.getSize(),
            [&](int i)
            {
                newTriangles[i].m_a = geometry.m_triangles[i].m_a + baseVertex;
                newTriangles[i].m_b = geometry.m_triangles[i].m_b + baseVertex;
                newTriangles[i].m_c = geometry.m_triangles[i].m_c + baseVertex;
            });
    }
    else
    {
        Triangle* newTriangles = m_triangles.expandByUnchecked(geometry.m_triangles.getSize());
        heartbeat.tick<0x5ddacd3e>();
        hkAlgorithm::memCpy(newTriangles, geometry.m_triangles.begin(), geometry.m_triangles.getSize() * sizeof(Triangle), heartbeat);
    }

    return HK_SUCCESS;
}

/// Append a vertex and return its index
int hkGeometry::appendVertex( hkVector4Parameter v )
{
    int a = m_vertices.getSize();
    m_vertices.pushBack(v);
    return a;
}

/// Append a vertex and return the triangle index
int hkGeometry::appendTriangle( hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, int material  )
{
    int tIndex = m_triangles.getSize();
    Triangle& t = m_triangles.expandOne();
    t.m_a = appendVertex( a );
    t.m_b = appendVertex( b );
    t.m_c = appendVertex( c );
    t.m_material = material;

    return tIndex;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
