// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Types/Geometry/Frustum/hkFrustum.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>

//
//  Constructor

hkFrustum::hkFrustum()
{
    // Initialize to a valid frustum, so that when added to the UI we get to see something
    const hkReal fovX = 0.25f * HK_REAL_PI;
    const hkReal cosX = hkMath::cos(fovX);
    const hkReal sinX = hkMath::sin(fovX);

    const hkReal fovY = 0.25f * HK_REAL_PI;
    const hkReal cosY = hkMath::cos(fovY);
    const hkReal sinY = hkMath::sin(fovY);

    const hkReal nearX  = 0.5f;
    const hkReal farX   = 1.0f;

    // Set-up a 45 degrees frustum along the X axis
    m_planes[LEFT_PLANE].   m_equation.set(-sinX, 0.0f, -cosX);
    m_planes[RIGHT_PLANE].  m_equation.set(-sinX, 0.0f, +cosX);
    m_planes[TOP_PLANE].    m_equation.set(-sinY, +cosY, 0.0f);
    m_planes[BOTTOM_PLANE]. m_equation.set(-sinY, -cosY, 0.0f);
    m_planes[NEAR_PLANE].   m_equation.set(-1.0f, 0.0f, 0.0f, nearX);
    m_planes[FAR_PLANE].    m_equation.set(+1.0f, 0.0f, 0.0f, -farX);
}

void hkFrustum::setFromViewProjection( const hkMatrix4& viewProjection, hkMatrix4CameraUtil::ClipSpaceDepthRange::Enum depthRange )
{
    // Transpose view projection since we want to add rows
    hkMatrix4 transpViewProjection;
    transpViewProjection.setTranspose(viewProjection);

    hkVector4 temp;

    temp = transpViewProjection.getColumn<2>();
    if ( depthRange == hkMatrix4CameraUtil::ClipSpaceDepthRange::MinusOneToOne )
    {
        temp.add(transpViewProjection.getColumn<3>());
    }

    m_planes[NEAR_PLANE].   m_equation = temp;
    m_planes[FAR_PLANE].    m_equation.setSub(transpViewProjection.getColumn<3>(), transpViewProjection.getColumn<2>());
    m_planes[LEFT_PLANE].   m_equation.setAdd(transpViewProjection.getColumn<3>(), transpViewProjection.getColumn<0>());
    m_planes[RIGHT_PLANE].  m_equation.setSub(transpViewProjection.getColumn<3>(), transpViewProjection.getColumn<0>());
    m_planes[BOTTOM_PLANE]. m_equation.setAdd(transpViewProjection.getColumn<3>(), transpViewProjection.getColumn<1>());
    m_planes[TOP_PLANE].    m_equation.setSub(transpViewProjection.getColumn<3>(), transpViewProjection.getColumn<1>());

    // Normalize planes
    for (int plane = 0; plane < ENUM_COUNT; plane++)
    {
        m_planes[plane].normalize();
    }
}

bool hkFrustum::contains( hkVector4Parameter point ) const
{
    for (hkUint32 plane = 0; plane < ENUM_COUNT; ++plane)
    {
        if (m_planes[plane].planePointDistance( point ).isLessZero())
        {
            // Outside
            return false;
        }
    }

    // Inside
    return true;
}

//
//  Computes a point of the frustum

void hkFrustum::computePoint(Points pointIn, hkVector4& pointOut) const
{
    const Planes planeA = Planes((pointIn >> 8) & 0xF);
    const Planes planeB = Planes((pointIn >> 4) & 0xF);
    const Planes planeC = Planes(pointIn & 0xF);

    hkPlane::intersectThreePlanes(m_planes[planeA], m_planes[planeB], m_planes[planeC], pointOut);
}

//
//  Calculate 8 intersection points for the frustum's planes

void hkFrustum::calculatePlaneIntersectionPoints(hkVector4 (&intersectionPoints)[NUM_POINTS]) const
{
    HK_ASSERT_NO_MSG(0xf8839ad, intersectionPoints != HK_NULL );

    computePoint(NEAR_TOP_LEFT_POINT, intersectionPoints[0]);
    computePoint(NEAR_BOTTOM_LEFT_POINT, intersectionPoints[1]);
    computePoint(NEAR_BOTTOM_RIGHT_POINT, intersectionPoints[2]);
    computePoint(NEAR_TOP_RIGHT_POINT, intersectionPoints[3]);

    computePoint(FAR_TOP_LEFT_POINT, intersectionPoints[4]);
    computePoint(FAR_BOTTOM_LEFT_POINT, intersectionPoints[5]);
    computePoint(FAR_BOTTOM_RIGHT_POINT, intersectionPoints[6]);
    computePoint(FAR_TOP_RIGHT_POINT, intersectionPoints[7]);
}

//
//  Returns the geometry of the frustum

void hkFrustum::computeGeometry(hkGeometry& geomOut) const
{
    geomOut.m_vertices.setSize(NUM_POINTS);
    geomOut.m_triangles.setSize(12);

    calculatePlaneIntersectionPoints(reinterpret_cast<hkVector4(&)[NUM_POINTS]>(geomOut.m_vertices[0]));

    // Add triangles
    hkGeometry::Triangle* tris = geomOut.m_triangles.begin();
    tris[0].set(0, 1, 3);   tris[1].set(3, 1, 2);
    tris[2].set(3, 2, 7);   tris[3].set(2, 6, 7);
    tris[4].set(5, 4, 7);   tris[5].set(5, 7, 6);
    tris[6].set(1, 0, 4);   tris[7].set(1, 4, 5);
    tris[8].set(0, 3, 4);   tris[9].set(3, 7, 4);
    tris[10].set(1, 5, 2);  tris[11].set(2, 5, 6);
}

//
//  Returns the horizontal FOV angle (in radians)

hkReal hkFrustum::computeHorizontalFov() const
{
    const hkVector4 nL      = m_planes[LEFT_PLANE].m_equation;
    const hkVector4 nR      = m_planes[RIGHT_PLANE].m_equation;
    const hkReal cosLR      = nL.dot<3>(nR).getReal();
    const hkReal angleLR    = hkMath::acos(cosLR);

    return HK_REAL_PI - angleLR;
}

//
//  Returns the vertical FOV angle (in radians)

hkReal hkFrustum::computeVerticalFov() const
{
    const hkVector4 nT      = m_planes[TOP_PLANE].m_equation;
    const hkVector4 nB      = m_planes[BOTTOM_PLANE].m_equation;
    const hkReal cosTB      = nT.dot<3>(nB).getReal();
    const hkReal angleTB    = hkMath::acos(cosTB);

    return HK_REAL_PI - angleTB;
}

//
//  Utility functions

namespace
{
    //
    //  Find the bisector plane

    static void HK_CALL computeBisector(hkVector4Parameter planeA, hkVector4Parameter planeB, hkVector4& bisectorOut)
    {
        // We've two choices planeA = planeB or planeA = -planeB
        hkVector4 b0;   b0.setAdd(planeA, planeB);
        hkVector4 b1;   b1.setSub(planeA, planeB);

        if ( b0.dot<3>(planeA).isLessZero() )
        {
            b0.setNeg<4>(b0);
        }
        if ( b1.dot<3>(planeA).isLessZero() )
        {
            b1.setNeg<4>(b1);
        }

        const hkSimdReal dotB0 = b0.dot<3>(planeB);
        const hkSimdReal dotB1 = b1.dot<3>(planeB);
        if ( dotB0.isGreaterZero() )
        {
            HK_ASSERT_NO_MSG(0x6bed87f0, dotB1.isLessEqualZero());
            bisectorOut = b1;
            return;
        }

        bisectorOut = b0;
    }
}

//
//  Computes the axis of the frustum

void hkFrustum::computeAxis(hkVector4& axisOut) const
{
    // Get the plane equations
    const hkVector4& pL = m_planes[LEFT_PLANE].m_equation;
    const hkVector4& pR = m_planes[RIGHT_PLANE].m_equation;
    const hkVector4& pT = m_planes[TOP_PLANE].m_equation;
    const hkVector4& pB = m_planes[BOTTOM_PLANE].m_equation;

    // Compute the LR / TB bisector planes
    hkVector4 planeLR;  computeBisector(pL, pR, planeLR);
    hkVector4 planeTB;  computeBisector(pT, pB, planeTB);

    // Intersect the two bisectors
    axisOut.setCross(planeLR, planeTB);
    axisOut.setNormalizedEnsureUnitLength<3, HK_ACC_FULL>(axisOut);

    // Make sure the axis has the proper orientation
    hkVector4 dots;
    hkVector4Util::dot3_1vs4(axisOut, pL, pR, pT, pB, dots);

    // We want all dots to be <= 0 so that the axis is inside the frustum
    const hkVector4Comparison cmp = dots.lessEqualZero();
    const int numCorrect = hkMath::countBitsSet(cmp.getMask());
    if ( numCorrect < 3 )
    {
        axisOut.setNeg<4>(axisOut);
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
