// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// this: #include <Common/Base/Types/Geometry/Connectivity/hkConnectivity.h>

#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>

/// Geometry connectivity.
struct hkConnectivity
{
    /// Build flags.
    enum BuildFlags
    {
        /// If set, build triangle patches.
        BUILD_TRIANGLE_PATCHES = 1,

        /// If set, force the build to be single-threaded.
        BUILD_SINGLE_THREADED = 2,

        /// Build all.
        BUILD_ALL = BUILD_TRIANGLE_PATCHES,

        /// Default build flags.
        BUILD_DEFAULT = 0
    };

    /// Triangle flags.
    enum TriangleFlags
    {
        TRIANGLE_IS_INVALID = 1 << 0,
        NUM_TRIANGLE_FLAGS = 1
    };

    /// Vertex.
    struct Vertex
    {
    private:

        /// Number of triangles referencing this vertex, or, equivalently, number of edge starting from this vertex.
        int m_cardinality : 24;

        /// true if the vertex is on the boundary, false otherwise.
        /// Note: if true the first edge of the ring is naked.
        /// Conditions: number of naked edges in the 1-ring is greater than 0.
        bool m_boundary : 1;

        /// true if the vertex is on the border, false otherwise.
        /// Note: if true the first edge of the ring is naked.
        /// Conditions: number of naked edges in the 1-ring is equal to 1.
        bool m_border : 1;

        /// true is the vertex 1-ring is manifold.
        /// Conditions: number of naked edges in the 1-ring is less than 2 and cardinality is greater than 0.
        bool m_manifold : 1;

        /// Index of the first edge.
        int m_firstEdge;

        friend struct hkConnectivity;
    };

    /// (Half) Edge.
    struct Edge
    {
        HK_INLINE Edge() {}
        HK_INLINE Edge( int triangle, int start ) : m_triangle( triangle ), m_start( start ) {}
        HK_INLINE bool isValid() const { return uid() != 0xffffffff; }
        HK_INLINE int triangle() const { return int( m_triangle ); }
        HK_INLINE int start() const { return int( m_start ); }
        HK_INLINE hkUint32& uid() { return *reinterpret_cast<hkUint32*>( this ); }
        HK_INLINE hkUint32 uid() const { return *reinterpret_cast<const hkUint32*>( this ); }
        static HK_INLINE Edge invalid() { Edge e; e.uid() = 0xffffffff; return e; }

    private:

        hkUint32 m_triangle : 30; ///< Triangle index.
        hkUint32 m_start : 2; ///< Starting vertex index.
    };

    /// Triangle.
    struct Triangle
    {
        HK_INLINE Edge& operator[]( int index ) { HK_ASSERT( 0x474A73AF, index >= 0 && index < 3, "Index out-of-range" ); return m_links[ index ]; }
        HK_INLINE Edge operator[]( int index ) const { HK_ASSERT( 0x474A73AF, index >= 0 && index < 3, "Index out-of-range" ); return m_links[ index ]; }

    private:

        Edge m_links[ 3 ]; ///< Linked edges.
        hkUint32 m_flags; ///< Triangle flags.

        friend struct hkConnectivity;
    };

    /// Ring triangle.
    struct RingTriangle
    {
        HK_INLINE RingTriangle() {}
        HK_INLINE RingTriangle( int t, int r ) : m_triangle( hkUint32( t ) ), m_ring( hkUint32( r ) ) {}
        int triangle() const { return int( m_triangle ); }
        int ring() const { return int( m_ring ); }

    private:

        hkUint32 m_triangle : 30; ///< Triangle index.
        hkUint32 m_ring : 2; ///< Ring that triangle belongs too.
    };

    /// Edge data pair.
    template <typename DATA>
    struct EdgeData
    {
        DATA m_data;
        Edge m_edge;
    };

    /// Quad collector.
    struct QuadCollector
    {
        virtual ~QuadCollector() {}

        /// Emit a quad represented by its spine.
        /// Note: if the method returns false, the quad is ignored and will be emitted as a triangle later on.
        virtual bool emitQuad( Edge edge ) = 0;

        /// Emit a triangle represented by one of its edge.
        virtual void emitTriangle( Edge edge ) = 0;
    };

    /// Triangle patch.
    typedef hkArrayView<const RingTriangle> TrianglePatch;

    /// Vertex ring.
    typedef hkArrayView<const Edge> VertexRing;

    /// Ctor.
    HK_INLINE hkConnectivity() { clear(); }

    /// Clear connectivity.
    HK_EXPORT_COMMON void clear();

    /// Build connectivity from hkGeometry.
    HK_EXPORT_COMMON void build( const hkGeometry& geometry, int flags = BUILD_DEFAULT );

    /// Return the flags used to build the connectivity or 0 is connectivity was not built.
    HK_INLINE int getBuildFlags() const { return m_buildFlags; }

    /// Returns the number of vertices.
    HK_INLINE int getNumVertices() const { return m_vertices.getSize(); }

    /// Returns a given vertex.
    HK_INLINE const Vertex& getVertex( int v ) const { return m_vertices[ v ]; }

    /// Returns an array of all vertices.
    HK_INLINE const hkArray<Vertex>& getAllVertices() const { return m_vertices; }

    /// Returns the number of edges.
    HK_INLINE int getNumEdges() const { return m_edges.getSize(); }

    /// Returns a given edge.
    HK_INLINE Edge getEdge( int edgeIndex ) const { return m_edges[ edgeIndex ]; }

    /// Returns an array of all edges.
    HK_INLINE const hkArray<Edge>& getAllEdges() const { return m_edges; }

    /// Returns the number of triangles.
    HK_INLINE int getNumTriangles() const { return m_triangles.getSize(); }

    /// Returns a given triangle.
    HK_INLINE const Triangle& getTriangle( int triangleIndex ) const { return m_triangles[ triangleIndex ]; }

    /// Return the triangle links.
    HK_INLINE void getTriangleLinks( int triangleIndex, int* linksOut, int unboundTriangleIndexValue = -1 ) const;

    /// Returns an array of all triangles.
    HK_INLINE const hkArray<Triangle>& getAllTriangles() const { return m_triangles; }

    /// Get start vertex index from edge.
    HK_INLINE int getStartVertexIndex( const hkGeometry& g, Edge e ) const { return ( &g.m_triangles[ e.triangle() ].m_a )[ e.start() ]; }

    /// Get end vertex index from edge.
    HK_INLINE int getEndVertexIndex( const hkGeometry& g, Edge e ) const { return ( &g.m_triangles[ e.triangle() ].m_a )[ ( e.start() + 1 ) % 3 ]; }

    /// Get end vertex index from edge.
    HK_INLINE int getApexVertexIndex( const hkGeometry& g, Edge e ) const { return ( &g.m_triangles[ e.triangle() ].m_a )[ ( e.start() + 2 ) % 3 ]; }

    /// Get the vertex indices of the start and the end of a given edge.
    HK_INLINE void getEndpointsVertexIndices( const hkGeometry& g, Edge e, int* indices ) const { indices[ 0 ] = getStartVertexIndex( g, e ); indices[ 1 ] = getEndVertexIndex( g, e ); }

    /// Get vertex indices from edge in the following order:
    /// 0 = start vertex.
    /// 1 = end vertex.
    /// 2 = apex vertex.
    HK_EXPORT_COMMON void getVertexIndices(const hkGeometry& g, Edge e, int* HK_RESTRICT indices) const;

    /// Get the maximum possible edge UID.
    HK_INLINE int getMaxEdgeUID() const { return getNumTriangles() * 4; }

    /// Get the opposite edge.
    HK_INLINE Edge getLink( Edge e ) const { return e.isValid() ? m_triangles[ e.triangle() ][ e.start() ] : e; }

    /// Get the master edge.
    HK_INLINE Edge getMaster( Edge e ) const { return isMaster( e ) ? e : getLink( e ); }

    /// Get the next edge.
    HK_INLINE Edge getNext( Edge e ) const { return e.isValid() ? Edge( e.triangle(), ( e.start() + 1 ) % 3 ) : e; }

    /// Get the previous edge.
    HK_INLINE Edge getPrev( Edge e ) const { return e.isValid() ? Edge( e.triangle(), ( e.start() + 2 ) % 3 ) : e; }

    /// Returns true if the edge is bound, false otherwise.
    HK_INLINE bool isBound( Edge e ) const { return getLink( e ).isValid(); }

    /// Returns true if the edge is naked, false otherwise.
    HK_INLINE bool isNaked( Edge e ) const { return !isBound( e ); }

    /// Returns true if the edge is a master edge.
    HK_INLINE bool isMaster( Edge e ) const { return !e.isValid() || isNaked( e ) || e.uid() < getLink( e ).uid(); }

    /// Returns true if the vertex is on the boundary, false otherwise.
    HK_INLINE bool isBoundaryVertex( int vertexIndex ) const { return m_vertices[ vertexIndex ].m_boundary; }

    /// Returns true if the vertex ring is on the border.
    HK_INLINE bool isBorderVertex( int vertexIndex ) const { return m_vertices[ vertexIndex ].m_border; }

    /// Returns true if the vertex is not referenced by any valid triangle, false otherwise.
    HK_INLINE bool isOrphanVertex( int vertexIndex ) const { return m_vertices[ vertexIndex ].m_cardinality == 0; }

    /// Returns true if the vertex ring is manifold.
    HK_INLINE bool isManifoldVertex( int vertexIndex ) const { return m_vertices[ vertexIndex ].m_manifold; }

    /// Return the cardinality of a vertex.
    HK_INLINE int getVertexCardinality( int vertexIndex ) const { return m_vertices[ vertexIndex ].m_cardinality; }

    /// Get a vertex ring from its index.
    HK_INLINE VertexRing getVertexRing( int vertexIndex ) const { const Vertex& v = m_vertices[ vertexIndex ]; return VertexRing( &m_edges[ v.m_firstEdge ], v.m_cardinality ); }

    /// Get a triangle patch from its index.
    /// Note: The patch include the triangle itself in the first position.
    HK_INLINE TrianglePatch getTrianglePatch( int triangleIndex ) const { const auto* p = &m_trianglePatches[ m_trianglePatches[ triangleIndex ].triangle() ]; return TrianglePatch( p + 1, p->triangle() ); }

    /// Returns true if the triangle is not bound to any other triangles.
    HK_INLINE bool isOrphanTriangle( int triangleIndex ) const { const Triangle& t = m_triangles[ triangleIndex ]; return isNaked( t[ 0 ] ) && isNaked( t[ 1 ] ) && isNaked( t[ 2 ] ); }

    /// Returns true if the triangle is valid.
    HK_INLINE bool isValidTriangle( int triangleIndex ) const { return 0 == ( m_triangles[ triangleIndex ].m_flags & TRIANGLE_IS_INVALID ); }

    /// Mark a triangle as invalid.
    HK_INLINE void markTriangleAsInvalid( int triangleIndex ) { m_triangles[ triangleIndex ].m_flags |= TRIANGLE_IS_INVALID; }

    /// Compute vertex angle weighted normal.
    HK_INLINE hkVector4 computeVertexAngleWeightedNormal( const hkGeometry& geometry, int vertexIndex ) const;

    /// Emit an optimized set of quads and triangles.
    /// Note: the collector is allowed to modify the geometry.
    HK_EXPORT_COMMON void enumerateQuadDominantGeometry( const hkGeometry& geometry, QuadCollector* collector ) const;

    /// Volume conserving regularization for geometry generated by marching-cube like method.
    HK_EXPORT_COMMON void regularize( hkGeometry& geometry, int iterations = 16, hkReal maxAngle = HK_REAL_PI, bool conserveVolume = true, bool protectBoundaries = false ) const;

protected:

    /// Build flags.
    int m_buildFlags;

    /// Vertices.
    hkArray<Vertex> m_vertices;

    /// Edges.
    hkArray<Edge> m_edges;

    /// Triangles.
    hkArray<Triangle> m_triangles;

    /// Map from triangle index to patch, see getTrianglePatch.
    /// Optional, see Flags.
    hkArray<RingTriangle> m_trianglePatches;
};

#include <Common/Base/Types/Geometry/Connectivity/hkConnectivity.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
