// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Types/Geometry/Connectivity/hkConnectivity.h>
#include <Common/Base/Math/Vector/hkIntVector.h>
#include <Common/Base/Math/Vector/hkVector4UtilInternal.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Thread/Concurrency/hkConcurrency.h>
#include <Common/Base/Container/BitField/hkBitField.h>

#include <Geometry/Collide/Algorithms/Triangle/hkcdTriangleUtil.h>

namespace
{
    void computeTriangleCentroids( const hkGeometry& geometry, hkArray<hkVector4>& triangleCentroidsOut )
    {
        triangleCentroidsOut.setSize( geometry.m_triangles.getSize() );

        hkConcurrency::parallelFor( triangleCentroidsOut.getSize(), [&geometry, &triangleCentroidsOut]( int triangleIndex )
        {
            hkVector4 vertices[ 3 ]; geometry.getTriangle( triangleIndex, vertices );
            triangleCentroidsOut[ triangleIndex ] = ( vertices[ 0 ] + vertices[ 1 ] + vertices[ 2 ] ) * hkSimdReal_Inv3;
        } );
    }
}

void hkConnectivity::clear()
{
    m_buildFlags = 0;
    m_vertices.clear();
    m_edges.clear();
    m_triangles.clear();
    m_trianglePatches.clear();
}

void hkConnectivity::build( const hkGeometry& geometry, int flags )
{
    clear();

    HK_COMPILE_TIME_ASSERT( sizeof( Edge ) == sizeof( hkUint32 ) );
    HK_COMPILE_TIME_ASSERT( sizeof( RingTriangle ) == sizeof( hkUint32 ) );

    m_buildFlags = flags;

    const bool singleThreaded = flags & BUILD_SINGLE_THREADED ? true : false;
    const int numTriangles = geometry.m_triangles.getSize();
    const int numVertices = geometry.m_vertices.getSize();
    int numEdges = 0;

    HK_ASSERT( 0x062E71DA, numTriangles <= ( 1 << 30 ), "Too many triangles." );

    {
        // Clear vertices.
        Vertex nullVertex;
        hkString::memSet( &nullVertex, 0, sizeof( Vertex ) );
        m_vertices.setSize( numVertices, nullVertex );

        // Clear triangles.
        Triangle nullTriangle;
        nullTriangle[ 0 ] = Edge::invalid();
        nullTriangle[ 1 ] = Edge::invalid();
        nullTriangle[ 2 ] = Edge::invalid();
        nullTriangle.m_flags = 0;
        m_triangles.reserveExactly( numTriangles );
        m_triangles.setSize( numTriangles, nullTriangle );

        // Compute cardinality and triangle flags.
        for ( int triangleIndex = 0; triangleIndex < numTriangles; ++triangleIndex )
        {
            const hkGeometry::Triangle& triangle = geometry.m_triangles[ triangleIndex ];
            const bool valid = triangle.m_a != triangle.m_b && triangle.m_b != triangle.m_c &&triangle.m_c != triangle.m_a;
            if ( valid )
            {
                m_vertices[ triangle.m_a ].m_cardinality++;
                m_vertices[ triangle.m_b ].m_cardinality++;
                m_vertices[ triangle.m_c ].m_cardinality++;
            }
            else
            {
                m_triangles[ triangleIndex ].m_flags = TRIANGLE_IS_INVALID;
            }
        }

        // Compute vertex first edge index.
        for ( int vertexIndex = 0; vertexIndex < numVertices; ++vertexIndex )
        {
            const int cardinality = m_vertices[ vertexIndex ].m_cardinality;
            m_vertices[ vertexIndex ].m_firstEdge = cardinality ? numEdges : 0;
            numEdges += cardinality;
        }
    }

    {
        // Compute edges and triangles links.
        hkArray<int> counters( numVertices, 0 );

        m_edges.setSize( numEdges );

        for ( int triangleIndex = 0; triangleIndex < numTriangles; ++triangleIndex )
        {
            if ( 0 == ( m_triangles[ triangleIndex ].m_flags & TRIANGLE_IS_INVALID ) )
            {
                const hkGeometry::Triangle& triangle = geometry.m_triangles[ triangleIndex ];
                const int* vertices = &triangle.m_a;
                for ( int i = 2, j = 0; j < 3; i = j++ )
                {
                    const int vertexI = vertices[ i ];
                    const int thisEdgeIndex = m_vertices[ vertexI ].m_firstEdge + counters[ vertexI ]++;
                    m_edges[ thisEdgeIndex ] = Edge( triangleIndex, i );

                    const int vertexJ = vertices[ j ];
                    const Vertex& other = m_vertices[ vertexJ ];
                    const int count = counters[ vertexJ ];
                    for ( int k = 0; k < count; ++k )
                    {
                        const Edge& edge = m_edges[ other.m_firstEdge + k ];
                        const int endVertex = ( &geometry.m_triangles[ edge.triangle() ].m_a )[ ( edge.start() + 1 ) % 3 ];
                        if ( endVertex == vertexI )
                        {
                            m_triangles[ triangleIndex ][ i ] = edge;
                            m_triangles[ edge.triangle() ][ edge.start() ] = m_edges[ thisEdgeIndex ];
                            break;
                        }
                    }
                }
            }
        }

        // Compute vertices attributes.
        hkConcurrency::parallelFor( m_vertices.getSize(), singleThreaded, [this]( int vertexIndex )
        {
            int nakedEdgeIndex = -1;
            int numNakedEdge = 0;
            const VertexRing ring = getVertexRing( vertexIndex );
            for ( int edgeIndex = 0; edgeIndex < ring.getSize(); ++edgeIndex )
            {
                if ( isNaked( ring[ edgeIndex ] ) )
                {
                    nakedEdgeIndex = edgeIndex;
                    numNakedEdge++;
                }
            }

            Vertex& vertex = m_vertices[ vertexIndex ];
            vertex.m_manifold = numNakedEdge < 2 && vertex.m_cardinality > 0;
            vertex.m_boundary = numNakedEdge > 0;
            vertex.m_border = numNakedEdge == 1 && vertex.m_manifold;

            // Make sure that is there a naked edge, it appear first.
            if ( nakedEdgeIndex > 0 )
            {
                hkMath::swap( m_edges[ vertex.m_firstEdge ], m_edges[ vertex.m_firstEdge + nakedEdgeIndex ] );
            }

            // Order ring as fan.
            if ( vertex.m_manifold )
            {
                const int count = vertex.m_cardinality;
                Edge* edges = &m_edges[ vertex.m_firstEdge ];
                for ( int i = 0; i < count - 1; ++i )
                {
                    const Edge prev = getPrev( edges[ i ] );
                    if ( isBound( prev ) )
                    {
                        const int triangle = getLink( prev ).triangle();
                        if ( edges[ i + 1 ].triangle() != triangle )
                        {
                            bool found = false;
                            for ( int j = i + 2; j < count; ++j )
                            {
                                if ( edges[ j ].triangle() == triangle )
                                {
                                    hkMath::swap( edges[ i + 1 ], edges[ j ] );
                                    found = true;
                                    break;
                                }
                            }
                            if ( !found )
                            {
                                vertex.m_manifold = false;
                                vertex.m_border = false;
                                break;
                            }
                        }
                    }
                }
                if ( vertex.m_manifold )
                {
                    const Edge lastEdge = getPrev( edges[ count - 1 ] );
                    if ( vertex.m_border )
                    {
                        if ( isBound( lastEdge ) )
                        {
                            vertex.m_manifold = false;
                            vertex.m_border = false;
                        }
                    }
                    else
                    {
                        if ( isNaked( lastEdge ) || getLink( lastEdge ).triangle() != edges[ 0 ].triangle() )
                        {
                            vertex.m_manifold = false;
                        }
                    }
                }
            }
        } );

        // Compute triangle's patch.
        if ( flags & BUILD_TRIANGLE_PATCHES )
        {
            m_trianglePatches.reserve( numTriangles * 16 );
            m_trianglePatches.setSize( numTriangles, RingTriangle( 0, 0 ) );

            for ( int triangleIndex = 0; triangleIndex < numTriangles; triangleIndex++ )
            {
                // Build triangle ring.
                hkInplaceArray<RingTriangle, 64> neighborhood;
                const Triangle& triangle = m_triangles[ triangleIndex ];
                if ( 0 == ( triangle.m_flags & TRIANGLE_IS_INVALID ) )
                {
                    hkIntVector triangleRing; triangleRing.set( triangleIndex, triangle[ 0 ].triangle(), triangle[ 1 ].triangle(), triangle[ 2 ].triangle() );

                    // Add self.
                    neighborhood.pushBackUnchecked( RingTriangle( triangleIndex, 0 ) );

                    // Add 1-ring.
                    if ( triangle[ 0 ].isValid() ) neighborhood.pushBackUnchecked( RingTriangle( triangle[ 0 ].triangle(), 1 ) );
                    if ( triangle[ 1 ].isValid() ) neighborhood.pushBackUnchecked( RingTriangle( triangle[ 1 ].triangle(), 1 ) );
                    if ( triangle[ 2 ].isValid() ) neighborhood.pushBackUnchecked( RingTriangle( triangle[ 2 ].triangle(), 1 ) );

                    // Add 2-ring.
                    const int* vi = &geometry.m_triangles[ triangleIndex ].m_a;
                    for ( int i = 0; i < 3; ++i )
                    {
                        const Vertex infos = m_vertices[ vi[ i ] ];
                        const Edge* edges = &m_edges[ infos.m_firstEdge ];
                        for ( int j = 0; j < infos.m_cardinality; ++j )
                        {
                            hkIntVector tiv; tiv.setAll( edges[ j ].triangle() );
                            if ( !triangleRing.compareEqualS32( tiv ).anyIsSet() )
                            {
                                neighborhood.pushBack( RingTriangle( tiv.getU32<0>(), 2 ) );
                            }
                        }
                    }
                }

                // Write patch.
                m_trianglePatches[ triangleIndex ] = RingTriangle( m_trianglePatches.getSize(), 0 );
                m_trianglePatches.pushBack( RingTriangle( neighborhood.getSize(), 0 ) );
                m_trianglePatches.append( neighborhood );
            }
        }

        #if defined(HK_DEBUG_SLOW)
        for ( int i = 0; i < numVertices; ++i )
        {
            HK_ASSERT( 0x062E70D9, m_vertices[ i ].m_cardinality == counters[ i ], "Invalid counter" );
            HK_ASSERT( 0x062E71D9, !m_vertices[ i ].m_boundary || isNaked( getVertexRing( i )[ 0 ] ), "First edge must be naked if on the boundary." );
        }
        for ( int i = 0; i < numTriangles; ++i )
        {
            HK_ASSERT( 0x062E71DA, !m_triangles[ i ][ 0 ].isValid() || m_triangles[ i ][ 0 ].triangle() <= numTriangles, "Invalid link" );
            HK_ASSERT( 0x062E71DB, !m_triangles[ i ][ 1 ].isValid() || m_triangles[ i ][ 1 ].triangle() <= numTriangles, "Invalid link" );
            HK_ASSERT( 0x062E71DC, !m_triangles[ i ][ 2 ].isValid() || m_triangles[ i ][ 2 ].triangle() <= numTriangles, "Invalid link" );
        }
        #endif
    }
}

void hkConnectivity::getVertexIndices( const hkGeometry& g, Edge e, _Out_writes_all_(3) int* HK_RESTRICT indices ) const
{
    const int* i = &g.m_triangles[ e.triangle() ].m_a, s = e.start();
    indices[ 0 ] = i[ s ];
    indices[ 1 ] = i[ ( s + 1 ) % 3 ];
    indices[ 2 ] = i[ ( s + 2 ) % 3 ];
}

void hkConnectivity::enumerateQuadDominantGeometry( const hkGeometry& geometry, hkConnectivity::QuadCollector* collector ) const
{
    // Build quad edge spines.
    const int numTriangles = geometry.m_triangles.getSize();
    hkArray< EdgeData<hkReal> > edges; edges.reserve( getAllEdges().getSize() );

    for ( const Edge edge : getAllEdges() )
    {
        EdgeData<hkReal>& edgeData = edges.expandOne();
        edgeData.m_edge = Edge::invalid();
        edgeData.m_data = HK_REAL_MAX;
        if ( isBound( edge ) )
        {
            const auto calcTwiceSurfaceArea = []( const hkVector4& a, const hkVector4& b, const hkVector4& c )
            {
                const hkVector4 d0 = b - a;
                const hkVector4 d1 = c - a;
                hkVector4 n; n.setCross( d0, d1 );
                return n.length<3, HK_ACC_FULL, HK_SQRT_SET_ZERO>();
            };

            int vis[ 4 ];
            getVertexIndices( geometry, edge, vis );
            vis[ 3 ] = getApexVertexIndex( geometry, getLink( edge ) );

            const hkVector4 vertices[ 4 ] = { geometry.m_vertices[ vis[ 0 ] ], geometry.m_vertices[ vis[ 1 ] ], geometry.m_vertices[ vis[ 2 ] ], geometry.m_vertices[ vis[ 3 ] ] };

            hkAabb quadAabb; quadAabb.setFromTetrahedron( vertices[ 0 ], vertices[ 1 ], vertices[ 2 ], vertices[ 3 ] );

            const hkSimdReal aabbSurfaceArea = hkVector4UtilInternal::surfaceArea( quadAabb );

            if ( aabbSurfaceArea > hkSimdReal_Eps )
            {
                const hkSimdReal quadSurfaceArea = calcTwiceSurfaceArea( vertices[ 0 ], vertices[ 1 ], vertices[ 2 ] ) + calcTwiceSurfaceArea( vertices[ 0 ], vertices[ 1 ], vertices[ 3 ] );
                edgeData.m_data = ( ( aabbSurfaceArea - quadSurfaceArea ) / aabbSurfaceArea ).getReal();

                if ( vis[ 0 ] < vis[ 1 ] )
                    edgeData.m_edge = edge;
                else
                    edgeData.m_edge = getLink( edge );
            }
        }
    }

    // Sort.
    hkSort( edges.begin(), edges.getSize(), []( const EdgeData<hkReal>& a, const EdgeData<hkReal>& b ) { return a.m_data < b.m_data; } );

    // Generate quads.
    hkBitField freeTriangles; freeTriangles.setSizeAndFill( 0, numTriangles, 1 );
    for ( const auto& edgeData : edges )
    {
        if ( !edgeData.m_edge.isValid() ) break;

        const int t0 = edgeData.m_edge.triangle();
        const int t1 = getLink( edgeData.m_edge ).triangle();
        if ( freeTriangles.get( t0 ) && freeTriangles.get( t1 ) )
        {
            if ( collector->emitQuad( edgeData.m_edge ) )
            {
                freeTriangles.clear( t0 );
                freeTriangles.clear( t1 );
            }
        }
    }

    // Generate triangles.
    for ( int index = 0; index < numTriangles; ++index )
    {
        if ( freeTriangles.get( index ) == 0 ) continue;
        Edge edge = Edge( index, 0 );
        if ( getStartVertexIndex( geometry, edge ) < getEndVertexIndex( geometry, edge ) ) edge = getNext( edge );
        if ( getStartVertexIndex( geometry, edge ) < getEndVertexIndex( geometry, edge ) ) edge = getNext( edge );
        collector->emitTriangle( edge );
    }
}

void hkConnectivity::regularize( hkGeometry& geometry, int iterations, hkReal maxAngle, bool conserveVolume, bool protectBoundaries ) const
{
    hkArray<hkVector4> temp( geometry.m_vertices.getSize() );
    hkArray<hkVector4> triangleCentroids; computeTriangleCentroids( geometry, triangleCentroids );
    hkArray<hkVector4[ 2 ]> borderCentroids;
    hkBitField activeVertices; activeVertices.setSizeAndFill( 0, geometry.m_vertices.getSize(), 1 );

    if ( maxAngle > 0 )
    {
        const hkSimdReal maxCosAngle = hkSimdReal::fromFloat( hkMath::cos( maxAngle ) );
        for ( const auto& edge : getAllEdges() )
        {
            if ( isBound( edge ) )
            {
                hkVector4 t0[ 3 ]; geometry.getTriangle( edge.triangle(), t0 );
                hkVector4 n0; hkcdTriangleUtil::calcUnitNormal( t0[ 0 ], t0[ 1 ], t0[ 2 ], n0 );
                hkVector4 t1[ 3 ]; geometry.getTriangle( getLink( edge ).triangle(), t1 );
                hkVector4 n1; hkcdTriangleUtil::calcUnitNormal( t1[ 0 ], t1[ 1 ], t1[ 2 ], n1 );
                if ( n0.dot<3>( n1 ) <= maxCosAngle )
                {
                    activeVertices.clear( getStartVertexIndex( geometry, edge ) );
                    activeVertices.clear( getEndVertexIndex( geometry, edge ) );
                }
            }
        }
    }

    if ( protectBoundaries )
    {
        for ( int vertexIndex = 0; vertexIndex < geometry.m_vertices.getSize(); ++vertexIndex )
        {
            if ( activeVertices.get( vertexIndex ) && isBorderVertex( vertexIndex ) )
            {
                activeVertices.clear( vertexIndex );
            }
        }
    }
    else
    {
        borderCentroids.setSize( geometry.m_vertices.getSize() );
        hkConcurrency::parallelFor( geometry.m_vertices.getSize(), [this, &geometry, &borderCentroids, &activeVertices]( int vertexIndex )
        {
            if ( isBorderVertex( vertexIndex ) && activeVertices.get( vertexIndex ) )
            {
                const auto ring = getVertexRing( vertexIndex );
                const hkVector4 origin = geometry.m_vertices[ vertexIndex ];
                const hkVector4 vertexA = geometry.m_vertices[ getApexVertexIndex( geometry, ring.back() ) ];
                const hkVector4 vertexB = geometry.m_vertices[ getEndVertexIndex( geometry, ring.front() ) ];
                hkVector4* bc = borderCentroids[ vertexIndex ];
                bc[ 0 ] = ( vertexA + origin ) * hkSimdReal_Inv2;
                bc[ 1 ] = ( vertexB + origin ) * hkSimdReal_Inv2;
            }
        } );
    }

    for ( int iteration = 0; iteration < iterations; ++iteration )
    {
        const auto* HK_RESTRICT verticesIn = geometry.m_vertices.begin();
        auto* HK_RESTRICT verticesOut = temp.begin();

        hkConcurrency::parallelFor( geometry.m_vertices.getSize(), [this, verticesIn, verticesOut, &geometry, &triangleCentroids, &borderCentroids, &activeVertices, protectBoundaries, maxAngle]( int vertexIndex )
        {
            const auto vertexPosition = verticesIn[ vertexIndex ];
            verticesOut[ vertexIndex ] = vertexPosition;

            if ( !activeVertices.get( vertexIndex ) || !isManifoldVertex( vertexIndex ) ) return;

            hkVector4 newVertexPosition = vertexPosition;
            hkSimdReal weightSum = hkSimdReal_0;
            hkVector4 sum; sum.setZero();

            if ( isBorderVertex( vertexIndex ) )
            {
                const hkVector4* centroids = borderCentroids[ vertexIndex ];
                for ( int i = 0; i < 2; ++i )
                {
                    const auto weight = vertexPosition.distanceTo<HK_ACC_FULL, HK_SQRT_SET_ZERO>( centroids[ i ] );
                    sum.addMul( centroids[ i ], weight );
                    weightSum.add( weight );
                }
            }
            else
            {
                for ( const auto edge : getVertexRing( vertexIndex ) )
                {
                    const auto centroid = triangleCentroids[ edge.triangle() ];
                    const auto weight = vertexPosition.distanceTo<HK_ACC_FULL, HK_SQRT_SET_ZERO>( centroid );
                    sum.addMul( centroid, weight );
                    weightSum.add( weight );
                }

            }

            if ( weightSum <= hkSimdReal_Eps ) return;

            verticesOut[ vertexIndex ].setMul( sum, hkSimdReal_1 / weightSum );
        } );

        geometry.m_vertices.swap( temp );

        if ( !conserveVolume && ( iteration + 1 ) < iterations )
        {
            computeTriangleCentroids( geometry, triangleCentroids );
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
