// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0


#pragma once


// this: #include <Common/Base/Types/Geometry/Aabb/hkAabbUtil.h>

#include <Common/Base/Math/Vector/hkVector4Util.h>  // needed for .inl file

class hkAabb;
struct hkAabbUint32;

namespace hkAsyncHeartbeat
{
    class Heartbeat;
}

/// A utility class for creating AABBs that contain various geometric objects.
namespace hkAabbUtil
{
        //
        // Non-continuous AABB calculations
        //

            /// Calculates an AABB from an array of vertices.
        HK_EXPORT_COMMON void HK_CALL               calcAabb( _In_reads_bytes_(numVertices * striding) const hkReal* vertexArray, int numVertices, int striding, hkAabb& aabbOut );

            /// Calculates an AABB from an array of hkVector4.
        HK_EXPORT_COMMON void HK_CALL               calcAabb(_In_reads_(numVertices) const hkVector4* vertexArray, int numVertices, hkAabb& aabbOut);

        HK_EXPORT_COMMON void HK_CALL               calcAabb( const hkVector4* vertexArray, int numVertices, hkAabb& aabbOut, hkAsyncHeartbeat::Heartbeat& heartbeat );

            /// Calculates an AABB from an array of hkVector4 pointers.
        HK_EXPORT_COMMON void HK_CALL               calcAabb(_In_reads_(numVertices) const hkVector4*const* vertexArray, int numVertices, hkAabb& aabbOut );

            /// Calculates an AABB from an OBB specified by a transform, a center, and an halfExtents vector and an extra radius
        HK_INLINE void HK_CALL      calcAabb( const hkTransform& BvToWorld, const hkVector4& halfExtents, const hkVector4& center, hkSimdRealParameter extraRadius, hkAabb& aabbOut);

            /// Calculates an AABB from an OBB specified by a transform, a center, and an halfExtents vector.
        HK_INLINE void HK_CALL      calcAabb( const hkTransform& BvToWorld, const hkVector4& halfExtents, const hkVector4& center, hkAabb& aabbOut);

        /// Calculates an AABB from an OBB specified by a matrix, a center, and an halfExtents vector.
        HK_INLINE void HK_CALL      calcAabb( const hkMatrix4& localToWorld, const hkVector4& halfExtents, const hkVector4& center, hkAabb& aabbOut );

            /// Calculates an AABB from an OBB specified by a transform, and an halfExtents vector.
        HK_INLINE void HK_CALL      calcAabb( const hkTransform& BvToWorld, const hkVector4& halfExtents, hkSimdRealParameter extraRadius, hkAabb& aabbOut);

            /// Calculates an AABB from an OBB specified by a transform, an AABB and an extra radius.
        HK_EXPORT_COMMON void HK_CALL   calcAabb(const hkTransform& BvToWorld, const hkAabb& aabb, hkSimdRealParameter extraRadius, hkAabb& aabbOut);

            /// Calculates an AABB from an OBB specified by a matrix and an AABB.
        HK_EXPORT_COMMON void HK_CALL   calcAabb(const hkMatrix4& BvToWorld, const hkAabb& aabb, hkAabb& aabbOut);

            /// Calculates an AABB from an OBB specified by a transform and an AABB.
        HK_EXPORT_COMMON void HK_CALL   calcAabb(const hkTransform& BvToWorld, const hkAabb& aabb, hkAabb& aabbOut);

            /// Calculates an AABB from an OBB specified by a transform with scale and an AABB.
        HK_EXPORT_COMMON void HK_CALL               calcAabb(const hkQsTransform& bvToWorld, const hkAabb& aabb, hkAabb& aabbOut);

            /// Calculates an AABB from an OBB specified by a transform without scale and an AABB.
        HK_EXPORT_COMMON void HK_CALL               calcAabb(const hkQTransform& bvToWorld, const hkAabb& aabb, hkAabb& aabbOut);

            /// Returns the minimum gap between the inner and outer aabb.
        HK_INLINE hkSimdReal HK_CALL calcMinGap( const hkAabb& innerAabb, const hkAabb& outerAabb );

            /// Calculates an AABB from center and inner radius.
        HK_INLINE void HK_CALL      calcAabb( const hkVector4& center, hkSimdRealParameter innerRadius, hkAabb& aabbOut);

        HK_INLINE void HK_CALL      transformAabbIntoLocalSpace( const hkTransform& localToWorld, const hkAabb& aabb, hkAabb& aabbOut );

            /// Get the outer radius of an AABB.
        HK_INLINE hkSimdReal HK_CALL    getOuterRadius(const hkAabb& aabb);

            /// Get the distance squared from point to an AABB, with point inside AABB distance == 0.
        HK_INLINE hkSimdReal            distanceSquared(const hkAabb& a, hkVector4Parameter b);

        HK_INLINE hkSimdReal            distanceSquared(const hkAabb& a, const hkAabb& b);

            /// Get a vertex of an AABB. (with index bits set for positive as follow: Z | Y | X)
        HK_INLINE void              getVertex(const hkAabb& aabb, int index, hkVector4& vertexOut);

            /// Get the 8 corner of an aabb
        HK_INLINE void              get8Vertices(const hkAabb& aabb, _Inout_updates_all_(8) hkVector4* verticesOut);

            /// Scale an AABB.
        HK_INLINE void      scaleAabb(const hkVector4& scale, const hkAabb& aabbIn, hkAabb& aabbOut);

            /// Expand an AABB by a combination of linear and angular motion.
            /// The resulting AABB is as follows:
            ///  - newMin = MIN(aabb.min + motionXYZ , aabb.min) - {anyDirectionExpansion,anyDirectionExpansion,anyDirectionExpansion}
            ///  - newMax = MAX(aabb.max + motionXYZ , aabb.max) + {anyDirectionExpansion,anyDirectionExpansion,anyDirectionExpansion}
        HK_INLINE void      expandAabbByMotion(const hkAabb& aabbIn, const hkVector4& motion, hkSimdRealParameter anyDirectionExpansion, hkAabb& aabbOut);

            /// expand aabb by the linear motion of a body.
        HK_INLINE void      expandAabbByMotion(const hkAabb& aabbIn, const hkVector4& motion, hkAabb& aabbOut);

            /// expand aabb by two different linear motions plus any direction motion of a body.
        HK_INLINE void      expandAabbByMotion(const hkAabb& aabbIn, hkVector4Parameter motionA, hkVector4Parameter motionB, hkSimdRealParameter anyDirectionExpansion, hkAabb& aabbOut);

            /// expand aabb by two different linear motions plus any direction motion of a body. This version is needed for neighbor welding (needs a larger aabb)
        HK_INLINE void      expandAabbByMotionCircle(const hkAabb& aabbIn, hkVector4Parameter motionA, hkSimdRealParameter anyDirectionExpansion, hkAabb& aabbOut);

            /// Project an AABB on an axis.
        HK_INLINE void      projectAabbOnAxis(const hkVector4& axis, const hkAabb& aabb, hkSimdReal& minOut, hkSimdReal& maxOut);

            /// Project the minimum extend of an AABB on an axis.
        HK_INLINE void      projectAabbMinOnAxis(const hkVector4& axis, const hkAabb& aabb, hkSimdReal& prjOut);

            /// Project the maximum extend of an AABB on an axis.
        HK_INLINE void      projectAabbMaxOnAxis(const hkVector4& axis, const hkAabb& aabb, hkSimdReal& prjOut);

            /// Compute the widths span of an AABB wrt. a plane.
        HK_INLINE void      computeAabbPlaneSpan(const hkVector4& plane, const hkAabb& aabb, hkSimdReal& minOut, hkSimdReal& maxOut);

            /// Construct the plane equations for \a aabb. Note \a planesOut may hold less than 6 planes if the \a aabb has no extent in any dimension.
        void HK_CALL getPlanes(const hkAabb& aabb, hkArray<hkVector4>& planesOut);

        /// Converts a floating-point AABB to an integer AABB
        HK_INLINE void HK_CALL convertAabbToUint32( const hkAabb& aabb, hkVector4fParameter offsetLow, hkVector4fParameter offsetHigh, hkVector4fParameter scale, hkAabbUint32& aabbOut );

        /// Converts an integer AABB to a floating-point AABB
        HK_INLINE void HK_CALL convertAabbFromUint32( const hkAabbUint32& aabbIn, hkVector4fParameter offsetLow, hkVector4fParameter scale, hkAabb& aabbOut);

        /// Store the difference between a big expandedAabbInt and a smaller, enclosed unexpandedAabbInt_InOut in m_expansionXXX components of unexpandedAabbInt_InOut.
        HK_INLINE void HK_CALL compressExpandedAabbUint32(const hkAabbUint32& expandedAabbInt, hkAabbUint32& unexpandedAabbInt_InOut);

        /// Expand a compressed unexpandedAabbInt to an expandedAabbOut using the information stored in the m_expansionXXX components.
        HK_INLINE void HK_CALL uncompressExpandedAabbUint32(const hkAabbUint32& unexpandedAabbInt, hkAabbUint32& expandedAabbOut);

        /// Calculates the optimal clipped screen space bounding box for a world space axis aligned bounding box,
        /// matrix convention for viewProjection is clip space depth = minus one to one
        HK_EXPORT_COMMON void HK_CALL calculateScreenSpaceAabb(const hkAabb& worldSpace, const hkMatrix4& viewProjection, hkAabb& screenSpace);

        /// Calculates the screen space bounding box for a world space axis aligned bounding box,
        /// matrix convention for viewProjection is clip space depth = minus one to one
        HK_EXPORT_COMMON void HK_CALL calculateScreenSpaceAabbUnclipped(const hkAabb& worldSpace, const hkMatrix4& viewProjection, hkAabb& screenSpace);

        enum { AABB_UINT32_MAX_FVALUE = 0x7ffe0000 };
        extern HK_EXPORT_COMMON hkQuadReal hkAabbUint32MaxVal;
}

#include <Common/Base/Types/Geometry/Aabb/hkAabbUtil.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
