// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

    /// Gets the Alpha component of a color.
inline  hkUint8 HK_CALL hkColor::getAlphaAsChar(Argb color)
{
    return hkUint8((color >> 24) & 0xFF);
}

inline  hkReal HK_CALL hkColor::getAlphaAsFloat(Argb color)
{
    return hkReal((color>>24)&0xff) / hkReal(255.0f);
}

    /// Gets the Red component of a color.
inline  hkUint8 HK_CALL hkColor::getRedAsChar(Argb color)
{
    return hkUint8((color >> 16) & 0xFF);
}

inline  hkReal HK_CALL hkColor::getRedAsFloat(Argb color)
{
    return hkReal((color>>16)&0xff) / hkReal(255.0f);
}

    /// Gets the Green component of a color.
inline  hkUint8 HK_CALL hkColor::getGreenAsChar(Argb color)
{
    return hkUint8((color >> 8) & 0xFF);
}

inline  hkReal HK_CALL hkColor:: getGreenAsFloat(Argb color)
{
    return hkReal((color>>8)&0xff) / hkReal(255.0f);
}

    /// Gets the Blue component of a color.
inline  hkUint8 HK_CALL hkColor::getBlueAsChar(Argb color)
{
    return hkUint8(color & 0xFF);
}

inline  hkReal HK_CALL hkColor::getBlueAsFloat(Argb color)
{
    return hkReal(color&0xff) / hkReal(255.0f);
}

hkColor::Argb HK_CALL hkColor::rgbFromChars( unsigned char red, unsigned char green, unsigned char blue, unsigned char alpha )
{
    hkUint32 color = hkUint32(alpha)<<24;
    color |= hkUint32( red ) << 16;
    color |= hkUint32( green ) << 8;
    color |= hkUint32( blue );
    return color;
}

    /// Gets the entire color as an array of four chars in the order R,G,B,A
inline  void HK_CALL hkColor::getRgbaAsChars(Argb color, _Out_writes_(4) hkUint8* rgba)
{
    rgba[0] = getRedAsChar(color); rgba[1] = getGreenAsChar(color); rgba[2] = getBlueAsChar(color); rgba[3] = getAlphaAsChar(color);
}

    /// Gets the entire color as a hkVector4 (R,G,B,A)
inline  void HK_CALL hkColor::getRgbaAsVector4(Argb color, hkVector4f& floatColor)
{
    floatColor.set(getRedAsChar(color), getGreenAsChar(color), getBlueAsChar(color), getAlphaAsChar(color));
    floatColor.mul(hkVector4f::getConstant< HK_QUADREAL_INV_255 >());
}

inline  void HK_CALL hkColor::getRgbaAsVector4(_In_reads_(4) const hkUint8* color, hkVector4f& floatColor)
{
    floatColor.set(color[0], color[1], color[2], color[3]);
    floatColor.mul(hkVector4f::getConstant< HK_QUADREAL_INV_255 >());
}

    /// Combine two colors. Uses the identity (a+b)/2 == (a^b)>>1 + (a&b).
inline hkColor::Argb HK_CALL hkColor::average(Argb a, Argb b)
{
    return ( ((a^b)&0xfefefefe) >> 1 ) + (a&b);
}

    /// Halve the alpha.
inline hkColor::Argb HK_CALL hkColor::semiTransparent(Argb color, int n)
{
    return ((color>>n) & 0xFF000000) | (color&0x00ffffff);
}

    /// Make the color darker.
inline hkColor::Argb HK_CALL hkColor::darken(Argb col, int n)
{
    Argb masklo = (((1<<n) - 1) * 0x00010101); // Build a mask for the lower n bits of each of the bytes
    Argb maskLoAndAlpha = masklo | 0xff000000;
    Argb colShifted = ( col & ~maskLoAndAlpha ) >> n; // Clear alpha, Mask out the lower bits of each component and shift
    return (col & 0xff000000) | colShifted; // Restore alpha
}

    /// Make the color lighter.
inline int HK_CALL hkColor::lighten(Argb col, int n)
{
    Argb lobits = (( (1<<   n ) - 1) * 0x00010101); // populate new low bits with 1s
    Argb maskhi = (( (1<<(8-n)) - 1) * 0x00010101); // mask of bits shifted out
    return (col & ~maskhi) // preserve alpha and high bits of each component. consider lighten(col=0xff808080,n=1)
        | ((col&maskhi) << n) // shift bits in each component upward, mask ensures components don't interfere
        | lobits; // fill new low bits with 1s
}

HK_INLINE hkColor::Argb hkColor::fromArgb( unsigned char alpha, unsigned char red, unsigned char green, unsigned char blue )
{
    return ((static_cast<Argb>(alpha)<<24) | (static_cast<Argb>(red)<<16) | (static_cast<Argb>(green)<<8) | static_cast<Argb>(blue));
}

HK_INLINE hkColor::Argb hkColor::replaceAlpha( unsigned char alpha, Argb rgb )
{
    return ((static_cast<Argb>(alpha)<<24) | (rgb & 0x00ffffff));
}

HK_INLINE hkColor::Argb hkColor::scaleAlpha( hkReal factor, Argb rgb )
{
    return replaceAlpha( static_cast<unsigned char>( factor * getAlphaAsChar( rgb ) ), rgb );
}

inline hkColorf::hkColorf() { }

inline hkColorf::hkColorf(hkFloat32 r, hkFloat32 g, hkFloat32 b, hkFloat32 a)
{
    set(r, g, b, a);
}

inline hkColorf::hkColorf(const hkColorf& other)
    : hkVector4f()
{
    *this = other;
}

inline hkColorf::hkColorf(const hkColor::Argb& argb)
{
    *this = argb;
}

inline hkColorf::hkColorf(const hkColorUbGamma& ubGamma)
{
    *this = ubGamma;
}

inline hkColorf::hkColorf(const hkColorUbLinear& ubLinear)
{
    *this = ubLinear;
}

inline hkColorf::hkColorf(const hkVector4f& other)
    : hkVector4f(other)
{
}

inline hkColorf& hkColorf::operator=(const hkColor::Argb& argb)
{
    hkColor::getRgbaAsVector4(argb, *this);
    return *this;
}

inline hkColorf& hkColorf::operator=(const hkColorUbGamma& ubGamma)
{
    hkFloat32 rgba[4];
    rgba[0] = gammaToLinear(uint8ToFloat(ubGamma.r));
    rgba[1] = gammaToLinear(uint8ToFloat(ubGamma.g));
    rgba[2] = gammaToLinear(uint8ToFloat(ubGamma.b));
    rgba[3] = uint8ToFloat(ubGamma.a);

    load<4, HK_IO_NATIVE_ALIGNED>(rgba);

    return *this;
}

inline hkColorf& hkColorf::operator=(const hkColorUbLinear& ubLinear)
{
    hkFloat32 rgba[4];
    rgba[0] = uint8ToFloat(ubLinear.r);
    rgba[1] = uint8ToFloat(ubLinear.g);
    rgba[2] = uint8ToFloat(ubLinear.b);
    rgba[3] = uint8ToFloat(ubLinear.a);

    load<4, HK_IO_NATIVE_ALIGNED>(rgba);

    return *this;
}

HK_INLINE bool hkColorf::operator==(const hkColorf& other) const
{
    return equal(other).allAreSet() != 0;
}

HK_INLINE bool hkColorf::operator!=(const hkColorf& other) const
{
    return equal(other).allAreSet() == 0;
}

_Ret_notnull_
inline hkColorf::operator const hkFloat32* () const
{
    return (const hkFloat32*)this;
}

HK_INLINE hkFloat32 hkColorf::getRed() const
{
    return getComponent<0>().getReal();
}

HK_INLINE hkFloat32 hkColorf::getGreen() const
{
    return getComponent<1>().getReal();
}

HK_INLINE hkFloat32 hkColorf::getBlue() const
{
    return getComponent<2>().getReal();
}

HK_INLINE hkFloat32 hkColorf::getAlpha() const
{
    return getComponent<3>().getReal();
}

HK_INLINE void hkColorf::setRed(hkFloat32 r)
{
    setComponent<0>(hkSimdFloat32::fromFloat(r));
}

HK_INLINE void hkColorf::setGreen(hkFloat32 g)
{
    setComponent<1>(hkSimdFloat32::fromFloat(g));
}

HK_INLINE void hkColorf::setBlue(hkFloat32 b)
{
    setComponent<2>(hkSimdFloat32::fromFloat(b));
}

HK_INLINE void hkColorf::setAlpha(hkFloat32 a)
{
    setComponent<3>(hkSimdFloat32::fromFloat(a));
}

inline hkBool32 hkColorf::compare(const hkColorf& other, const hkSimdFloat32& tolerance, int numChannels) const
{
    hkVector4f diff; diff.setSub(*this, other);
    hkVector4f diffA; diffA.setAbs( diff );
    hkVector4f tol; tol.setAll( tolerance );
    hkVector4ComparisonMask::Mask mask = ( hkVector4ComparisonMask::Mask )( (1 << numChannels) - 1 );
    return diffA.less( tol ).allAreSet( mask );
}

inline bool hkColorf::isHdr() const
{
    hkBool32 isInRange = lessEqual(hkVector4f::ctor(hkSimdFloat32_1)).allAreSet() && greaterEqualZero().allAreSet();
    return isInRange == hkFalse32;
}



HK_INLINE hkUint8 hkColorf::floatToUint8(hkFloat32 f)
{
    return static_cast<hkUint8>(hkMath::clamp(f * 255.0f + 0.5f, 0.0f, 255.0f));
}

HK_INLINE hkFloat32 hkColorf::uint8ToFloat(hkUint8 i)
{
    return i * (1.0f / 255.0f);
}

hkColorf hkColorf::replaceAlpha(hkFloat32 newAlpha) const
{
    hkColorf copy = *this;
    copy.setComponent<3>(hkSimdFloat32::fromFloat(newAlpha));
    return copy;
}


// ----------------------------------------------------------------------------
//  hkColorUbBase
// ----------------------------------------------------------------------------
template <int C> hkUint8 getColorComponent(hkUint32 color);

template <>
HK_INLINE hkUint8 getColorComponent<0>(hkUint32 color)
{
    return color & 0xff;
}

template <>
HK_INLINE hkUint8 getColorComponent<1>(hkUint32 color)
{
    return (color >> 8) & 0xff;
}

template <>
HK_INLINE hkUint8 getColorComponent<2>(hkUint32 color)
{
    return (color >> 16) & 0xff;
}

template <>
HK_INLINE hkUint8 getColorComponent<3>(hkUint32 color)
{
    return (color >> 24) & 0xff;
}


template <int C> void setColorComponent(hkUint32& color, hkUint8 value);

template <>
HK_INLINE void setColorComponent<0>(hkUint32& color, hkUint8 value)
{
    color = (color & 0xffffff00) | value;
}

template <>
HK_INLINE void setColorComponent<1>(hkUint32& color, hkUint8 value)
{
    color = (color & 0xffff00ff) | (static_cast<hkUint32>(value) << 8);
}

template <>
HK_INLINE void setColorComponent<2>(hkUint32& color, hkUint8 value)
{
    color = (color & 0xff00ffff) | (static_cast<hkUint32>(value) << 16);
}

template <>
HK_INLINE void setColorComponent<3>(hkUint32& color, hkUint8 value)
{
    color = (color & 0x00ffffff) | (static_cast<hkUint32>(value) << 24);
}


HK_INLINE hkColorUbBase::hkColorUbBase()
{
}

HK_INLINE hkColorUbBase::hkColorUbBase(hkUint8 red, hkUint8 green, hkUint8 blue, hkUint8 alpha)
    : r(red)
    , g(green)
    , b(blue)
    , a(alpha)
{
}

_Ret_notnull_
HK_INLINE const hkUint8* hkColorUbBase::getData() const
{
    return &r;
}

_Ret_notnull_
HK_INLINE hkUint8* hkColorUbBase::getData()
{
    return &r;
}

template<>
HK_INLINE void hkColorUbBase::setFromBuffer<hkColorUbBase::ORDER_ARGB>(_In_reads_(4) const hkUint8* buffer)
{
    //  ARGB
    //  0123
    r = buffer[1];
    g = buffer[2];
    b = buffer[3];
    a = buffer[0];
}

template<>
HK_INLINE void hkColorUbBase::setFromBuffer<hkColorUbBase::ORDER_ABGR>(_In_reads_(4) const hkUint8* buffer)
{
    //  ABGR
    //  0123
    r = buffer[3];
    g = buffer[2];
    b = buffer[1];
    a = buffer[0];
}

template<>
HK_INLINE void hkColorUbBase::setFromBuffer<hkColorUbBase::ORDER_RGBA>(_In_reads_(4) const hkUint8* buffer)
{
    //  RGBA
    //  0123
    r = buffer[0];
    g = buffer[1];
    b = buffer[2];
    a = buffer[3];
}

template<>
HK_INLINE void hkColorUbBase::setFromBuffer<hkColorUbBase::ORDER_BGRA>(_In_reads_(4) const hkUint8* buffer)
{
    //  BGRA
    //  0123
    r = buffer[2];
    g = buffer[1];
    b = buffer[0];
    a = buffer[3];
}

HK_INLINE void hkColorUbBase::setFromFloat(hkFloat32 rf, hkFloat32 gf, hkFloat32 bf, hkFloat32 af)
{
    r = hkColorf::floatToUint8(rf);
    g = hkColorf::floatToUint8(gf);
    b = hkColorf::floatToUint8(bf);
    a = hkColorf::floatToUint8(af);
}

template<>
HK_INLINE void hkColorUbBase::setFromUint32<hkColorUbBase::ORDER_ARGB>(hkUint32 color)
{
    //  ARGB
    //  3210
    r = getColorComponent<2>(color);
    g = getColorComponent<1>(color);
    b = getColorComponent<0>(color);
    a = getColorComponent<3>(color);
}

template<>
HK_INLINE void hkColorUbBase::setFromUint32<hkColorUbBase::ORDER_ABGR>(hkUint32 color)
{
    //  ABGR
    //  3210
    r = getColorComponent<0>(color);
    g = getColorComponent<1>(color);
    b = getColorComponent<2>(color);
    a = getColorComponent<3>(color);
}

template<>
HK_INLINE void hkColorUbBase::setFromUint32<hkColorUbBase::ORDER_RGBA>(hkUint32 color)
{
    //  RGBA
    //  3210
    r = getColorComponent<3>(color);
    g = getColorComponent<2>(color);
    b = getColorComponent<1>(color);
    a = getColorComponent<0>(color);
}

template<>
HK_INLINE void hkColorUbBase::setFromUint32<hkColorUbBase::ORDER_BGRA>(hkUint32 color)
{
    //  BGRA
    //  3210
    r = getColorComponent<1>(color);
    g = getColorComponent<2>(color);
    b = getColorComponent<3>(color);
    a = getColorComponent<0>(color);
}

template<>
HK_INLINE hkUint32 hkColorUbBase::toUint32<hkColorUbBase::ORDER_ARGB>() const
{
    hkUint32 color = 0;
    //  ARGB
    //  3210
    setColorComponent<2>(color, r);
    setColorComponent<1>(color, g);
    setColorComponent<0>(color, b);
    setColorComponent<3>(color, a);
    return color;
}

template<>
HK_INLINE hkUint32 hkColorUbBase::toUint32<hkColorUbBase::ORDER_ABGR>() const
{
    hkUint32 color = 0;
    //  ABGR
    //  3210
    setColorComponent<0>(color, r);
    setColorComponent<1>(color, g);
    setColorComponent<2>(color, b);
    setColorComponent<3>(color, a);
    return color;
}

template<>
HK_INLINE hkUint32 hkColorUbBase::toUint32<hkColorUbBase::ORDER_RGBA>() const
{
    hkUint32 color = 0;
    //  RGBA
    //  3210
    setColorComponent<3>(color, r);
    setColorComponent<2>(color, g);
    setColorComponent<1>(color, b);
    setColorComponent<0>(color, a);
    return color;
}

template<>
HK_INLINE hkUint32 hkColorUbBase::toUint32<hkColorUbBase::ORDER_BGRA>() const
{
    hkUint32 color = 0;
    //  BGRA
    //  3210
    setColorComponent<1>(color, r);
    setColorComponent<2>(color, g);
    setColorComponent<3>(color, b);
    setColorComponent<0>(color, a);
    return color;
}

template<>
HK_INLINE void hkColorUbBase::toBuffer<hkColorUbBase::ORDER_ARGB>(_Out_writes_all_(4) hkUint8* buffer) const
{
    //  ARGB
    //  0123
    buffer[1] = r;
    buffer[2] = g;
    buffer[3] = b;
    buffer[0] = a;
}

template<>
HK_INLINE void hkColorUbBase::toBuffer<hkColorUbBase::ORDER_ABGR>(_Out_writes_all_(4) hkUint8* buffer) const
{
    //  ABGR
    //  0123
    buffer[3] = r;
    buffer[2] = g;
    buffer[1] = b;
    buffer[0] = a;
}

template<>
HK_INLINE void hkColorUbBase::toBuffer<hkColorUbBase::ORDER_RGBA>(_Out_writes_all_(4) hkUint8* buffer) const
{
    //  RGBA
    //  0123
    buffer[0] = r;
    buffer[1] = g;
    buffer[2] = b;
    buffer[3] = a;
}

template<>
HK_INLINE void hkColorUbBase::toBuffer<hkColorUbBase::ORDER_BGRA>(_Out_writes_all_(4) hkUint8* buffer) const
{
    //  BGRA
    //  0123
    buffer[2] = r;
    buffer[1] = g;
    buffer[0] = b;
    buffer[3] = a;
}


// ----------------------------------------------------------------------------
//  hkColorUbGamma
// ----------------------------------------------------------------------------
HK_INLINE hkColorUbGamma::hkColorUbGamma()
{
}

HK_INLINE hkColorUbGamma::hkColorUbGamma(hkUint8 gammaR, hkUint8 gammaG, hkUint8 gammaB, hkUint8 linearA)
    : hkColorUbBase(gammaR, gammaG, gammaB, linearA)
{
}

HK_INLINE hkColorUbGamma::hkColorUbGamma(const hkColorf& color)
{
    *this = color;
}

HK_INLINE hkColorUbGamma::hkColorUbGamma(const hkColorUbLinear& color)
{
    *this = color;
}

HK_INLINE hkColorUbGamma& hkColorUbGamma::operator=(const hkColorf& color)
{
    r = hkColorf::floatToUint8(hkColorf::linearToGamma(color.getComponent<0>().getReal()));
    g = hkColorf::floatToUint8(hkColorf::linearToGamma(color.getComponent<1>().getReal()));
    b = hkColorf::floatToUint8(hkColorf::linearToGamma(color.getComponent<2>().getReal()));
    a = hkColorf::floatToUint8(color.getComponent<3>().getReal());

    return *this;
}

HK_INLINE hkColorUbGamma& hkColorUbGamma::operator=(const hkColorUbLinear& color)
{
    r = m_linearToGammaLut[color.r];
    g = m_linearToGammaLut[color.g];
    b = m_linearToGammaLut[color.b];
    a = color.a;

    return *this;
}

HK_INLINE bool hkColorUbGamma::operator==(const hkColorUbGamma& color) const
{
    return r == color.r && g == color.g && b == color.b && a == color.a;
}

HK_INLINE bool hkColorUbGamma::operator!=(const hkColorUbGamma& color) const
{
    return r != color.r || g != color.g || b != color.b || a != color.a;
}

template<hkColorUbGamma::ColorOrder O>
HK_INLINE hkColorUbGamma hkColorUbGamma::fromBuffer(_In_reads_(4) const hkUint8* buffer)
{
    hkColorUbGamma ret;
    ret.setFromBuffer<O>(buffer);
    return ret;
}

HK_INLINE hkColorUbGamma hkColorUbGamma::fromFloat(hkFloat32 gammaR, hkFloat32 gammaG, hkFloat32 gammaB, hkFloat32 linearA)
{
    hkColorUbGamma ret;
    ret.setFromFloat(gammaR, gammaG, gammaB, linearA);
    return ret;
}

template<hkColorUbGamma::ColorOrder O>
HK_INLINE hkColorUbGamma hkColorUbGamma::fromUint32(hkUint32 color)
{
    hkColorUbGamma ret;
    ret.setFromUint32<O>(color);
    return ret;
}

HK_INLINE hkColorUbGamma hkColorUbGamma::replaceAlpha(hkUint8 alpha) const
{
    return hkColorUbGamma(r, g, b, alpha);
}



// ----------------------------------------------------------------------------
//  hkColorUbLinear
// ----------------------------------------------------------------------------
HK_INLINE hkColorUbLinear::hkColorUbLinear()
{
}

HK_INLINE hkColorUbLinear::hkColorUbLinear(hkUint8 linearR, hkUint8 linearG, hkUint8 linearB, hkUint8 linearA)
    : hkColorUbBase(linearR, linearG, linearB, linearA)
{
}

HK_INLINE hkColorUbLinear::hkColorUbLinear(const hkColorf& color)
{
    *this = color;
}

HK_INLINE hkColorUbLinear::hkColorUbLinear(const hkColorUbGamma& color)
{
    *this = color;
}

HK_INLINE hkColorUbLinear& hkColorUbLinear::operator=(const hkColorf& color)
{
    r = hkColorf::floatToUint8(color.getComponent<0>().getReal());
    g = hkColorf::floatToUint8(color.getComponent<1>().getReal());
    b = hkColorf::floatToUint8(color.getComponent<2>().getReal());
    a = hkColorf::floatToUint8(color.getComponent<3>().getReal());

    return *this;
}

HK_INLINE hkColorUbLinear& hkColorUbLinear::operator=(const hkColorUbGamma& color)
{
    r = m_gammaToLinearLut[color.r];
    g = m_gammaToLinearLut[color.g];
    b = m_gammaToLinearLut[color.b];
    a = color.a;

    return *this;
}

HK_INLINE bool hkColorUbLinear::operator==(const hkColorUbLinear& color) const
{
    return r == color.r && g == color.g && b == color.b && a == color.a;
}

HK_INLINE bool hkColorUbLinear::operator!=(const hkColorUbLinear& color) const
{
    return r != color.r || g != color.g || b != color.b || a != color.a;
}

template<hkColorUbLinear::ColorOrder O>
HK_INLINE hkColorUbLinear hkColorUbLinear::fromBuffer(_In_reads_(4) const hkUint8* buffer)
{
    hkColorUbLinear ret;
    ret.setFromBuffer<O>(buffer);
    return ret;
}

HK_INLINE hkColorUbLinear hkColorUbLinear::fromFloat(hkFloat32 linearR, hkFloat32 linearG, hkFloat32 linearB, hkFloat32 linearA)
{
    hkColorUbLinear ret;
    ret.setFromFloat(linearR, linearG, linearB, linearA);
    return ret;
}

template<hkColorUbLinear::ColorOrder O>
HK_INLINE hkColorUbLinear hkColorUbLinear::fromUint32(hkUint32 color)
{
    hkColorUbLinear ret;
    ret.setFromUint32<O>(color);
    return ret;
}

HK_INLINE hkColorUbLinear hkColorUbLinear::replaceAlpha(hkUint8 alpha) const
{
    return hkColorUbLinear(r, g, b, alpha);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
