// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

class hkColorf;
class hkColorUbGamma;
class hkColorUbLinear;
class hkPseudoRandomGenerator;

    /// Colors are represented as unsigned ints [0xAARRGGBB] and there are helper
    /// functions to create them from floats or chars
class hkColor
{
    public:

        HK_DECLARE_CLASS(hkColor, New, Reflect);

        typedef hkUint32 Argb;

            /// Creates a color using the specified char Argbs.
        HK_INLINE static Argb HK_CALL rgbFromChars(unsigned char red, unsigned char green, unsigned char blue, unsigned char alpha = 0xff);

            /// Creates a color using the specified floating-point Argbs interpreted as RGB colors in [0,1].
        HK_EXPORT_COMMON static Argb HK_CALL rgbFromFloats(const hkReal red, const hkReal green, const hkReal blue, const hkReal alpha = 1.0f);

            /// Creates a color using the specified floating-point Argbs interpreted as HSV colors in [0,1]
        HK_EXPORT_COMMON static Argb HK_CALL rgbFromHSV(const hkReal h, const hkReal s, const hkReal v, const hkReal alpha = 1.0f);

        /// Generates a set of 'aesthetically pleasing' random colors (the generation is done in HSV space)
        HK_EXPORT_COMMON static void HK_CALL generateRandomColors(hkArray<Argb>& colorsOut, hkReal initialHue = 0.5f, hkReal saturation = 0.5f, hkReal value = 0.99f);

        /// Generates a set of 'aesthetically pleasing' random colors (the generation is done in HSV space)
        HK_EXPORT_COMMON static void HK_CALL generateRandomColors(hkArray<hkColorf>& colorsOut, hkReal initialHue = 0.5f, hkReal saturation = 0.5f, hkReal value = 0.99f);

            /// Creates a random color
        HK_EXPORT_COMMON static Argb HK_CALL getRandomColor();

            /// Creates a random color using the given hkPseudoRandomGenerator
        HK_EXPORT_COMMON static Argb HK_CALL getRandomColor(hkPseudoRandomGenerator& rand);

            /// Create a color from the visible spectrum.
        HK_EXPORT_COMMON static Argb HK_CALL getSpectrumColor(hkReal Argb);

            /// Get the i'th color from s_colorTable with optional alpha.
        HK_EXPORT_COMMON static Argb HK_CALL getPaletteColor(int i, unsigned char alpha = 0xff);

            /// Gets the Alpha component of a color.
        HK_INLINE static hkUint8 HK_CALL getAlphaAsChar(Argb color);
        HK_INLINE static hkReal HK_CALL getAlphaAsFloat(Argb color);

            /// Gets the Red component of a color.
        HK_INLINE static hkUint8 HK_CALL getRedAsChar(Argb color);
        HK_INLINE static hkReal HK_CALL getRedAsFloat(Argb color);

            /// Gets the Green component of a color.
        HK_INLINE static hkUint8 HK_CALL getGreenAsChar(Argb color);
        HK_INLINE static hkReal HK_CALL getGreenAsFloat(Argb color);

            /// Gets the Blue component of a color.
        HK_INLINE static hkUint8 HK_CALL getBlueAsChar(Argb color);
        HK_INLINE static hkReal HK_CALL getBlueAsFloat(Argb color);

            /// Gets the entire color as an array of four chars in the order R,G,B,A
        HK_INLINE static void HK_CALL getRgbaAsChars(Argb color, _Out_writes_(4) hkUint8* rgba);

            /// Gets the entire color as a hkVector4 (R,G,B,A)
        HK_INLINE static void HK_CALL getRgbaAsVector4(Argb color, hkVector4f& floatColor);

        HK_INLINE static void HK_CALL getRgbaAsVector4(_In_reads_(4) const hkUint8* rgba, hkVector4f& floatColor);

            /// Combine two colors. Uses the identity (a+b)/2 == (a^b)>>1 + (a&b).
        HK_INLINE static Argb HK_CALL average(Argb a, Argb b);

            /// Halve the alpha.
        HK_INLINE static Argb HK_CALL semiTransparent(Argb color, int n=1);

            /// Make the color darker.
        HK_INLINE static Argb HK_CALL darken(Argb col, int n=1);

            /// Make the color lighter.
        HK_INLINE static int HK_CALL lighten(Argb col, int n=1);

        HK_INLINE static Argb fromArgb( unsigned char alpha, unsigned char red, unsigned char green, unsigned char blue );

        HK_INLINE static Argb replaceAlpha( unsigned char alpha, Argb rgb );
        HK_INLINE static Argb scaleAlpha( hkReal factor, Argb rgb );

    public:

        // Standard Colors
        

        HK_EXPORT_COMMON static const Argb MAROON;
        HK_EXPORT_COMMON static const Argb DARKRED;
        HK_EXPORT_COMMON static const Argb RED;
        HK_EXPORT_COMMON static const Argb LIGHTPINK;
        HK_EXPORT_COMMON static const Argb CRIMSON;
        HK_EXPORT_COMMON static const Argb PALEVIOLETRED;
        HK_EXPORT_COMMON static const Argb HOTPINK;
        HK_EXPORT_COMMON static const Argb DEEPPINK;
        HK_EXPORT_COMMON static const Argb MEDIUMVIOLETRED;
        HK_EXPORT_COMMON static const Argb PURPLE;
        HK_EXPORT_COMMON static const Argb DARKMAGENTA;
        HK_EXPORT_COMMON static const Argb ORCHID;
        HK_EXPORT_COMMON static const Argb THISTLE;
        HK_EXPORT_COMMON static const Argb PLUM;
        HK_EXPORT_COMMON static const Argb VIOLET;
        HK_EXPORT_COMMON static const Argb FUCHSIA;
        HK_EXPORT_COMMON static const Argb MAGENTA;
        HK_EXPORT_COMMON static const Argb MEDIUMORCHID;
        HK_EXPORT_COMMON static const Argb DARKVIOLET;
        HK_EXPORT_COMMON static const Argb DARKORCHID;
        HK_EXPORT_COMMON static const Argb BLUEVIOLET;
        HK_EXPORT_COMMON static const Argb INDIGO;
        HK_EXPORT_COMMON static const Argb MEDIUMPURPLE;
        HK_EXPORT_COMMON static const Argb SLATEBLUE;
        HK_EXPORT_COMMON static const Argb MEDIUMSLATEBLUE;
        HK_EXPORT_COMMON static const Argb DARKBLUE;
        HK_EXPORT_COMMON static const Argb MEDIUMBLUE;
        HK_EXPORT_COMMON static const Argb BLUE;
        HK_EXPORT_COMMON static const Argb NAVY;
        HK_EXPORT_COMMON static const Argb MIDNIGHTBLUE;
        HK_EXPORT_COMMON static const Argb DARKSLATEBLUE;
        HK_EXPORT_COMMON static const Argb ROYALBLUE;
        HK_EXPORT_COMMON static const Argb CORNFLOWERBLUE;
        HK_EXPORT_COMMON static const Argb LIGHTSTEELBLUE;
        HK_EXPORT_COMMON static const Argb ALICEBLUE;
        HK_EXPORT_COMMON static const Argb GHOSTWHITE;
        HK_EXPORT_COMMON static const Argb LAVENDER;
        HK_EXPORT_COMMON static const Argb DODGERBLUE;
        HK_EXPORT_COMMON static const Argb STEELBLUE;
        HK_EXPORT_COMMON static const Argb DEEPSKYBLUE;
        HK_EXPORT_COMMON static const Argb SLATEGRAY;
        HK_EXPORT_COMMON static const Argb LIGHTSLATEGRAY;
        HK_EXPORT_COMMON static const Argb LIGHTSKYBLUE;
        HK_EXPORT_COMMON static const Argb SKYBLUE;
        HK_EXPORT_COMMON static const Argb LIGHTBLUE;
        HK_EXPORT_COMMON static const Argb TEAL;
        HK_EXPORT_COMMON static const Argb DARKCYAN;
        HK_EXPORT_COMMON static const Argb DARKTURQUOISE;
        HK_EXPORT_COMMON static const Argb CYAN;
        HK_EXPORT_COMMON static const Argb MEDIUMTURQUOISE;
        HK_EXPORT_COMMON static const Argb CADETBLUE;
        HK_EXPORT_COMMON static const Argb PALETURQUOISE;
        HK_EXPORT_COMMON static const Argb LIGHTCYAN;
        HK_EXPORT_COMMON static const Argb AZURE;
        HK_EXPORT_COMMON static const Argb LIGHTSEAGREEN;
        HK_EXPORT_COMMON static const Argb TURQUOISE;
        HK_EXPORT_COMMON static const Argb POWDERBLUE;
        HK_EXPORT_COMMON static const Argb DARKSLATEGRAY;
        HK_EXPORT_COMMON static const Argb AQUAMARINE;
        HK_EXPORT_COMMON static const Argb MEDIUMSPRINGGREEN;
        HK_EXPORT_COMMON static const Argb MEDIUMAQUAMARINE;
        HK_EXPORT_COMMON static const Argb SPRINGGREEN;
        HK_EXPORT_COMMON static const Argb MEDIUMSEAGREEN;
        HK_EXPORT_COMMON static const Argb SEAGREEN;
        HK_EXPORT_COMMON static const Argb LIMEGREEN;
        HK_EXPORT_COMMON static const Argb DARKGREEN;
        HK_EXPORT_COMMON static const Argb GREEN;
        HK_EXPORT_COMMON static const Argb LIME;
        HK_EXPORT_COMMON static const Argb FORESTGREEN;
        HK_EXPORT_COMMON static const Argb DARKSEAGREEN;
        HK_EXPORT_COMMON static const Argb LIGHTGREEN;
        HK_EXPORT_COMMON static const Argb PALEGREEN;
        HK_EXPORT_COMMON static const Argb MINTCREAM;
        HK_EXPORT_COMMON static const Argb HONEYDEW;
        HK_EXPORT_COMMON static const Argb CHARTREUSE;
        HK_EXPORT_COMMON static const Argb LAWNGREEN;
        HK_EXPORT_COMMON static const Argb OLIVEDRAB;
        HK_EXPORT_COMMON static const Argb DARKOLIVEGREEN;
        HK_EXPORT_COMMON static const Argb YELLOWGREEN;
        HK_EXPORT_COMMON static const Argb GREENYELLOW;
        HK_EXPORT_COMMON static const Argb BEIGE;
        HK_EXPORT_COMMON static const Argb LINEN;
        HK_EXPORT_COMMON static const Argb LIGHTGOLDENRODYELLOW;
        HK_EXPORT_COMMON static const Argb OLIVE;
        HK_EXPORT_COMMON static const Argb YELLOW;
        HK_EXPORT_COMMON static const Argb LIGHTYELLOW;
        HK_EXPORT_COMMON static const Argb IVORY;
        HK_EXPORT_COMMON static const Argb DARKKHAKI;
        HK_EXPORT_COMMON static const Argb KHAKI;
        HK_EXPORT_COMMON static const Argb PALEGOLDENROD;
        HK_EXPORT_COMMON static const Argb WHEAT;
        HK_EXPORT_COMMON static const Argb GOLD;
        HK_EXPORT_COMMON static const Argb LEMONCHIFFON;
        HK_EXPORT_COMMON static const Argb PAPAYAWHIP;
        HK_EXPORT_COMMON static const Argb DARKGOLDENROD;
        HK_EXPORT_COMMON static const Argb GOLDENROD;
        HK_EXPORT_COMMON static const Argb ANTIQUEWHITE;
        HK_EXPORT_COMMON static const Argb CORNSILK;
        HK_EXPORT_COMMON static const Argb OLDLACE;
        HK_EXPORT_COMMON static const Argb MOCCASIN;
        HK_EXPORT_COMMON static const Argb NAVAJOWHITE;
        HK_EXPORT_COMMON static const Argb ORANGE;
        HK_EXPORT_COMMON static const Argb BISQUE;
        HK_EXPORT_COMMON static const Argb TAN;
        HK_EXPORT_COMMON static const Argb DARKORANGE;
        HK_EXPORT_COMMON static const Argb BURLYWOOD;
        HK_EXPORT_COMMON static const Argb SADDLEBROWN;
        HK_EXPORT_COMMON static const Argb SANDYBROWN;
        HK_EXPORT_COMMON static const Argb BLANCHEDALMOND;
        HK_EXPORT_COMMON static const Argb LAVENDERBLUSH;
        HK_EXPORT_COMMON static const Argb SEASHELL;
        HK_EXPORT_COMMON static const Argb FLORALWHITE;
        HK_EXPORT_COMMON static const Argb SNOW;
        HK_EXPORT_COMMON static const Argb PERU;
        HK_EXPORT_COMMON static const Argb PEACHPUFF;
        HK_EXPORT_COMMON static const Argb CHOCOLATE;
        HK_EXPORT_COMMON static const Argb SIENNA;
        HK_EXPORT_COMMON static const Argb LIGHTSALMON;
        HK_EXPORT_COMMON static const Argb CORAL;
        HK_EXPORT_COMMON static const Argb DARKSALMON;
        HK_EXPORT_COMMON static const Argb MISTYROSE;
        HK_EXPORT_COMMON static const Argb ORANGERED;
        HK_EXPORT_COMMON static const Argb SALMON;
        HK_EXPORT_COMMON static const Argb TOMATO;
        HK_EXPORT_COMMON static const Argb ROSYBROWN;
        HK_EXPORT_COMMON static const Argb PINK;
        HK_EXPORT_COMMON static const Argb INDIANRED;
        HK_EXPORT_COMMON static const Argb LIGHTCORAL;
        HK_EXPORT_COMMON static const Argb BROWN;
        HK_EXPORT_COMMON static const Argb FIREBRICK;
        HK_EXPORT_COMMON static const Argb BLACK;
        HK_EXPORT_COMMON static const Argb DIMGRAY;
        HK_EXPORT_COMMON static const Argb GRAY;
        HK_EXPORT_COMMON static const Argb DARKGRAY;
        HK_EXPORT_COMMON static const Argb SILVER;
        HK_EXPORT_COMMON static const Argb LIGHTGREY;
        HK_EXPORT_COMMON static const Argb GAINSBORO;
        HK_EXPORT_COMMON static const Argb WHITESMOKE;
        HK_EXPORT_COMMON static const Argb WHITE;
        HK_EXPORT_COMMON static const Argb GREY;
        HK_EXPORT_COMMON static const Argb GREY25;
        HK_EXPORT_COMMON static const Argb GREY50;
        HK_EXPORT_COMMON static const Argb GREY75;

        // Havok product colors
        HK_EXPORT_COMMON static const Argb PHYSICS;
        HK_EXPORT_COMMON static const Argb DESTRUCTION;
        HK_EXPORT_COMMON static const Argb ANIMATION;
        HK_EXPORT_COMMON static const Argb BEHAVIOR;
        HK_EXPORT_COMMON static const Argb CLOTH;
        HK_EXPORT_COMMON static const Argb AI;
        HK_EXPORT_COMMON static const Argb SCRIPT;

        // Effectively empty color
        HK_EXPORT_COMMON static const Argb NONE;

        // A simple color table - arrange in batches of 8 colors light to dark
        HK_EXPORT_COMMON static Argb s_colorTable[32];
};

HK_REFLECT_TYPEDEF( HK_EXPORT_COMMON, hkColor::Argb, hkColor_Argb );

class hkColorf : public hkVector4f
{
public:

    HK_DECLARE_CLASS(hkColorf, New, Pod);
    HK_DETAIL_DECLARE_REFLECT_EXPORT(hkColorf, HK_EXPORT_COMMON);

    
    HK_REFLECT_AS_ARRAY_FIXED(4,float);
    HK_RECORD_ATTR(hk::TypeLayout(hkMath::computeTypeSizeAlign));
    HK_RECORD_ATTR(hk::ReflectDetails(fields=false));
    HK_RECORD_ATTR(hk::IncludeInMgd(false));
    HK_RECORD_ATTR(hk::Ui_NotExpandable);
    typedef hkColorf ReflectDefaultType;

    enum ColorOrder
    {
        UINT8_ARGB,
        UINT8_ABGR,
        UINT8_RGBA,
        UINT8_BGRA,
    };

    HK_INLINE hkColorf();
    HK_INLINE hkColorf(hkFloat32 r, hkFloat32 g, hkFloat32 b, hkFloat32 a = 1.f);
    HK_INLINE hkColorf(const hkColorf& other);
    HK_INLINE hkColorf(const hkColor::Argb& argb);
    HK_INLINE hkColorf(const hkColorUbGamma& ubGamma);
    HK_INLINE hkColorf(const hkColorUbLinear& ubLinear);
    HK_INLINE explicit hkColorf(const hkVector4f& other);

    HK_INLINE hkColorf& operator=(const hkColor::Argb& argb);
    HK_INLINE hkColorf& operator=(const hkColorUbGamma& ubGamma);
    HK_INLINE hkColorf& operator=(const hkColorUbLinear& ubLinear);

    bool operator==(const hkColorf& other) const;
    bool operator!=(const hkColorf& other) const;

    HK_INLINE _Ret_notnull_ operator const hkFloat32* () const;

    HK_INLINE hkFloat32 getRed() const;
    HK_INLINE hkFloat32 getGreen() const;
    HK_INLINE hkFloat32 getBlue() const;
    HK_INLINE hkFloat32 getAlpha() const;

    HK_INLINE void setRed(hkFloat32 r);
    HK_INLINE void setGreen(hkFloat32 g);
    HK_INLINE void setBlue(hkFloat32 b);
    HK_INLINE void setAlpha(hkFloat32 a);

    /// returns true if any component (r,g,b or a) is outside the 0..1 range
    HK_INLINE bool isHdr() const;

    HK_EXPORT_COMMON void setFromHSV(hkFloat32 h, hkFloat32 s, hkFloat32 v, hkFloat32 alpha = 1.0f);

    HK_EXPORT_COMMON void lighten(hkFloat32 amount = 0.5f);
    HK_EXPORT_COMMON void darken(hkFloat32 amount = 0.5f);

    HK_EXPORT_COMMON hkFloat32 calcLuminance() const;

    // Calculate average color from RGB.
    HK_EXPORT_COMMON hkFloat32 calcAverage() const;

    HK_INLINE hkBool32 compare(const hkColorf& other, const hkSimdFloat32& tolerance, int numChannels) const;

    HK_EXPORT_COMMON static hkFloat32 gammaToLinear(hkFloat32 f);
    HK_EXPORT_COMMON static hkFloat32 linearToGamma(hkFloat32 f);

    /// Converts a linear color value in [0;1] float range to [0;255] byte range with correct rounding and clamping
    HK_EXPORT_COMMON static hkUint8 floatToUint8(hkFloat32 f);
    /// Converts a linear color value in [0;255] byte range to [0;1] float range
    HK_EXPORT_COMMON static hkFloat32 uint8ToFloat(hkUint8 i);

    HK_INLINE hkColorf replaceAlpha(hkFloat32 newAlpha) const;

};

typedef const hkColorf& hkColorfParameter;

/// \brief
///   Base class for color classes storing RGBA channel values as unsigned bytes
class HK_EXPORT_COMMON hkColorUbBase
{
    public:
        HK_DECLARE_CLASS(hkColorUbBase, New, Reflect, Pod);

        /// \brief
        ///   Color component ordering variants
        enum ColorOrder
        {
            ORDER_ARGB = 0,
            ORDER_ABGR,
            ORDER_RGBA,
            ORDER_BGRA,
        };

        /// \brief
        ///   Non-initializing constructor
        hkColorUbBase();

        /// \brief
        ///   Initializing constructor
        hkColorUbBase(hkUint8 r, hkUint8 g, hkUint8 b, hkUint8 a);

        /// \brief Conversion to const hkUint8*.
        _Ret_notnull_ const hkUint8* getData() const;

        /// \brief Conversion to hkUint8*
        _Ret_notnull_ hkUint8* getData();

        /// \brief Conversion to hkUint32 with the specified component ordering (MSB to LSB)
        template<ColorOrder O>
        hkUint32 toUint32() const;

        /// \brief Writes the color components into a buffer, using the specified component order.
        ///   Note that the buffer must be at least four bytes large.
        template<ColorOrder O>
        void toBuffer(_Out_writes_all_(4) hkUint8* buffer) const;

    protected:
        /// \brief
        ///   Set the values of this color from a buffer of 8-bit integers. The order or the components
        ///   must be specified.
        template<ColorOrder O>
        void setFromBuffer(_In_reads_(4) const hkUint8* buffer);

        /// \brief
        ///   Sets the values of this color from float values (which must be in the [0.0, 1.0] range).
        void setFromFloat(hkFloat32 rf, hkFloat32 gf, hkFloat32 bf, hkFloat32 af);

        /// \brief
        ///   Set the values of this color from a 32-bit integer representation. The order or the components
        ///   must be specified.
        template<ColorOrder O>
        void setFromUint32(hkUint32 color);

    public:
        hkUint8 r HK_ATTR(hk::Ui_Type(hk::UI_NONE ));
        hkUint8 g HK_ATTR(hk::Ui_Type(hk::UI_NONE ));
        hkUint8 b HK_ATTR(hk::Ui_Type(hk::UI_NONE ));
        hkUint8 a HK_ATTR(hk::Ui_Type(hk::UI_NONE ));

    protected:
        static const hkUint8 m_gammaToLinearLut[256];
        static const hkUint8 m_linearToGammaLut[256];
};


/// \brief
///   Color class storing RGBA channel values as unsigned bytes, compressed using sRGB gamma.
class HK_EXPORT_COMMON hkColorUbGamma : public hkColorUbBase
{
    public:
        HK_DECLARE_CLASS(hkColorUbGamma, New, Reflect, Pod);

        /// \brief
        ///   Non-initializing constructor
        HK_INLINE hkColorUbGamma();

        /// \brief
        ///   Initializing constructor. Note that the alpha value is linear, while the color values are
        ///   in gamma space.
        HK_INLINE hkColorUbGamma(hkUint8 gammaR, hkUint8 gammaG, hkUint8 gammaB, hkUint8 linearA = 255);

        /// \brief
        ///   Initializes the color with a hkColorf, converting the color values to gamma space in the
        ///   process. The float color values will be clamped to the [0.0, 1.0] range.
        HK_INLINE hkColorUbGamma(const hkColorf& color);

        /// \brief
        ///   Initializes the color with a hkColorUbLinear, converting the color values to gamma space in the
        ///   process. Note that due to the limited channel resolution, this is a lossy operation!
        HK_INLINE hkColorUbGamma(const hkColorUbLinear& color);

        /// \brief
        ///   Assigns a hkColorf to this color, converting the color values to gamma space in the
        ///   process. The float color values will be clamped to the [0.0, 1.0] range.
        HK_INLINE hkColorUbGamma& operator=(const hkColorf& color);

        /// \brief
        ///   Assigns a hkColorf to this color, converting the color values to gamma space in the
        ///   process. Note that due to the limited channel resolution, this is a lossy operation!
        HK_INLINE hkColorUbGamma& operator=(const hkColorUbLinear& color);

        /// \brief
        ///   Tests this color for equality to \c color.
        HK_INLINE bool operator==(const hkColorUbGamma& color) const;

        /// \brief
        ///   Tests this color for equality to \c color.
        HK_INLINE bool operator!=(const hkColorUbGamma& color) const;

        /// \brief
        ///   Creates a new color from from a buffer of 8-bit integers. The order or the components
        ///   must be specified. The color components are assumed to be in gamma space; the alpha component
        ///   is assumed to be in linear space.
        template<ColorOrder O>
        HK_INLINE static hkColorUbGamma fromBuffer(_In_reads_(4) const hkUint8* buffer);

        /// \brief
        ///   Creates a new color from float values (which must be in the [0.0, 1.0] range).
        ///   The color components are interpreted as being in gamma space, while the alpha component
        ///   is interpreted as being in linear space.
        HK_INLINE static hkColorUbGamma fromFloat(hkFloat32 gammaR, hkFloat32 gammaG, hkFloat32 gammaB, hkFloat32 linearA = 1.0f);

        /// \brief
        ///   Creates a new color from a 32-bit integer representation. The order or the components
        ///   must be specified. The color components are assumed to be in gamma space; the alpha component
        ///   is assumed to be in linear space.
        template<ColorOrder O>
        HK_INLINE static hkColorUbGamma fromUint32(hkUint32 color);

        /// \brief
        ///   Returns a copy of this color with the specified alpha value
        HK_INLINE hkColorUbGamma replaceAlpha(hkUint8 alpha) const;

        /// \brief
        ///   Returns a color from an internal indexed palette of predefined colors.
        static hkColorUbGamma getPaletteColor(int i, unsigned char alpha = 0xff);

        /// \brief
        ///   Returns a random color generated with the specified random generator.
        static hkColorUbGamma getRandomColor(hkPseudoRandomGenerator& rand);

    protected:
        // A simple color table - arrange in batches of 8 colors light to dark
        static hkColorUbGamma s_palette[32];
};


/// \brief
///   Color class storing RGBA channel values as unsigned bytes in linear color space. Note that
///   the only use case for this class is when colors have to be passed into the rendering pipeline
///   as unsigned bytes, and the pipeline expects that data to be in linear space. Converting from
///   and to this representation will lose accuracy!
class HK_EXPORT_COMMON hkColorUbLinear : public hkColorUbBase
{
    public:
        HK_DECLARE_CLASS(hkColorUbLinear, New, Reflect);

        /// \brief
        ///   Non-initializing constructor
        HK_INLINE hkColorUbLinear();

        /// \brief
        ///   Initializing constructor.
        HK_INLINE hkColorUbLinear(hkUint8 linearR, hkUint8 linearG, hkUint8 linearB, hkUint8 linearA = 255);

        /// \brief
        ///   Initializes the color with a hkColorf. The float color values will be clamped to the
        ///   [0.0, 1.0] range.
        HK_INLINE hkColorUbLinear(const hkColorf& color);

        /// \brief
        ///   Initializes the color with a hkColorUbGamma, converting the color values to linear space in the
        ///   process. Note that due to the limited channel resolution, this is a lossy operation!
        HK_INLINE hkColorUbLinear(const hkColorUbGamma& color);

        /// \brief
        ///   Assigns a hkColorf to this color. The float color values will be clamped to the
        ///   [0.0, 1.0] range.
        HK_INLINE hkColorUbLinear& operator=(const hkColorf& color);

        /// \brief
        ///   Assigns a hkColorUbGamma to this color, converting the color values to linear space in the
        ///   process. Note that due to the limited channel resolution, this is a lossy operation!
        HK_INLINE hkColorUbLinear& operator=(const hkColorUbGamma& color);

        /// \brief
        ///   Tests this color for equality to \c color.
        HK_INLINE bool operator==(const hkColorUbLinear& color) const;

        /// \brief
        ///   Tests this color for equality to \c color.
        HK_INLINE bool operator!=(const hkColorUbLinear& color) const;

        /// \brief
        ///   Creates a new color from a buffer of 8-bit integers. The order or the components
        ///   must be specified. All components are assumed to be in linear space.
        template<ColorOrder O>
        HK_INLINE  static hkColorUbLinear fromBuffer(_In_reads_(4) const hkUint8* buffer);

        /// \brief
        ///   Creates a new color from float values (which must be in the [0.0, 1.0] range).
        ///   All components are interpreted as being in linear space.
        HK_INLINE static hkColorUbLinear fromFloat(hkFloat32 linearR, hkFloat32 linearG, hkFloat32 linearB, hkFloat32 linearA = 1.0f);

        /// \brief
        ///   Creates a new color from a 32-bit integer representation. The order or the components
        ///   must be specified. All components are assumed to be in linear space.
        template<ColorOrder O>
        HK_INLINE static hkColorUbLinear fromUint32(hkUint32 color);

        /// \brief
        ///   Returns a copy of this color with the specified alpha value
        HK_INLINE hkColorUbLinear replaceAlpha(hkUint8 alpha) const;
};


class HK_EXPORT_COMMON hkColors
{
    public:
        static const hkColorUbGamma AliceBlue;              ///< \#F0F8FF
        static const hkColorUbGamma AntiqueWhite;           ///< \#FAEBD7
        static const hkColorUbGamma Aqua;                   ///< \#00FFFF
        static const hkColorUbGamma Aquamarine;             ///< \#7FFFD4
        static const hkColorUbGamma Azure;                  ///< \#F0FFFF
        static const hkColorUbGamma Beige;                  ///< \#F5F5DC
        static const hkColorUbGamma Bisque;                 ///< \#FFE4C4
        static const hkColorUbGamma Black;                  ///< \#000000
        static const hkColorUbGamma BlanchedAlmond;         ///< \#FFEBCD
        static const hkColorUbGamma Blue;                   ///< \#0000FF
        static const hkColorUbGamma BlueViolet;             ///< \#8A2BE2
        static const hkColorUbGamma Brown;                  ///< \#A52A2A
        static const hkColorUbGamma BurlyWood;              ///< \#DEB887
        static const hkColorUbGamma CadetBlue;              ///< \#5F9EA0
        static const hkColorUbGamma Chartreuse;             ///< \#7FFF00
        static const hkColorUbGamma Chocolate;              ///< \#D2691E
        static const hkColorUbGamma Coral;                  ///< \#FF7F50
        static const hkColorUbGamma CornflowerBlue;         ///< \#6495ED  The original!
        static const hkColorUbGamma Cornsilk;               ///< \#FFF8DC
        static const hkColorUbGamma Crimson;                ///< \#DC143C
        static const hkColorUbGamma Cyan;                   ///< \#00FFFF
        static const hkColorUbGamma DarkBlue;               ///< \#00008B
        static const hkColorUbGamma DarkCyan;               ///< \#008B8B
        static const hkColorUbGamma DarkGoldenRod;          ///< \#B8860B
        static const hkColorUbGamma DarkGray;               ///< \#A9A9A9
        static const hkColorUbGamma DarkGreen;              ///< \#006400
        static const hkColorUbGamma DarkKhaki;              ///< \#BDB76B
        static const hkColorUbGamma DarkMagenta;            ///< \#8B008B
        static const hkColorUbGamma DarkOliveGreen;         ///< \#556B2F
        static const hkColorUbGamma DarkOrange;             ///< \#FF8C00
        static const hkColorUbGamma DarkOrchid;             ///< \#9932CC
        static const hkColorUbGamma DarkRed;                ///< \#8B0000
        static const hkColorUbGamma DarkSalmon;             ///< \#E9967A
        static const hkColorUbGamma DarkSeaGreen;           ///< \#8FBC8F
        static const hkColorUbGamma DarkSlateBlue;          ///< \#483D8B
        static const hkColorUbGamma DarkSlateGray;          ///< \#2F4F4F
        static const hkColorUbGamma DarkTurquoise;          ///< \#00CED1
        static const hkColorUbGamma DarkViolet;             ///< \#9400D3
        static const hkColorUbGamma DeepPink;               ///< \#FF1493
        static const hkColorUbGamma DeepSkyBlue;            ///< \#00BFFF
        static const hkColorUbGamma DimGray;                ///< \#696969
        static const hkColorUbGamma DodgerBlue;             ///< \#1E90FF
        static const hkColorUbGamma FireBrick;              ///< \#B22222
        static const hkColorUbGamma FloralWhite;            ///< \#FFFAF0
        static const hkColorUbGamma ForestGreen;            ///< \#228B22
        static const hkColorUbGamma Fuchsia;                ///< \#FF00FF
        static const hkColorUbGamma Gainsboro;              ///< \#DCDCDC
        static const hkColorUbGamma GhostWhite;             ///< \#F8F8FF
        static const hkColorUbGamma Gold;                   ///< \#FFD700
        static const hkColorUbGamma GoldenRod;              ///< \#DAA520
        static const hkColorUbGamma Gray;                   ///< \#808080
        static const hkColorUbGamma Green;                  ///< \#008000
        static const hkColorUbGamma GreenYellow;            ///< \#ADFF2F
        static const hkColorUbGamma HoneyDew;               ///< \#F0FFF0
        static const hkColorUbGamma HotPink;                ///< \#FF69B4
        static const hkColorUbGamma IndianRed;              ///< \#CD5C5C
        static const hkColorUbGamma Indigo;                 ///< \#4B0082
        static const hkColorUbGamma Ivory;                  ///< \#FFFFF0
        static const hkColorUbGamma Khaki;                  ///< \#F0E68C
        static const hkColorUbGamma Lavender;               ///< \#E6E6FA
        static const hkColorUbGamma LavenderBlush;          ///< \#FFF0F5
        static const hkColorUbGamma LawnGreen;              ///< \#7CFC00
        static const hkColorUbGamma LemonChiffon;           ///< \#FFFACD
        static const hkColorUbGamma LightBlue;              ///< \#ADD8E6
        static const hkColorUbGamma LightCoral;             ///< \#F08080
        static const hkColorUbGamma LightCyan;              ///< \#E0FFFF
        static const hkColorUbGamma LightGoldenRodYellow;   ///< \#FAFAD2
        static const hkColorUbGamma LightGray;              ///< \#D3D3D3
        static const hkColorUbGamma LightGreen;             ///< \#90EE90
        static const hkColorUbGamma LightPink;              ///< \#FFB6C1
        static const hkColorUbGamma LightSalmon;            ///< \#FFA07A
        static const hkColorUbGamma LightSeaGreen;          ///< \#20B2AA
        static const hkColorUbGamma LightSkyBlue;           ///< \#87CEFA
        static const hkColorUbGamma LightSlateGray;         ///< \#778899
        static const hkColorUbGamma LightSteelBlue;         ///< \#B0C4DE
        static const hkColorUbGamma LightYellow;            ///< \#FFFFE0
        static const hkColorUbGamma Lime;                   ///< \#00FF00
        static const hkColorUbGamma LimeGreen;              ///< \#32CD32
        static const hkColorUbGamma Linen;                  ///< \#FAF0E6
        static const hkColorUbGamma Magenta;                ///< \#FF00FF
        static const hkColorUbGamma Maroon;                 ///< \#800000
        static const hkColorUbGamma MediumAquaMarine;       ///< \#66CDAA
        static const hkColorUbGamma MediumBlue;             ///< \#0000CD
        static const hkColorUbGamma MediumOrchid;           ///< \#BA55D3
        static const hkColorUbGamma MediumPurple;           ///< \#9370DB
        static const hkColorUbGamma MediumSeaGreen;         ///< \#3CB371
        static const hkColorUbGamma MediumSlateBlue;        ///< \#7B68EE
        static const hkColorUbGamma MediumSpringGreen;      ///< \#00FA9A
        static const hkColorUbGamma MediumTurquoise;        ///< \#48D1CC
        static const hkColorUbGamma MediumVioletRed;        ///< \#C71585
        static const hkColorUbGamma MidnightBlue;           ///< \#191970
        static const hkColorUbGamma MintCream;              ///< \#F5FFFA
        static const hkColorUbGamma MistyRose;              ///< \#FFE4E1
        static const hkColorUbGamma Moccasin;               ///< \#FFE4B5
        static const hkColorUbGamma NavajoWhite;            ///< \#FFDEAD
        static const hkColorUbGamma Navy;                   ///< \#000080
        static const hkColorUbGamma OldLace;                ///< \#FDF5E6
        static const hkColorUbGamma Olive;                  ///< \#808000
        static const hkColorUbGamma OliveDrab;              ///< \#6B8E23
        static const hkColorUbGamma Orange;                 ///< \#FFA500
        static const hkColorUbGamma OrangeRed;              ///< \#FF4500
        static const hkColorUbGamma Orchid;                 ///< \#DA70D6
        static const hkColorUbGamma PaleGoldenRod;          ///< \#EEE8AA
        static const hkColorUbGamma PaleGreen;              ///< \#98FB98
        static const hkColorUbGamma PaleTurquoise;          ///< \#AFEEEE
        static const hkColorUbGamma PaleVioletRed;          ///< \#DB7093
        static const hkColorUbGamma PapayaWhip;             ///< \#FFEFD5
        static const hkColorUbGamma PeachPuff;              ///< \#FFDAB9
        static const hkColorUbGamma Peru;                   ///< \#CD853F
        static const hkColorUbGamma Pink;                   ///< \#FFC0CB
        static const hkColorUbGamma Plum;                   ///< \#DDA0DD
        static const hkColorUbGamma PowderBlue;             ///< \#B0E0E6
        static const hkColorUbGamma Purple;                 ///< \#800080
        static const hkColorUbGamma RebeccaPurple;          ///< \#663399
        static const hkColorUbGamma Red;                    ///< \#FF0000
        static const hkColorUbGamma RosyBrown;              ///< \#BC8F8F
        static const hkColorUbGamma RoyalBlue;              ///< \#4169E1
        static const hkColorUbGamma SaddleBrown;            ///< \#8B4513
        static const hkColorUbGamma Salmon;                 ///< \#FA8072
        static const hkColorUbGamma SandyBrown;             ///< \#F4A460
        static const hkColorUbGamma SeaGreen;               ///< \#2E8B57
        static const hkColorUbGamma SeaShell;               ///< \#FFF5EE
        static const hkColorUbGamma Sienna;                 ///< \#A0522D
        static const hkColorUbGamma Silver;                 ///< \#C0C0C0
        static const hkColorUbGamma SkyBlue;                ///< \#87CEEB
        static const hkColorUbGamma SlateBlue;              ///< \#6A5ACD
        static const hkColorUbGamma SlateGray;              ///< \#708090
        static const hkColorUbGamma Snow;                   ///< \#FFFAFA
        static const hkColorUbGamma SpringGreen;            ///< \#00FF7F
        static const hkColorUbGamma SteelBlue;              ///< \#4682B4
        static const hkColorUbGamma Tan;                    ///< \#D2B48C
        static const hkColorUbGamma Teal;                   ///< \#008080
        static const hkColorUbGamma Thistle;                ///< \#D8BFD8
        static const hkColorUbGamma Tomato;                 ///< \#FF6347
        static const hkColorUbGamma Turquoise;              ///< \#40E0D0
        static const hkColorUbGamma Violet;                 ///< \#EE82EE
        static const hkColorUbGamma Wheat;                  ///< \#F5DEB3
        static const hkColorUbGamma White;                  ///< \#FFFFFF
        static const hkColorUbGamma WhiteSmoke;             ///< \#F5F5F5
        static const hkColorUbGamma Yellow;                 ///< \#FFFF00
        static const hkColorUbGamma YellowGreen;            ///< \#9ACD32

        static const hkColorUbGamma Grey;                   ///< \#888888
        static const hkColorUbGamma Grey25;                 ///< \#404040
        static const hkColorUbGamma Grey50;                 ///< \#808080
        static const hkColorUbGamma Grey75;                 ///< \#C0C0C0

        // Havok product colors
        static const hkColorUbGamma Physics;                ///< \#FFB300
        static const hkColorUbGamma Destruction;            ///< \#DB0020
        static const hkColorUbGamma Animation;              ///< \#02A22B
        static const hkColorUbGamma Behavior;               ///< \#3370B8
        static const hkColorUbGamma Cloth;                  ///< \#B29CDC
        static const hkColorUbGamma Ai;                     ///< \#ACCEF0
        static const hkColorUbGamma Script;                 ///< \#BFB630
};

// Typedef to be used when you need a hkColorf that is also supposed to expose the alpha value in the UI
typedef hkColorf hkColorfAlpha HK_ATTR(hk::Ui_Type(hk::UI_COLOR_PICKER_ALPHA));
HK_REFLECT_TYPEDEF(HK_EXPORT_COMMON, hkColorfAlpha, hkColorfAlpha_Tag);

#include <Common/Base/Types/Color/hkColor.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
