// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

/*
* Copyright 1993, David Koblas (koblas@netcom.com)
* Copyright 1995, 1996 Torsten Martinsen (bullestock@dk-online.dk)

* Permission to use, copy, modify, and to distribute this software and
* its documentation for any purpose is hereby granted without fee, provided
* that the above copyright notice appear in all copies and that both that
* copyright notice and this permission notice appear in supporting documentation.
* There is no representations about the suitability of this software for any purpose.
* This software is provided "as is" without express or implied warranty.
*/

#include <Common/Base/hkBase.h>
#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Math/Vector/hkIntVector.h>

hkColor::Argb hkColor::s_colorTable[32] =
{
    0xffe0e0e0, // grey
    0xffff0000, // red
    0xff00ff00, // green
    0xff0000ff, // blue
    0xffffff00, // yellow
    0xff00ffff, // cyan
    0xffff00ff, // magenta
    0xffffffff, // white

    0xffa0a0a0, // grey
    0xffc00000, // red
    0xff00c000, // green
    0xff0000c0, // blue
    0xffc0c000, // yellow
    0xff00c0c0, // cyan
    0xffc000c0, // magenta
    0xffc0c0c0, // white

    0xff606060, // grey
    0xff800000, // red
    0xff008000, // green
    0xff000080, // blue
    0xff808000, // yellow
    0xff008080, // cyan
    0xff800080, // magenta
    0xff808080, // white

    0xff202020, // grey
    0xff400000, // red
    0xff004000, // green
    0xff000040, // blue
    0xff404000, // yellow
    0xff004040, // cyan
    0xff400040, // magenta
    0xff404040  // white
};

const hkColor::Argb hkColor::MAROON = 0xFF800000;
const hkColor::Argb hkColor::DARKRED = 0xFF8B0000;
const hkColor::Argb hkColor::RED = 0xFFFF0000;
const hkColor::Argb hkColor::LIGHTPINK = 0xFFFFB6C1;
const hkColor::Argb hkColor::CRIMSON = 0xFFDC143C;
const hkColor::Argb hkColor::PALEVIOLETRED = 0xFFDB7093;
const hkColor::Argb hkColor::HOTPINK = 0xFFFF69B4;
const hkColor::Argb hkColor::DEEPPINK = 0xFFFF1493;
const hkColor::Argb hkColor::MEDIUMVIOLETRED = 0xFFC71585;
const hkColor::Argb hkColor::PURPLE = 0xFF800080;
const hkColor::Argb hkColor::DARKMAGENTA = 0xFF8B008B;
const hkColor::Argb hkColor::ORCHID = 0xFFDA70D6;
const hkColor::Argb hkColor::THISTLE = 0xFFD8BFD8;
const hkColor::Argb hkColor::PLUM = 0xFFDDA0DD;
const hkColor::Argb hkColor::VIOLET = 0xFFEE82EE;
const hkColor::Argb hkColor::FUCHSIA = 0xFFFF00FF;
const hkColor::Argb hkColor::MAGENTA = 0xFFFF00FF;
const hkColor::Argb hkColor::MEDIUMORCHID = 0xFFBA55D3;
const hkColor::Argb hkColor::DARKVIOLET = 0xFF9400D3;
const hkColor::Argb hkColor::DARKORCHID = 0xFF9932CC;
const hkColor::Argb hkColor::BLUEVIOLET = 0xFF8A2BE2;
const hkColor::Argb hkColor::INDIGO = 0xFF4B0082;
const hkColor::Argb hkColor::MEDIUMPURPLE = 0xFF9370DB;
const hkColor::Argb hkColor::SLATEBLUE = 0xFF6A5ACD;
const hkColor::Argb hkColor::MEDIUMSLATEBLUE = 0xFF7B68EE;
const hkColor::Argb hkColor::DARKBLUE = 0xFF00008B;
const hkColor::Argb hkColor::MEDIUMBLUE = 0xFF0000CD;
const hkColor::Argb hkColor::BLUE = 0xFF0000FF;
const hkColor::Argb hkColor::NAVY = 0xFF000080;
const hkColor::Argb hkColor::MIDNIGHTBLUE = 0xFF191970;
const hkColor::Argb hkColor::DARKSLATEBLUE = 0xFF483D8B;
const hkColor::Argb hkColor::ROYALBLUE = 0xFF4169E1;
const hkColor::Argb hkColor::CORNFLOWERBLUE = 0xFF6495ED;
const hkColor::Argb hkColor::LIGHTSTEELBLUE = 0xFFB0C4DE;
const hkColor::Argb hkColor::ALICEBLUE = 0xFFF0F8FF;
const hkColor::Argb hkColor::GHOSTWHITE = 0xFFF8F8FF;
const hkColor::Argb hkColor::LAVENDER = 0xFFE6E6FA;
const hkColor::Argb hkColor::DODGERBLUE = 0xFF1E90FF;
const hkColor::Argb hkColor::STEELBLUE = 0xFF4682B4;
const hkColor::Argb hkColor::DEEPSKYBLUE = 0xFF00BFFF;
const hkColor::Argb hkColor::SLATEGRAY = 0xFF708090;
const hkColor::Argb hkColor::LIGHTSLATEGRAY = 0xFF778899;
const hkColor::Argb hkColor::LIGHTSKYBLUE = 0xFF87CEFA;
const hkColor::Argb hkColor::SKYBLUE = 0xFF87CEEB;
const hkColor::Argb hkColor::LIGHTBLUE = 0xFFADD8E6;
const hkColor::Argb hkColor::TEAL = 0xFF008080;
const hkColor::Argb hkColor::DARKCYAN = 0xFF008B8B;
const hkColor::Argb hkColor::DARKTURQUOISE = 0xFF00CED1;
const hkColor::Argb hkColor::CYAN = 0xFF00FFFF;
const hkColor::Argb hkColor::MEDIUMTURQUOISE = 0xFF48D1CC;
const hkColor::Argb hkColor::CADETBLUE = 0xFF5F9EA0;
const hkColor::Argb hkColor::PALETURQUOISE = 0xFFAFEEEE;
const hkColor::Argb hkColor::LIGHTCYAN = 0xFFE0FFFF;
const hkColor::Argb hkColor::AZURE = 0xFFF0FFFF;
const hkColor::Argb hkColor::LIGHTSEAGREEN = 0xFF20B2AA;
const hkColor::Argb hkColor::TURQUOISE = 0xFF40E0D0;
const hkColor::Argb hkColor::POWDERBLUE = 0xFFB0E0E6;
const hkColor::Argb hkColor::DARKSLATEGRAY = 0xFF2F4F4F;
const hkColor::Argb hkColor::AQUAMARINE = 0xFF7FFFD4;
const hkColor::Argb hkColor::MEDIUMSPRINGGREEN = 0xFF00FA9A;
const hkColor::Argb hkColor::MEDIUMAQUAMARINE = 0xFF66CDAA;
const hkColor::Argb hkColor::SPRINGGREEN = 0xFF00FF7F;
const hkColor::Argb hkColor::MEDIUMSEAGREEN = 0xFF3CB371;
const hkColor::Argb hkColor::SEAGREEN = 0xFF2E8B57;
const hkColor::Argb hkColor::LIMEGREEN = 0xFF32CD32;
const hkColor::Argb hkColor::DARKGREEN = 0xFF006400;
const hkColor::Argb hkColor::GREEN = 0xFF008000;
const hkColor::Argb hkColor::LIME = 0xFF00FF00;
const hkColor::Argb hkColor::FORESTGREEN = 0xFF228B22;
const hkColor::Argb hkColor::DARKSEAGREEN = 0xFF8FBC8F;
const hkColor::Argb hkColor::LIGHTGREEN = 0xFF90EE90;
const hkColor::Argb hkColor::PALEGREEN = 0xFF98FB98;
const hkColor::Argb hkColor::MINTCREAM = 0xFFF5FFFA;
const hkColor::Argb hkColor::HONEYDEW = 0xFFF0FFF0;
const hkColor::Argb hkColor::CHARTREUSE = 0xFF7FFF00;
const hkColor::Argb hkColor::LAWNGREEN = 0xFF7CFC00;
const hkColor::Argb hkColor::OLIVEDRAB = 0xFF6B8E23;
const hkColor::Argb hkColor::DARKOLIVEGREEN = 0xFF556B2F;
const hkColor::Argb hkColor::YELLOWGREEN = 0xFF9ACD32;
const hkColor::Argb hkColor::GREENYELLOW = 0xFFADFF2F;
const hkColor::Argb hkColor::BEIGE = 0xFFF5F5DC;
const hkColor::Argb hkColor::LINEN = 0xFFFAF0E6;
const hkColor::Argb hkColor::LIGHTGOLDENRODYELLOW = 0xFFFAFAD2;
const hkColor::Argb hkColor::OLIVE = 0xFF808000;
const hkColor::Argb hkColor::YELLOW = 0xFFFFFF00;
const hkColor::Argb hkColor::LIGHTYELLOW = 0xFFFFFFE0;
const hkColor::Argb hkColor::IVORY = 0xFFFFFFF0;
const hkColor::Argb hkColor::DARKKHAKI = 0xFFBDB76B;
const hkColor::Argb hkColor::KHAKI = 0xFFF0E68C;
const hkColor::Argb hkColor::PALEGOLDENROD = 0xFFEEE8AA;
const hkColor::Argb hkColor::WHEAT = 0xFFF5DEB3;
const hkColor::Argb hkColor::GOLD = 0xFFFFD700;
const hkColor::Argb hkColor::LEMONCHIFFON = 0xFFFFFACD;
const hkColor::Argb hkColor::PAPAYAWHIP = 0xFFFFEFD5;
const hkColor::Argb hkColor::DARKGOLDENROD = 0xFFB8860B;
const hkColor::Argb hkColor::GOLDENROD = 0xFFDAA520;
const hkColor::Argb hkColor::ANTIQUEWHITE = 0xFFFAEBD7;
const hkColor::Argb hkColor::CORNSILK = 0xFFFFF8DC;
const hkColor::Argb hkColor::OLDLACE = 0xFFFDF5E6;
const hkColor::Argb hkColor::MOCCASIN = 0xFFFFE4B5;
const hkColor::Argb hkColor::NAVAJOWHITE = 0xFFFFDEAD;
const hkColor::Argb hkColor::ORANGE = 0xFFFFA500;
const hkColor::Argb hkColor::BISQUE = 0xFFFFE4C4;
const hkColor::Argb hkColor::TAN = 0xFFD2B48C;
const hkColor::Argb hkColor::DARKORANGE = 0xFFFF8C00;
const hkColor::Argb hkColor::BURLYWOOD = 0xFFDEB887;
const hkColor::Argb hkColor::SADDLEBROWN = 0xFF8B4513;
const hkColor::Argb hkColor::SANDYBROWN = 0xFFF4A460;
const hkColor::Argb hkColor::BLANCHEDALMOND = 0xFFFFEBCD;
const hkColor::Argb hkColor::LAVENDERBLUSH = 0xFFFFF0F5;
const hkColor::Argb hkColor::SEASHELL = 0xFFFFF5EE;
const hkColor::Argb hkColor::FLORALWHITE = 0xFFFFFAF0;
const hkColor::Argb hkColor::SNOW = 0xFFFFFAFA;
const hkColor::Argb hkColor::PERU = 0xFFCD853F;
const hkColor::Argb hkColor::PEACHPUFF = 0xFFFFDAB9;
const hkColor::Argb hkColor::CHOCOLATE = 0xFFD2691E;
const hkColor::Argb hkColor::SIENNA = 0xFFA0522D;
const hkColor::Argb hkColor::LIGHTSALMON = 0xFFFFA07A;
const hkColor::Argb hkColor::CORAL = 0xFFFF7F50;
const hkColor::Argb hkColor::DARKSALMON = 0xFFE9967A;
const hkColor::Argb hkColor::MISTYROSE = 0xFFFFE4E1;
const hkColor::Argb hkColor::ORANGERED = 0xFFFF4500;
const hkColor::Argb hkColor::SALMON = 0xFFFA8072;
const hkColor::Argb hkColor::TOMATO = 0xFFFF6347;
const hkColor::Argb hkColor::ROSYBROWN = 0xFFBC8F8F;
const hkColor::Argb hkColor::PINK = 0xFFFFC0CB;
const hkColor::Argb hkColor::INDIANRED = 0xFFCD5C5C;
const hkColor::Argb hkColor::LIGHTCORAL = 0xFFF08080;
const hkColor::Argb hkColor::BROWN = 0xFFA52A2A;
const hkColor::Argb hkColor::FIREBRICK = 0xFFB22222;
const hkColor::Argb hkColor::BLACK = 0xFF000000;
const hkColor::Argb hkColor::DIMGRAY = 0xFF696969;
const hkColor::Argb hkColor::GRAY = 0xFF808080;
const hkColor::Argb hkColor::DARKGRAY = 0xFFA9A9A9;
const hkColor::Argb hkColor::SILVER = 0xFFC0C0C0;
const hkColor::Argb hkColor::LIGHTGREY = 0xFFD3D3D3;
const hkColor::Argb hkColor::GAINSBORO = 0xFFDCDCDC;
const hkColor::Argb hkColor::WHITESMOKE = 0xFFF5F5F5;
const hkColor::Argb hkColor::WHITE = 0xFFFFFFFF;
const hkColor::Argb hkColor::GREY       = 0xff888888;
const hkColor::Argb hkColor::GREY25 = 0xff404040;
const hkColor::Argb hkColor::GREY50 = 0xff808080;
const hkColor::Argb hkColor::GREY75 = 0xffc0c0c0;

// Havok product colors
const hkColor::Argb hkColor::PHYSICS = 0xFFFFB300;
const hkColor::Argb hkColor::DESTRUCTION = 0xFFDB0020;
const hkColor::Argb hkColor::ANIMATION = 0xFF02A22B;
const hkColor::Argb hkColor::BEHAVIOR = 0xFF3370B8;
const hkColor::Argb hkColor::CLOTH = 0xFFB29CDC;
const hkColor::Argb hkColor::AI = 0xFFACCEF0;
const hkColor::Argb hkColor::SCRIPT = 0xFFBFB630;

// Effectively empty color
const hkColor::Argb hkColor::NONE = 0x00FFFFFF;



hkColor::Argb HK_CALL hkColor::rgbFromFloats(const hkReal red, const hkReal green, const hkReal blue, const hkReal alpha)
{
    HK_ASSERT(0x5d228af1,  (alpha >= 0.0f) && (alpha <= 1.0f), "Color component is out of range!" );
    HK_ASSERT(0x293c6824,  (red >= 0.0f) && (red <= 1.0f), "Color component is out of range!" );
    HK_ASSERT(0x3df5f1a3,  (green >= 0.0f) && (green <= 1.0f), "Color component is out of range!" );
    HK_ASSERT(0x293ff0fe,  (blue >= 0.0f) && (blue <= 1.0f), "Color component is out of range!" );

    unsigned char cAlpha = (unsigned char)(hkMath::hkFloatToInt(alpha * 255));
    unsigned char cRed   = (unsigned char)(hkMath::hkFloatToInt(red * 255));
    unsigned char cGreen = (unsigned char)(hkMath::hkFloatToInt(green * 255));
    unsigned char cBlue  = (unsigned char)(hkMath::hkFloatToInt(blue * 255));

    return rgbFromChars(cRed, cGreen, cBlue, cAlpha);
}


// Convert from HSV color values to RGB values (all in range [0,1])
static inline void _HSVtoRGB(_Out_ hkFloat32 *r, _Out_ hkFloat32 *g, _Out_ hkFloat32 *b, hkFloat32 h, hkFloat32 s, hkFloat32 v )
{
    int i; hkFloat32 f, p, q, t;
    if ( s==0.0f )
    {
        *r = *g = *b = v;
        return;
    }
    if ( h==1.0f )
    {
        i=5;
        h=6.0f;
    }
    else
    {
        h *= 6.0f;
        i = static_cast<int>( hkMath::floor(h) );
    }

    f = h - i;
    p = v * (1.0f - s);
    q = v * (1.0f - s*f);
    t = v * (1.0f - s*(1.0f-f));
    switch(i)
    {
        case 0:  *r = v; *g = t; *b = p; break;
        case 1:  *r = q; *g = v; *b = p; break;
        case 2:  *r = p; *g = v; *b = t; break;
        case 3:  *r = p; *g = q; *b = v; break;
        case 4:  *r = t; *g = p; *b = v; break;
        default: *r = v; *g = p; *b = q; break;
    }
}


hkColor::Argb HK_CALL hkColor::rgbFromHSV(const hkReal h, const hkReal s, const hkReal v, const hkReal alpha)
{
    HK_ASSERT(0x5d228af2,  (alpha >= 0.0f) && (alpha <= 1.0f), "Color component is out of range!" );
    HK_ASSERT(0x293c6825,  (h >= 0.0f) && (h <= 1.0f), "Color component is out of range!" );
    HK_ASSERT(0x3df5f1a7,  (s >= 0.0f) && (s <= 1.0f), "Color component is out of range!" );
    HK_ASSERT(0x293ff0fb,  (v >= 0.0f) && (v <= 1.0f), "Color component is out of range!" );

    hkFloat32 r, g, b;
    _HSVtoRGB(&r, &g, &b, hkFloat32(h), hkFloat32(s), hkFloat32(v));

    return rgbFromFloats(r, g, b, alpha);
}

//
//  Generates a set of 'aesthetically pleasing' random colors (the generation is done in HSV space)
//  As described in: http://martin.ankerl.com/2009/12/09/how-to-create-random-colors-programmatically/

void HK_CALL hkColor::generateRandomColors(hkArray<Argb>& colorsOut, hkReal initialHue, hkReal saturation, hkReal value)
{
    const hkReal invGoldenRatio = 0.618033988749895f;
    const int numColors         = colorsOut.getSize();

    hkReal hue = initialHue;
    for (int k = 0; k < numColors; k++)
    {
        hue += invGoldenRatio;
        const int iHue = int(hue);
        hue -= hkReal(iHue);

        colorsOut[k] = rgbFromHSV(hue, saturation, value);
    }
}

void HK_CALL hkColor::generateRandomColors(hkArray<hkColorf>& colorsOut, hkReal initialHue, hkReal saturation, hkReal value)
{
    const hkReal invGoldenRatio = 0.618033988749895f;
    const int numColors = colorsOut.getSize();

    hkReal hue = initialHue;
    for (int k = 0; k < numColors; k++)
    {
        hue += invGoldenRatio;
        const int iHue = int(hue);
        hue -= hkReal(iHue);

        hkFloat32 r, g, b;
        _HSVtoRGB(&r, &g, &b, hkFloat32(hue), hkFloat32(saturation), hkFloat32(value));

        colorsOut[k].set(r, g, b, 1.0f);
    }
}

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    static hkPseudoRandomGenerator getRandomColor_prng(0);
}

hkColor::Argb HK_CALL hkColor::getRandomColor()
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getRandomColor(getRandomColor_prng);
}

hkColor::Argb HK_CALL hkColor::getRandomColor( hkPseudoRandomGenerator& rand)
{
    hkReal r = rand.getRandRange(0.0f, 1.0f);
    hkReal g = rand.getRandRange(0.0f, 1.0f);
    hkReal b = rand.getRandRange(0.0f, 1.0f);
    hkReal a = 1.0f;

    return rgbFromFloats(r,g,b,a);
}

hkColor::Argb HK_CALL hkColor::getSpectrumColor(hkReal value)
{
    hkVector4   palette[6];
    palette[0].set(0,0,0,1);
    palette[1].set(0,0,1,1);
    palette[2].set(0,1,1,1);
    palette[3].set(0,1,0,1);
    palette[4].set(1,1,0,1);
    palette[5].set(1,0,0,1);
    hkVector4       color; color.setZero();
    if(value <= 0.0f)       color = palette[0];
    else if(value >= 1.0f)  color = palette[5];
    else
    {
        hkReal      ms = value * 5.0f;
        int         c = (int) ms;
        hkSimdReal  u; u.setFromFloat(ms - c);
        color.setInterpolate(palette[c], palette[c+1], u);
    }
    return hkColor::rgbFromFloats(color(0),color(1),color(2),1);
}

hkColor::Argb HK_CALL hkColor::getPaletteColor(int i, unsigned char alpha)
{
    return (s_colorTable[i % HK_COUNT_OF(s_colorTable)] & 0x00ffffffU) | (static_cast<hkColor::Argb>( alpha )<<24);
}


// ----------------------------------------------------------------------------
//  hkColorf
// ----------------------------------------------------------------------------
void hkColorf::setFromHSV(hkFloat32 h, hkFloat32 s, hkFloat32 v, hkFloat32 alpha)
{
    // Make sure hue is within range (don't simply clamp, as it describes a circle and
    // values outside [0.0, 1.0[ wrap around)
    hkFloat32 dummy;
    h = modff(h, &dummy);
    if (h < 0.0f)
    {
        h += 1.0f;
    }

    hkFloat32 r, g, b;
    _HSVtoRGB(&r, &g, &b, h, hkMath::clamp(s, 0.0f, 1.0f), hkMath::clamp(v, 0.0f, 1.0f));

    set(r, g, b, alpha);
}

void hkColorf::lighten(hkFloat32 amount)
{
    float factor = hkMath::max2(1.0f, 1.0f + amount);
    hkVector4f v; v.set(factor, factor, factor, 1.0f);
    hkVector4f c; c.setMul(*this, v);
    setClampedZeroOne(c);
}

void hkColorf::darken(hkFloat32 amount)
{
    float factor = hkMath::clamp(1.0f - amount, 0.0f, 1.0f);
    hkVector4f v; v.set(factor, factor, factor, 1.0f);
    hkVector4f c; c.setMul(*this, v);
    setClampedZeroOne(c);
}

hkFloat32 hkColorf::calcLuminance() const
{
    return getRed() * 0.2126f + getGreen() * 0.7152f + getBlue() * 0.0722f;
}

hkFloat32 hkColorf::calcAverage() const
{
    return (getRed() + getGreen() + getBlue()) * 0.333333f;
}

hkFloat32 hkColorf::gammaToLinear(hkFloat32 f)
{
    // Fast and accurate sRGB curve, inspired by http://lists.blender.org/pipermail/bf-blender-cvs/2014-January/061278.html
    //
    // The linear part is computed as usual. The tricky part is computing
    // x^2.4. This is done by deriving x^0.8 via 3 Newton iterations as the
    // 5th root of x^4 (see http://en.wikipedia.org/wiki/Nth_root_algorithm)
    // starting from an initial approximation by reinterpreting the float
    // as int (similar to the inverse square root trick), and cubing
    // to get x^2.4.
    //
    // Constant factors during the Newton iterations have been mechanically
    // moved around to minimize the number of operations, resulting in the
    // weird factors below.
    //
    // Maximum error in the range [0, 1] is about 3.5e-7 compared to MSVC's powf,
    // while being roughly twice as fast.
    if (f <= 0.04045f)
    {
        return f * (1.0f / 12.92f);
    }

    f += 0.055f;

    union
    {
        hkFloat32 f;
        hkUint32 i;
    } val;

    // Initial approximation via magic
    hkFloat32 x = f * 3611622602.84f; // 2^(127/0.8 - 127))

    val.f = x;
    x = hkFloat32(val.i);
    x = x * 0.8f;
    val.i = int(x);
    x = val.f;

    hkFloat32 f2 = f * f;
    hkFloat32 f4 = f2 * f2;

    // Newton iter 1
    hkFloat32 x2 = x * x;
    hkFloat32 x4 = x2 * x2;
    hkFloat32 t = f4 / x4;
    x = 4.0f * x + t;

    // Newton iter 2
    x2 = x * x;
    x4 = x2 * x2;
    t = f4 / x4;
    x = 0.00128f * x + t;

    // Newton iter 3
    x2 = x * x;
    x4 = x2 * x2;
    t = f4 / x4;
    x = 122070312500.0f * x + t;

    // Cube and multiply in any remaining factors
    return (x * 4.834637627e-28f) * (x * x);
}

hkFloat32 hkColorf::linearToGamma(hkFloat32 f)
{
    // Maximum error in the range [0, 1] is about 3e-7 compared to MSVC's powf,
    // while being roughly 1.5 times as fast.

    if (f <= 0.0031308f)
    {
        return f * 12.92f;
    }

    // Initial approximation via magic
    union
    {
        hkFloat32 f;
        hkUint32 i;
    } val;

    val.f = f;
    hkUint32 i = val.i;
    i = (i + (127 << 24)) / 3;
    val.i = i;

    hkFloat32 x = val.f;

    // Two iterations of Halley's method for x ^ (1/3)
    hkFloat32 x3 = x * x * x;
    hkFloat32 x3f = x3 + f;
    x = x * ((x3f + f) / (x3f + x3));
    x3 = x * x * x;
    x3f = x3 + f;
    x = x * ((x3f + f) / (x3f + x3));

    // Square root twice to find x ^ (1/12)
    x = hkMath::sqrt(hkMath::sqrt(x));

    // Raise to 5th power (x ^ 5/12 = x ^ 1/2.4)
    hkFloat32 x2 = x * x;
    hkFloat32 x4 = x2 * x2;
    return x4 * x * 1.055f - 0.055f;
}


// ----------------------------------------------------------------------------
//  hkColorUbBase
// ----------------------------------------------------------------------------
const hkUint8 hkColorUbBase::m_gammaToLinearLut[256] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
    0x04, 0x04, 0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0x05, 0x06, 0x06, 0x06, 0x06, 0x07, 0x07, 0x07,
    0x08, 0x08, 0x08, 0x08, 0x09, 0x09, 0x09, 0x0a, 0x0a, 0x0a, 0x0b, 0x0b, 0x0c, 0x0c, 0x0c, 0x0d,
    0x0d, 0x0d, 0x0e, 0x0e, 0x0f, 0x0f, 0x10, 0x10, 0x11, 0x11, 0x11, 0x12, 0x12, 0x13, 0x13, 0x14,
    0x14, 0x15, 0x16, 0x16, 0x17, 0x17, 0x18, 0x18, 0x19, 0x19, 0x1a, 0x1b, 0x1b, 0x1c, 0x1d, 0x1d,
    0x1e, 0x1e, 0x1f, 0x20, 0x20, 0x21, 0x22, 0x23, 0x23, 0x24, 0x25, 0x25, 0x26, 0x27, 0x28, 0x29,
    0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2d, 0x2e, 0x2f, 0x30, 0x31, 0x32, 0x33, 0x33, 0x34, 0x35, 0x36,
    0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46,
    0x47, 0x48, 0x49, 0x4a, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x54, 0x55, 0x56, 0x57, 0x58,
    0x5a, 0x5b, 0x5c, 0x5d, 0x5f, 0x60, 0x61, 0x63, 0x64, 0x65, 0x67, 0x68, 0x69, 0x6b, 0x6c, 0x6d,
    0x6f, 0x70, 0x72, 0x73, 0x74, 0x76, 0x77, 0x79, 0x7a, 0x7c, 0x7d, 0x7f, 0x80, 0x82, 0x83, 0x85,
    0x86, 0x88, 0x8a, 0x8b, 0x8d, 0x8e, 0x90, 0x92, 0x93, 0x95, 0x97, 0x98, 0x9a, 0x9c, 0x9d, 0x9f,
    0xa1, 0xa3, 0xa4, 0xa6, 0xa8, 0xaa, 0xab, 0xad, 0xaf, 0xb1, 0xb3, 0xb5, 0xb7, 0xb8, 0xba, 0xbc,
    0xbe, 0xc0, 0xc2, 0xc4, 0xc6, 0xc8, 0xca, 0xcc, 0xce, 0xd0, 0xd2, 0xd4, 0xd6, 0xd8, 0xda, 0xdc,
    0xde, 0xe0, 0xe2, 0xe5, 0xe7, 0xe9, 0xeb, 0xed, 0xef, 0xf2, 0xf4, 0xf6, 0xf8, 0xfa, 0xfd, 0xff,
};

const hkUint8 hkColorUbBase::m_linearToGammaLut[256] = {
    0x00, 0x0d, 0x16, 0x1c, 0x22, 0x26, 0x2a, 0x2e, 0x32, 0x35, 0x38, 0x3b, 0x3d, 0x40, 0x42, 0x45,
    0x47, 0x49, 0x4b, 0x4d, 0x4f, 0x51, 0x53, 0x55, 0x56, 0x58, 0x5a, 0x5c, 0x5d, 0x5f, 0x60, 0x62,
    0x63, 0x65, 0x66, 0x68, 0x69, 0x6a, 0x6c, 0x6d, 0x6e, 0x70, 0x71, 0x72, 0x73, 0x75, 0x76, 0x77,
    0x78, 0x79, 0x7a, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88,
    0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91, 0x92, 0x93, 0x94, 0x94, 0x95, 0x96, 0x97,
    0x98, 0x99, 0x9a, 0x9b, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa3, 0xa4,
    0xa5, 0xa6, 0xa7, 0xa7, 0xa8, 0xa9, 0xaa, 0xaa, 0xab, 0xac, 0xad, 0xad, 0xae, 0xaf, 0xaf, 0xb0,
    0xb1, 0xb2, 0xb2, 0xb3, 0xb4, 0xb4, 0xb5, 0xb6, 0xb6, 0xb7, 0xb8, 0xb9, 0xb9, 0xba, 0xbb, 0xbb,
    0xbc, 0xbd, 0xbd, 0xbe, 0xbe, 0xbf, 0xc0, 0xc0, 0xc1, 0xc2, 0xc2, 0xc3, 0xc4, 0xc4, 0xc5, 0xc5,
    0xc6, 0xc7, 0xc7, 0xc8, 0xc8, 0xc9, 0xca, 0xca, 0xcb, 0xcb, 0xcc, 0xcd, 0xcd, 0xce, 0xce, 0xcf,
    0xd0, 0xd0, 0xd1, 0xd1, 0xd2, 0xd2, 0xd3, 0xd4, 0xd4, 0xd5, 0xd5, 0xd6, 0xd6, 0xd7, 0xd7, 0xd8,
    0xd8, 0xd9, 0xda, 0xda, 0xdb, 0xdb, 0xdc, 0xdc, 0xdd, 0xdd, 0xde, 0xde, 0xdf, 0xdf, 0xe0, 0xe0,
    0xe1, 0xe2, 0xe2, 0xe3, 0xe3, 0xe4, 0xe4, 0xe5, 0xe5, 0xe6, 0xe6, 0xe7, 0xe7, 0xe8, 0xe8, 0xe9,
    0xe9, 0xea, 0xea, 0xeb, 0xeb, 0xec, 0xec, 0xed, 0xed, 0xee, 0xee, 0xee, 0xef, 0xef, 0xf0, 0xf0,
    0xf1, 0xf1, 0xf2, 0xf2, 0xf3, 0xf3, 0xf4, 0xf4, 0xf5, 0xf5, 0xf6, 0xf6, 0xf6, 0xf7, 0xf7, 0xf8,
    0xf8, 0xf9, 0xf9, 0xfa, 0xfa, 0xfb, 0xfb, 0xfb, 0xfc, 0xfc, 0xfd, 0xfd, 0xfe, 0xfe, 0xff, 0xff,
};


// ----------------------------------------------------------------------------
//  hkColorUbGamma
// ----------------------------------------------------------------------------
hkColorUbGamma hkColorUbGamma::s_palette[32] =
{
    hkColorUbGamma(0xe0, 0xe0, 0xe0), // grey
    hkColorUbGamma(0xff, 0x00, 0x00), // red
    hkColorUbGamma(0x00, 0xff, 0x00), // green
    hkColorUbGamma(0x00, 0x00, 0xff), // blue
    hkColorUbGamma(0xff, 0xff, 0x00), // yellow
    hkColorUbGamma(0x00, 0xff, 0xff), // cyan
    hkColorUbGamma(0xff, 0x00, 0xff), // magenta
    hkColorUbGamma(0xff, 0xff, 0xff), // white

    hkColorUbGamma(0xa0, 0xa0, 0xa0), // grey
    hkColorUbGamma(0xc0, 0x00, 0x00), // red
    hkColorUbGamma(0x00, 0xc0, 0x00), // green
    hkColorUbGamma(0x00, 0x00, 0xc0), // blue
    hkColorUbGamma(0xc0, 0xc0, 0x00), // yellow
    hkColorUbGamma(0x00, 0xc0, 0xc0), // cyan
    hkColorUbGamma(0xc0, 0x00, 0xc0), // magenta
    hkColorUbGamma(0xc0, 0xc0, 0xc0), // white

    hkColorUbGamma(0x60, 0x60, 0x60), // grey
    hkColorUbGamma(0x80, 0x00, 0x00), // red
    hkColorUbGamma(0x00, 0x80, 0x00), // green
    hkColorUbGamma(0x00, 0x00, 0x80), // blue
    hkColorUbGamma(0x80, 0x80, 0x00), // yellow
    hkColorUbGamma(0x00, 0x80, 0x80), // cyan
    hkColorUbGamma(0x80, 0x00, 0x80), // magenta
    hkColorUbGamma(0x80, 0x80, 0x80), // white

    hkColorUbGamma(0x20, 0x20, 0x20), // grey
    hkColorUbGamma(0x40, 0x00, 0x00), // red
    hkColorUbGamma(0x00, 0x40, 0x00), // green
    hkColorUbGamma(0x00, 0x00, 0x40), // blue
    hkColorUbGamma(0x40, 0x40, 0x00), // yellow
    hkColorUbGamma(0x00, 0x40, 0x40), // cyan
    hkColorUbGamma(0x40, 0x00, 0x40), // magenta
    hkColorUbGamma(0x40, 0x40, 0x40)  // white
};

hkColorUbGamma hkColorUbGamma::getPaletteColor(int i, unsigned char alpha)
{
    return s_palette[i % HK_COUNT_OF(s_palette)].replaceAlpha(alpha);
}

hkColorUbGamma hkColorUbGamma::getRandomColor(hkPseudoRandomGenerator& rand)
{
    hkFloat32 r = hkFloat32(rand.getRandRange(0.0f, 1.0f));
    hkFloat32 g = hkFloat32(rand.getRandRange(0.0f, 1.0f));
    hkFloat32 b = hkFloat32(rand.getRandRange(0.0f, 1.0f));

    return hkColorUbGamma(
        hkColorf::floatToUint8(r),
        hkColorf::floatToUint8(g),
        hkColorf::floatToUint8(b));
}


// ----------------------------------------------------------------------------
//  hkColors
// ----------------------------------------------------------------------------
const hkColorUbGamma hkColors::AliceBlue(0xF0, 0xF8, 0xFF);
const hkColorUbGamma hkColors::AntiqueWhite(0xFA, 0xEB, 0xD7);
const hkColorUbGamma hkColors::Aqua(0x00, 0xFF, 0xFF);
const hkColorUbGamma hkColors::Aquamarine(0x7F, 0xFF, 0xD4);
const hkColorUbGamma hkColors::Azure(0xF0, 0xFF, 0xFF);
const hkColorUbGamma hkColors::Beige(0xF5, 0xF5, 0xDC);
const hkColorUbGamma hkColors::Bisque(0xFF, 0xE4, 0xC4);
const hkColorUbGamma hkColors::Black(0x00, 0x00, 0x00);
const hkColorUbGamma hkColors::BlanchedAlmond(0xFF, 0xEB, 0xCD);
const hkColorUbGamma hkColors::Blue(0x00, 0x00, 0xFF);
const hkColorUbGamma hkColors::BlueViolet(0x8A, 0x2B, 0xE2);
const hkColorUbGamma hkColors::Brown(0xA5, 0x2A, 0x2A);
const hkColorUbGamma hkColors::BurlyWood(0xDE, 0xB8, 0x87);
const hkColorUbGamma hkColors::CadetBlue(0x5F, 0x9E, 0xA0);
const hkColorUbGamma hkColors::Chartreuse(0x7F, 0xFF, 0x00);
const hkColorUbGamma hkColors::Chocolate(0xD2, 0x69, 0x1E);
const hkColorUbGamma hkColors::Coral(0xFF, 0x7F, 0x50);
const hkColorUbGamma hkColors::CornflowerBlue(0x64, 0x95, 0xED);
const hkColorUbGamma hkColors::Cornsilk(0xFF, 0xF8, 0xDC);
const hkColorUbGamma hkColors::Crimson(0xDC, 0x14, 0x3C);
const hkColorUbGamma hkColors::Cyan(0x00, 0xFF, 0xFF);
const hkColorUbGamma hkColors::DarkBlue(0x00, 0x00, 0x8B);
const hkColorUbGamma hkColors::DarkCyan(0x00, 0x8B, 0x8B);
const hkColorUbGamma hkColors::DarkGoldenRod(0xB8, 0x86, 0x0B);
const hkColorUbGamma hkColors::DarkGray(0xA9, 0xA9, 0xA9);
const hkColorUbGamma hkColors::DarkGreen(0x00, 0x64, 0x00);
const hkColorUbGamma hkColors::DarkKhaki(0xBD, 0xB7, 0x6B);
const hkColorUbGamma hkColors::DarkMagenta(0x8B, 0x00, 0x8B);
const hkColorUbGamma hkColors::DarkOliveGreen(0x55, 0x6B, 0x2F);
const hkColorUbGamma hkColors::DarkOrange(0xFF, 0x8C, 0x00);
const hkColorUbGamma hkColors::DarkOrchid(0x99, 0x32, 0xCC);
const hkColorUbGamma hkColors::DarkRed(0x8B, 0x00, 0x00);
const hkColorUbGamma hkColors::DarkSalmon(0xE9, 0x96, 0x7A);
const hkColorUbGamma hkColors::DarkSeaGreen(0x8F, 0xBC, 0x8F);
const hkColorUbGamma hkColors::DarkSlateBlue(0x48, 0x3D, 0x8B);
const hkColorUbGamma hkColors::DarkSlateGray(0x2F, 0x4F, 0x4F);
const hkColorUbGamma hkColors::DarkTurquoise(0x00, 0xCE, 0xD1);
const hkColorUbGamma hkColors::DarkViolet(0x94, 0x00, 0xD3);
const hkColorUbGamma hkColors::DeepPink(0xFF, 0x14, 0x93);
const hkColorUbGamma hkColors::DeepSkyBlue(0x00, 0xBF, 0xFF);
const hkColorUbGamma hkColors::DimGray(0x69, 0x69, 0x69);
const hkColorUbGamma hkColors::DodgerBlue(0x1E, 0x90, 0xFF);
const hkColorUbGamma hkColors::FireBrick(0xB2, 0x22, 0x22);
const hkColorUbGamma hkColors::FloralWhite(0xFF, 0xFA, 0xF0);
const hkColorUbGamma hkColors::ForestGreen(0x22, 0x8B, 0x22);
const hkColorUbGamma hkColors::Fuchsia(0xFF, 0x00, 0xFF);
const hkColorUbGamma hkColors::Gainsboro(0xDC, 0xDC, 0xDC);
const hkColorUbGamma hkColors::GhostWhite(0xF8, 0xF8, 0xFF);
const hkColorUbGamma hkColors::Gold(0xFF, 0xD7, 0x00);
const hkColorUbGamma hkColors::GoldenRod(0xDA, 0xA5, 0x20);
const hkColorUbGamma hkColors::Gray(0x80, 0x80, 0x80);
const hkColorUbGamma hkColors::Green(0x00, 0x80, 0x00);
const hkColorUbGamma hkColors::GreenYellow(0xAD, 0xFF, 0x2F);
const hkColorUbGamma hkColors::HoneyDew(0xF0, 0xFF, 0xF0);
const hkColorUbGamma hkColors::HotPink(0xFF, 0x69, 0xB4);
const hkColorUbGamma hkColors::IndianRed(0xCD, 0x5C, 0x5C);
const hkColorUbGamma hkColors::Indigo(0x4B, 0x00, 0x82);
const hkColorUbGamma hkColors::Ivory(0xFF, 0xFF, 0xF0);
const hkColorUbGamma hkColors::Khaki(0xF0, 0xE6, 0x8C);
const hkColorUbGamma hkColors::Lavender(0xE6, 0xE6, 0xFA);
const hkColorUbGamma hkColors::LavenderBlush(0xFF, 0xF0, 0xF5);
const hkColorUbGamma hkColors::LawnGreen(0x7C, 0xFC, 0x00);
const hkColorUbGamma hkColors::LemonChiffon(0xFF, 0xFA, 0xCD);
const hkColorUbGamma hkColors::LightBlue(0xAD, 0xD8, 0xE6);
const hkColorUbGamma hkColors::LightCoral(0xF0, 0x80, 0x80);
const hkColorUbGamma hkColors::LightCyan(0xE0, 0xFF, 0xFF);
const hkColorUbGamma hkColors::LightGoldenRodYellow(0xFA, 0xFA, 0xD2);
const hkColorUbGamma hkColors::LightGray(0xD3, 0xD3, 0xD3);
const hkColorUbGamma hkColors::LightGreen(0x90, 0xEE, 0x90);
const hkColorUbGamma hkColors::LightPink(0xFF, 0xB6, 0xC1);
const hkColorUbGamma hkColors::LightSalmon(0xFF, 0xA0, 0x7A);
const hkColorUbGamma hkColors::LightSeaGreen(0x20, 0xB2, 0xAA);
const hkColorUbGamma hkColors::LightSkyBlue(0x87, 0xCE, 0xFA);
const hkColorUbGamma hkColors::LightSlateGray(0x77, 0x88, 0x99);
const hkColorUbGamma hkColors::LightSteelBlue(0xB0, 0xC4, 0xDE);
const hkColorUbGamma hkColors::LightYellow(0xFF, 0xFF, 0xE0);
const hkColorUbGamma hkColors::Lime(0x00, 0xFF, 0x00);
const hkColorUbGamma hkColors::LimeGreen(0x32, 0xCD, 0x32);
const hkColorUbGamma hkColors::Linen(0xFA, 0xF0, 0xE6);
const hkColorUbGamma hkColors::Magenta(0xFF, 0x00, 0xFF);
const hkColorUbGamma hkColors::Maroon(0x80, 0x00, 0x00);
const hkColorUbGamma hkColors::MediumAquaMarine(0x66, 0xCD, 0xAA);
const hkColorUbGamma hkColors::MediumBlue(0x00, 0x00, 0xCD);
const hkColorUbGamma hkColors::MediumOrchid(0xBA, 0x55, 0xD3);
const hkColorUbGamma hkColors::MediumPurple(0x93, 0x70, 0xDB);
const hkColorUbGamma hkColors::MediumSeaGreen(0x3C, 0xB3, 0x71);
const hkColorUbGamma hkColors::MediumSlateBlue(0x7B, 0x68, 0xEE);
const hkColorUbGamma hkColors::MediumSpringGreen(0x00, 0xFA, 0x9A);
const hkColorUbGamma hkColors::MediumTurquoise(0x48, 0xD1, 0xCC);
const hkColorUbGamma hkColors::MediumVioletRed(0xC7, 0x15, 0x85);
const hkColorUbGamma hkColors::MidnightBlue(0x19, 0x19, 0x70);
const hkColorUbGamma hkColors::MintCream(0xF5, 0xFF, 0xFA);
const hkColorUbGamma hkColors::MistyRose(0xFF, 0xE4, 0xE1);
const hkColorUbGamma hkColors::Moccasin(0xFF, 0xE4, 0xB5);
const hkColorUbGamma hkColors::NavajoWhite(0xFF, 0xDE, 0xAD);
const hkColorUbGamma hkColors::Navy(0x00, 0x00, 0x80);
const hkColorUbGamma hkColors::OldLace(0xFD, 0xF5, 0xE6);
const hkColorUbGamma hkColors::Olive(0x80, 0x80, 0x00);
const hkColorUbGamma hkColors::OliveDrab(0x6B, 0x8E, 0x23);
const hkColorUbGamma hkColors::Orange(0xFF, 0xA5, 0x00);
const hkColorUbGamma hkColors::OrangeRed(0xFF, 0x45, 0x00);
const hkColorUbGamma hkColors::Orchid(0xDA, 0x70, 0xD6);
const hkColorUbGamma hkColors::PaleGoldenRod(0xEE, 0xE8, 0xAA);
const hkColorUbGamma hkColors::PaleGreen(0x98, 0xFB, 0x98);
const hkColorUbGamma hkColors::PaleTurquoise(0xAF, 0xEE, 0xEE);
const hkColorUbGamma hkColors::PaleVioletRed(0xDB, 0x70, 0x93);
const hkColorUbGamma hkColors::PapayaWhip(0xFF, 0xEF, 0xD5);
const hkColorUbGamma hkColors::PeachPuff(0xFF, 0xDA, 0xB9);
const hkColorUbGamma hkColors::Peru(0xCD, 0x85, 0x3F);
const hkColorUbGamma hkColors::Pink(0xFF, 0xC0, 0xCB);
const hkColorUbGamma hkColors::Plum(0xDD, 0xA0, 0xDD);
const hkColorUbGamma hkColors::PowderBlue(0xB0, 0xE0, 0xE6);
const hkColorUbGamma hkColors::Purple(0x80, 0x00, 0x80);
const hkColorUbGamma hkColors::RebeccaPurple(0x66, 0x33, 0x99);
const hkColorUbGamma hkColors::Red(0xFF, 0x00, 0x00);
const hkColorUbGamma hkColors::RosyBrown(0xBC, 0x8F, 0x8F);
const hkColorUbGamma hkColors::RoyalBlue(0x41, 0x69, 0xE1);
const hkColorUbGamma hkColors::SaddleBrown(0x8B, 0x45, 0x13);
const hkColorUbGamma hkColors::Salmon(0xFA, 0x80, 0x72);
const hkColorUbGamma hkColors::SandyBrown(0xF4, 0xA4, 0x60);
const hkColorUbGamma hkColors::SeaGreen(0x2E, 0x8B, 0x57);
const hkColorUbGamma hkColors::SeaShell(0xFF, 0xF5, 0xEE);
const hkColorUbGamma hkColors::Sienna(0xA0, 0x52, 0x2D);
const hkColorUbGamma hkColors::Silver(0xC0, 0xC0, 0xC0);
const hkColorUbGamma hkColors::SkyBlue(0x87, 0xCE, 0xEB);
const hkColorUbGamma hkColors::SlateBlue(0x6A, 0x5A, 0xCD);
const hkColorUbGamma hkColors::SlateGray(0x70, 0x80, 0x90);
const hkColorUbGamma hkColors::Snow(0xFF, 0xFA, 0xFA);
const hkColorUbGamma hkColors::SpringGreen(0x00, 0xFF, 0x7F);
const hkColorUbGamma hkColors::SteelBlue(0x46, 0x82, 0xB4);
const hkColorUbGamma hkColors::Tan(0xD2, 0xB4, 0x8C);
const hkColorUbGamma hkColors::Teal(0x00, 0x80, 0x80);
const hkColorUbGamma hkColors::Thistle(0xD8, 0xBF, 0xD8);
const hkColorUbGamma hkColors::Tomato(0xFF, 0x63, 0x47);
const hkColorUbGamma hkColors::Turquoise(0x40, 0xE0, 0xD0);
const hkColorUbGamma hkColors::Violet(0xEE, 0x82, 0xEE);
const hkColorUbGamma hkColors::Wheat(0xF5, 0xDE, 0xB3);
const hkColorUbGamma hkColors::White(0xFF, 0xFF, 0xFF);
const hkColorUbGamma hkColors::WhiteSmoke(0xF5, 0xF5, 0xF5);
const hkColorUbGamma hkColors::Yellow(0xFF, 0xFF, 0x00);
const hkColorUbGamma hkColors::YellowGreen(0x9A, 0xCD, 0x32);


const hkColorUbGamma hkColors::Grey(0x88, 0x88, 0x88);
const hkColorUbGamma hkColors::Grey25(0x40, 0x40, 0x40);
const hkColorUbGamma hkColors::Grey50(0x80, 0x80, 0x80);
const hkColorUbGamma hkColors::Grey75(0xc0, 0xc0, 0xc0);

// Havok product colors
const hkColorUbGamma hkColors::Physics(0xFF, 0xB3, 0x00);
const hkColorUbGamma hkColors::Destruction(0xDB, 0x00, 0x20);
const hkColorUbGamma hkColors::Animation(0x02, 0xA2, 0x2B);
const hkColorUbGamma hkColors::Behavior(0x33, 0x70, 0xB8);
const hkColorUbGamma hkColors::Cloth(0xB2, 0x9C, 0xDC);
const hkColorUbGamma hkColors::Ai(0xAC, 0xCE, 0xF0);
const hkColorUbGamma hkColors::Script(0xBF, 0xB6, 0x30);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
