// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM       : ALL
// PRODUCT        : COMMON
// VISIBILITY     : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Thread/Atomic/hkAtomicPrimitives.h>
#include <Common/Base/Container/BitField/hkBitField.h>


/// A thread safe, lock-free, pool of integers.
/// Note: CAPACITY must be a power of two.
template<hkUint32 CAPACITY>
class hkThreadSafeIntegerPool
{
    public:

        static_assert((CAPACITY & (CAPACITY - 1)) == 0, "CAPACITY must be a power of 2");
        enum
        {
            NUM_VALUE_BITS = hkMath::Log2<CAPACITY>::ANSWER+1,  // including the extra "invalid" value
            MASK_VALUE = (1<<(NUM_VALUE_BITS))-1,               // also used as the "invalid" value
            MASK_SERIAL = ~MASK_VALUE
        };

    public:

        hkThreadSafeIntegerPool()
        {
            HK_COMPILE_TIME_ASSERT( CAPACITY && !(CAPACITY & (CAPACITY - 1)) ); // check power of 2
            reset();
        }

        ~hkThreadSafeIntegerPool()
        {
            HK_ON_DEBUG( checkConsistency(); );
        }

        /// Reset the pool (not thread safe).
        void reset()
        {
            for( hkUint32 i=0; i<CAPACITY; ++i )
            {
                m_nextFree[i] = ( i < CAPACITY-1 ) ? i+1 : int(MASK_VALUE);
            }
            m_serialAndHead = 0;
        }

        /// Get the number of integers in the pool.
        HK_INLINE hkUint32 getCapacity() const
        {
            return CAPACITY;
        }

        /// Borrow an integer from the pool (thread safe).
        /// Asserts if there are no free integers available.
        HK_INLINE int allocate()
        {
            // Atomically set the head of the linked list to the next free index.
            // Compare the serial number too to avoid ABA problems. (http://en.wikipedia.org/wiki/ABA_problem)
            hkUint32* addr = &m_serialAndHead;
            hkUint32 oldSerialAndHead;
            hkUint32 newSerialAndHead;
            for (hkAtomic::Backoff<> b; ; b.pause())
            {
                oldSerialAndHead = *addr;
                newSerialAndHead = (oldSerialAndHead & MASK_SERIAL) + (MASK_VALUE + 1); // increment serial

                hkAtomic::readWriteBarrier();
                _Analysis_assume_((oldSerialAndHead & MASK_VALUE) < CAPACITY);
                newSerialAndHead |= m_nextFree[oldSerialAndHead & MASK_VALUE];          // replace head

                if ( hkAtomic::compareAndSwap( addr, oldSerialAndHead, newSerialAndHead ) )
                {
                    // Return the old head as our result.
                    const int i = oldSerialAndHead & MASK_VALUE;
                    HK_ASSERT( 0xa7f1546e, i >= 0 && i < int(CAPACITY) && i != MASK_VALUE, "Invalid index" );

                    m_nextFree[i] = MASK_VALUE;     // mark as allocated
                    hkAtomic::readWriteBarrier();   // Make sure m_nextFree[i] is written into memory before being read by allocate()

                    return i;
                }
            }
        }

        /// Return an integer to the pool (thread safe).
        HK_INLINE void free( int i )
        {
            // Make sure there are no pending writes of m_nextFree[i]
            HK_ON_DEBUG(hkAtomic::readWriteBarrier());
            hkUint32 volatile* nextFreeAddr = &m_nextFree[i];

            HK_ASSERT( 0xc7f21464, i >= 0 && i < int(CAPACITY), "Index out of range" );
            HK_ASSERT( 0xc7f21465, *nextFreeAddr == MASK_VALUE, "That index was not allocated" );

            // Atomically set the given index as the head of the linked list.
            hkUint32* addr = &m_serialAndHead;
            hkUint32 oldSerialAndHead;
            hkUint32 newSerialAndHead;
            for (hkAtomic::Backoff<> b; ; b.pause())
            {
                oldSerialAndHead = *addr;
                newSerialAndHead = (oldSerialAndHead & MASK_SERIAL) | i;

                *nextFreeAddr = oldSerialAndHead & MASK_VALUE;
                hkAtomic::readWriteBarrier();   // Make sure m_nextFree[i] is written into memory before being read by allocate()

                if ( hkAtomic::compareAndSwap( addr, oldSerialAndHead, newSerialAndHead ) )
                {
                    break;
                }
            }
        }

        /// Debug check that all nodes are either allocated or are part of the free list.
        /// This should catch any leaked nodes due to e.g. ABA problems.
        void checkConsistency() const
        {
#ifdef HK_DEBUG_SLOW
            hkAtomic::readWriteBarrier();

            hkBitField bf; bf.setSizeAndFill( 0, CAPACITY, 0 );
            int i = m_serialAndHead & MASK_VALUE;
            while( i != MASK_VALUE )
            {
                bf.set( i );
                i = m_nextFree[i];
            }
            for( hkBitField::Iterator it(bf); it.isValid(bf); it.getNext(bf) )
            {
                HK_ASSERT( 0x37bb3c8, it.isCurrentBitSet() || ( m_nextFree[it.getCurrentBit()] == MASK_VALUE ), "Node leaked!" );
            }
#endif  // HK_DEBUG_SLOW
        }

    private:

        /// A serial number (to protect against ABA problems), and the start of the linked list of free indices.
        /// The number of bits given to each is based on CAPACITY.
        hkUint32 m_serialAndHead;

        /// The linked list of free indices. Set to MASK_VALUE if allocated/invalid.
        hkUint32 m_nextFree[CAPACITY];
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
