// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : NX32 NX64
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Thread/Thread/hkThread.h>
#include <Common/Base/Memory/System/hkMemorySystem.h>

#include <nn/os.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/nn_Macro.h>

namespace
{
    struct ThreadImpl
    {
        ThreadImpl() : m_thread(), m_threadStack(nullptr) {}

        void* operator new(size_t size)
        {
            return hkMemoryRouter::easyAlloc(*hkMemorySystem::getInstance().getUncachedLockedHeapAllocator(), size);
        }
        void operator delete(void* ptr)
        {
            hkMemoryRouter::easyFree(*hkMemorySystem::getInstance().getUncachedLockedHeapAllocator(), ptr);
        }

        nn::os::ThreadType m_thread;
        uint8_t* m_threadStack;
    };
}

hkThread::hkThread()
{
    m_thread = new ThreadImpl();
    m_threadId = 0;
}

hkThread::~hkThread()
{
    ThreadImpl* threadImpl = (ThreadImpl*)m_thread;
    if ((threadImpl->m_thread._state == nn::os::ThreadType::State_Initialized) ||
        (threadImpl->m_thread._state == nn::os::ThreadType::State_Started) ||
        (threadImpl->m_thread._state == nn::os::ThreadType::State_Exited))
    {
        nn::os::DestroyThread(&threadImpl->m_thread);
    }
    if (threadImpl->m_threadStack != nullptr)
    {
        hkAlignedDeallocate<uint8_t>(threadImpl->m_threadStack);
    }
    delete threadImpl;
}

void hkThread::joinThread()
{
    ThreadImpl* threadImpl = (ThreadImpl*)m_thread;
    nn::os::WaitThread(&threadImpl->m_thread);
}

hkResult hkThread::startThread( hkThread::StartFunction func, void* arg, const char* name, int stackSize)
{
    ThreadImpl* threadImpl = (ThreadImpl*)m_thread;
    HK_ASSERT(0x41e5ceed, stackSize <= HK_THREAD_DEFAULT_STACKSIZE, "Declared stack size exceeds HK_THREAD_DEFAULT_STACKSIZE");
    threadImpl->m_threadStack = hkAlignedAllocate<uint8_t>(4096, stackSize, HK_MEMORY_CLASS_BASE);
    nn::Result result = nn::os::CreateThread(&threadImpl->m_thread, (void (*)(void *))func, arg, threadImpl->m_threadStack, stackSize, nn::os::DefaultThreadPriority);
    if (!result.IsSuccess())
    {
        return HK_FAILURE;
    }

    // assign to the next free CPU core
    static int idealCore = 1;
    nn::Bit64 mask = nn::os::GetThreadAvailableCoreMask();
    if (mask)
    {
        if ((1ULL << idealCore) > mask)
        {
            idealCore = 1;
        }
        for (; !((1ULL << idealCore) & mask); ++idealCore);

        nn::os::SetThreadCoreMask(&threadImpl->m_thread, idealCore, mask);
        ++idealCore;
    }

    nn::os::StartThread(&threadImpl->m_thread);

    return HK_SUCCESS;
}

hkThread::Status hkThread::getStatus() const
{
    const ThreadImpl* threadImpl = static_cast<const ThreadImpl*>(m_thread);
    switch (threadImpl->m_thread._state)
    {
    case nn::os::ThreadType::State_NotInitialized:
    case nn::os::ThreadType::State_Initialized:
        return THREAD_NOT_STARTED;
    case nn::os::ThreadType::State_Started:
        return THREAD_RUNNING;
    case nn::os::ThreadType::State_DestroyedBeforeStarted:
    case nn::os::ThreadType::State_Exited:
        return THREAD_TERMINATED;
    default:
        return THREAD_NOT_STARTED;
    }
}

namespace
{
    unsigned int calcHash(const char* str)
    {
        hkUlong h = 0;
        for (; *str != 0; ++str)
        {
            h = 31 * h + *str;
        }
        return (unsigned int)(h);
    }
}

hkUint64 HK_CALL hkThread::getMyThreadId()
{
#if (HK_CONFIG_THREAD==HK_CONFIG_MULTI_THREADED)
    //nn::os::ThreadType *currentThread = nn::os::GetCurrentThread();
    return calcHash(nn::os::GetThreadNamePointer(nn::os::GetCurrentThread()));
#else
    return 0;
#endif
}

hkUint64 hkThread::getChildThreadId() const
{
    return m_threadId;
}

void* hkThread::getHandle()
{
    return m_thread;
}

void hkThread::setPriority(int priority)
{
    // This platform treats lower numbers as higher priorities, and does not allow negative priorities.
    // Map to the allowable range, and the expected ordering.
    int platformPriority = nn::os::DefaultThreadPriority - priority;
    nn::os::ChangeThreadPriority(nn::os::GetCurrentThread(), platformPriority);
}

int hkThread::getPriority() const
{
    return nn::os::GetThreadPriority(nn::os::GetCurrentThread());
}

void hkThread::setIdealProcessor(int hardwareThreadId)
{
    // Not supported on NX
}

void hkThread::setThreadAffinityMask(hkUint32 mask)
{
    // Usable cores are specified in the mask
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), nn::os::IdealCoreDontCare, mask);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
