// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkBaseDefs.h>
#include <Common/Base/Thread/TaskQueue/hkTask.h>

class hkTask;
class hkTaskBatchProcessor;
struct hkTimerData;

/// An interface for scheduling the processing of hkTask's.
/// This interface uses abstract handles on which you define tasks, priorities and dependencies before submitting
/// them for processing. Implementations of this interface must provide a way to process submitted handles as they
/// become available (i.e. whenever they have no unprocessed predecessors), in order of priority.
/// See hkDefaultTaskQueue for our standard implementation.
class HK_EXPORT_COMMON hkTaskQueue
{
    public:

        /// Opaque undefined task handle.
        /// Implementations should cast this to their own desired type.
        struct _Handle;
        typedef _Handle* Handle;

        /// A dependency between two handles.
        struct Dependency
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, Dependency );

            Handle m_predecessor;   ///< This handle must be processed before m_successor.
            Handle m_successor;     ///< This handle must be processed after m_predecessor.
        };

        /// Defines how to schedule tasks which have equal priority values.
        struct Order
        {
            enum Enum
            {
                /// First in first out (behave like a queue),
                /// use this if you a want to execute one task after another and your calling function runs on the main thread
                FIFO,

                /// Last in first out (behave like a stack),
                /// use this if you use the taskqueue in a recursive context, e.g. if the calling function is already a task.
                LIFO
            };
        };

    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, hkTaskQueue );

        /// Destructor.
        virtual ~hkTaskQueue() {}

        // -------------------------------------------------------------------------------------------------------------
        // Submission
        // -------------------------------------------------------------------------------------------------------------

        /// Allocate a set of handles for use in the other methods.
        virtual void allocateHandles(_Inout_updates_bytes_(numHandles * striding) Handle* handlesOut, int numHandles, int striding = sizeof(Handle)) = 0;

        /// Free a set of allocated handles for reuse.
        /// Warning: Freeing a submitted but unfinished handle will result in undefined behavior.
        virtual void freeHandles(_Inout_updates_bytes_(numHandles * striding) Handle* handles, int numHandles, int striding = sizeof(Handle) ) = 0;

        /// Initialize a handle with a task and a priority.
        /// Note: "task" is allowed to be null, for example when using the handle purely as a sync point.
        virtual void initHandle( Handle handle, _In_ hkTask* task, enum hkTask::Priority::Enum priority = hkTask::Priority::MEDIUM) = 0;

        /// Set the multiplicity of a task. By default each task has a value of 1.
        /// If you set this to a number > 1, a task will be called up to this many times, it is similar
        /// to adding this task to the queue 'multiplicity' times with the following special behavior:
        ///     - Each time the hkTask::process() is called, the hkTask::Input::m_multiplicityIndex will be a number
        ///       between 0 and multiplicity.
        ///     - If you set mode to ABORT_ON_FIRST_FINISHED_TASK, the system will stop calling the task as soon as any
        ///       thread has finished it. This is useful (and used extensively) if you use the tasks to get a thread
        ///       instead of finishing a specific work item.
        ///       Example: you want to simulate 1000 particles, creating a task per particle is overkill.
        ///       So instead you create an atomic variable called particleIndex to synchronize work.
        ///       So you spawn n-threads, each thread increments this variable 'particleIndex' and works on the 
        ///       particle[particleIndex]. As soon as particleIndex reaches 1000, all particles have been processed
        ///       and there is no need to start new threads (so therefor ABORT_ON_FIRST_FINISHED_TASK). This
        ///       allows for specifying a multiplicity to be very high, but you pay now overhead if less threads
        ///       than 'multiplicity' become available.
        virtual void setMultiplicity( Handle handle, int multiplicity,  hkTask::MultiplicityMode::Enum mode ) = 0;

        /// Add a dependency between two handles.
        /// This means that "successor" may be processed only after "predecessor" has finished processing.
        /// Note: "predecessor" is allowed to be already submitted, but not "successor".
        virtual void addDependency( Handle predecessor, Handle successor ) = 0;

        /// Batch version of addDependency().
        /// This default implementation calls addDependency() on each.
        virtual void addDependencies(_Inout_updates_(numDependencies) Dependency* dependencies, int numDependencies );

        /// Submit a set of handles for processing.
        /// They will be processed in order of their assigned priorities, while respecting any dependencies.
        /// In cases where tasks have equal priority, they will be ordered based on your choice of FIFO or LIFO.
        /// Note: This call creates a reference point in the timer stream, so that these tasks will appear to have the
        /// same timer hierarchy that this thread had when it called this function.
        virtual void submitHandles(_Inout_updates_bytes_(numHandles * striding) Handle* handles, int numHandles,
            Order::Enum order = Order::FIFO, int striding = sizeof(Handle) ) = 0;

        /// Aborts the execution of some handles.
        ///
        /// A handle will be aborted if:
        /// * It is one of the handles passed to abortHandles().
        /// * It is a successor of, or recursively waiting for, a newly aborted handle.
        /// * It is abortable, and its last un-aborted successor (or task recursively waiting for it) is aborted.
        ///
        /// All handles chosen for abortion must be abortable. It is an error to abort a handle which is non-abortable
        /// or has a non-abortable descendant. (It is not an error to abort a handle with a non-abortable ancestor, but
        /// non-abortable ancestors will limit how many tasks can be aborted.)
        ///
        /// All tasks will be processed regardless of their abortion status. An aborted task should not assume that
        /// any of its ancestors have run to completion.
        ///
        /// An implementation is not required to implement this functionality.
        virtual void abortHandles(_Inout_updates_bytes_(numHandles * striding) Handle* handles, int numHandles, int striding = sizeof(Handle) ) {}

        // -------------------------------------------------------------------------------------------------------------
        // Processing
        // -------------------------------------------------------------------------------------------------------------

        /// Process any available handles until the given one has finished.
        virtual void processUntilFinished( Handle handle ) = 0;

        /// Batch version of processUntilFinished().
        /// This default implementation creates a single successor handle if necessary,
        /// and calls the singular processUntilFinished() on that.
        virtual void processUntilFinished(_Inout_updates_(numHandles) Handle* handles, int numHandles );

        /// Return true if a given handle is finished.
        /// Note: The default implementation calls processUntilFinished() and always returns true.
        virtual bool isFinished( Handle handle );

        // -------------------------------------------------------------------------------------------------------------
        // Combined calls for faster processing
        // -------------------------------------------------------------------------------------------------------------

        /// allocate a handle, initialize it and submit. This requires a single lock only
        virtual Handle addAndSubmitTask( hkTask* task, int multiplicity, hkTask::MultiplicityMode::Enum mode, enum hkTask::Priority::Enum priority = hkTask::Priority::MEDIUM );

        HK_INLINE Handle addAndSubmitTask( hkTask* task, enum hkTask::Priority::Enum priority = hkTask::Priority::MEDIUM )
        {
            return addAndSubmitTask( task, 1, hkTask::MultiplicityMode::ABORT_ON_FIRST_FINISHED_TASK, priority );
        }

        /// Wait for a handle and free handle when finished
        virtual void processAndFree( Handle handleToWait );

        // -------------------------------------------------------------------------------------------------------------
        // Helpers
        // -------------------------------------------------------------------------------------------------------------

        /// Helper function to process an entire dependency graph of tasks. This adds handles, processes them and removes them
        virtual void submitGraphAndWait( struct hkTaskGraph& graph, Order::Enum order = Order::FIFO );

        /// Process a set of independent items in parallel.
        /// The set is split into batches. Each batch is processed by one call to hkTaskBatchProcessor::processBatch().
        /// Note: If \p maxBatchSize is set to zero, the default implementation will automatically estimate a batch size.
        virtual void forEach(_Inout_ hkTaskBatchProcessor* batchProcessor, const hkTaskBatchProcessor::BatchInfo& batchInfo );

        /// Return an optional hint indicating the number of threads processing this queue.
        virtual int getNumThreadsHint() const { return 0; }     
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
