// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Thread/TaskQueue/hkTaskQueue.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Common/Base/Thread/TaskQueue/hkTaskGraph.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
#include <Common/Base/System/Hardware/hkHardwareInfo.h>

void hkTaskQueue::addDependencies(_Inout_updates_(numDependencies) Dependency* dependencies, int numDependencies)
{
    for( int i=0; i<numDependencies; ++i )
    {
        addDependency( dependencies[i].m_predecessor, dependencies[i].m_successor );
    }
}

bool hkTaskQueue::isFinished( Handle handle )
{
    processUntilFinished( handle );
    return true;
}

void hkTaskQueue::processUntilFinished(_Inout_updates_(numHandles) Handle* handles, int numHandles )
{
    if( numHandles == 1 )
    {
        processUntilFinished( handles[0] );
    }
    else if( numHandles > 1 )
    {
        Handle successor;
        allocateHandles( &successor, 1 );
        initHandle( successor, HK_NULL );   // dummy handle, no hkTask needed

        hkLocalBuffer<Dependency> dependencies( numHandles );
        for( int i=0; i<numHandles; ++i )
        {
            dependencies[i].m_predecessor = handles[i];
            dependencies[i].m_successor = successor;
        }
        addDependencies( dependencies.begin(), dependencies.getSize() );

        submitHandles( &successor, 1 );
        processUntilFinished( successor );
        freeHandles( &successor, 1 );
    }
}

hkTaskQueue::Handle hkTaskQueue::addAndSubmitTask(
    hkTask* task, int multiplicity, hkTask::MultiplicityMode::Enum mode, hkTask::Priority::Enum priority )
{
    Handle h; allocateHandles(&h, 1);
    initHandle(h, task, priority);
    if (multiplicity != 1)
    {
        setMultiplicity(h, multiplicity, mode);
    }
    Order::Enum order = Order::LIFO;
    submitHandles(&h, 1, order );
    return h;
}

void hkTaskQueue::processAndFree(Handle handle)
{
    processUntilFinished(handle);
    freeHandles(&handle, 1);
}


void hkTaskQueue::submitGraphAndWait( hkTaskGraph& graph, Order::Enum order )
{
    graph.submitToTaskQueueAndWait( *this, order );
}

struct hkTaskQueue_BatchingTask: public hkTask
{
    virtual void process( const hkTask::Input& );

    hkTaskBatchProcessor* m_batchProcessor;
    hkUint32* m_nextItem;
    hkTaskBatchProcessor::BatchInfo m_batchInfo;
};

void hkTaskQueue_BatchingTask::process( const hkTask::Input& input )
{
    m_batchProcessor->beginTask( input, m_batchInfo );

    for ( ;; )
    {
        const int batchStart = (int)hkAtomic::exchangeAdd32( m_nextItem, m_batchInfo.m_maxBatchSize );

        if ( batchStart >= m_batchInfo.m_numItems )
        {
            break;
        }
        const int lastIndex = hkMath::min2( m_batchInfo.m_numItems, batchStart + m_batchInfo.m_maxBatchSize );
        const int numItems = lastIndex - batchStart;
        m_batchProcessor->processBatch( input, m_batchInfo, batchStart, numItems );
    }

    m_batchProcessor->endTask( input, m_batchInfo );
}

void hkTaskQueue::forEach(_Inout_ hkTaskBatchProcessor* batchProcessor, const hkTaskBatchProcessor::BatchInfo& batchInfo)
{
    if( batchInfo.m_numItems <= 0 )
    {
        return;
    }

    HK_ALIGN32( hkUint32 ) nextIndex[32];   // align to minimize false cache sharing. also add buffer to fill a cache line
    nextIndex[0] = 0;

    hkTaskQueue_BatchingTask task;
    {
        task.m_batchProcessor = batchProcessor;
        task.m_batchInfo = batchInfo;   // copy
        task.m_nextItem = &nextIndex[0];
    }

    hkTaskBatchProcessor::BatchInfo& bi = task.m_batchInfo;
    if( bi.m_maxBatchSize <= 0 )
    {
        // Estimate a suitable batch size
        const int estimatedNumWorkers = hkHardwareInfo::getNumHardwareThreads();
        bi.m_maxBatchSize = (estimatedNumWorkers > 1) ? hkMath::max2(1, bi.m_numItems / (estimatedNumWorkers * 2)) : bi.m_numItems;
    }

    const int numBatches = ( bi.m_numItems / bi.m_maxBatchSize) + ( bi.m_numItems % bi.m_maxBatchSize ? 1 : 0 );
    batchProcessor->prepare( bi, numBatches );

    if( numBatches > 1 )
    {
        // Add the task using a "worst case" multiplicity plus the special abort flag
        Handle handle;
        allocateHandles( &handle, 1 );
        initHandle( handle, &task );
        setMultiplicity( handle, numBatches, hkTask::MultiplicityMode::ABORT_ON_FIRST_FINISHED_TASK );
        submitHandles( &handle, 1 );

        // Process the queue until the task is finished
        processUntilFinished( handle );
        freeHandles( &handle, 1 );
    }
    else
    {
        // Process the task directly
        hkTask::Input input;
        input.m_taskQueue = this;
        task.process( input );
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
