// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Thread/TaskQueue/hkTask.h>
#include <Common/Base/Thread/TaskQueue/hkTaskQueue.h>
#include <Common/Base/Types/hkHandle.h>

class hkOstream;


/// A container of tasks and dependencies between them.
/// To process a task graph, you can either:
///  - submit it to a hkTaskQueue by calling submitToTaskQueue(), or
///  - convert it to a form that suits your own scheduling system
struct HK_EXPORT_COMMON hkTaskGraph : public hkReferencedObject
{
    public:

        /// Identifier of a task within a graph.
        HK_DECLARE_HANDLE( TaskId, hkUint16, 0xFFFF );

        /// A node in the graph, describing a task and its attributes.
        struct HK_EXPORT_COMMON Node
        {
            HK_DECLARE_CLASS(Node, New, Pod);

            hkTask* m_task;                         ///< The task to be processed. Can be HK_NULL.
            hkUint16 m_multiplicity;                ///< The maximum number of times that the task should be processed.
            hkEnum < hkTask::Priority::Enum, hkUint8>  m_priority;  ///< The priority of the task.
            hkEnum < hkTask::MultiplicityMode::Enum, hkUint8> m_multiplicityMode;

            hkTaskQueue::Handle m_handle;   ///< The task queue handle
        };

        /// An edge in the graph, describing a dependency between two tasks.
        struct HK_EXPORT_COMMON Dependency
        {
            HK_DECLARE_CLASS(Dependency, New, Pod);

            TaskId m_predecessorId;     ///< The task that must be processed first.
            TaskId m_successorId;       ///< The task that must be processed after the predecessor is finished.
        };

        // Helper struct to map taskIds from one graph to another after calling addTaskGraph()
        struct HK_EXPORT_COMMON TaskIdMapping
        {
            HK_DECLARE_CLASS(TaskIdMapping, New, Pod);
            HK_INLINE TaskIdMapping(int index);

            // Return the TaskId of \in in the graph it was added to.
            HK_INLINE TaskId getMappedId(TaskId in) const;

            protected:
                int m_baseTaskIndex;
        };

    public:

        HK_DECLARE_CLASS(hkTaskGraph, New);

        /// Constructor.
        hkTaskGraph() {}

        /// Destructor.
        virtual ~hkTaskGraph() {}

        //
        // Building methods
        //

        /// Reserve space for the specified number of tasks and dependencies.
        void reserve( int numTasks, int numDependencies = 0 );

        /// Add a task to the graph.
        ///  - \a priority defines the global priority of the task.
        TaskId addTask(_Inout_ hkTask* task, hkTask::Priority::Enum priority = hkTask::Priority::MEDIUM );

        /// Add a task with multiplicity.
        ///  - \a multiplicity defines the maximum number of times that the task should be processed.
        ///  - Note that many implementations will use an internal atomic variable to iterate over a large list of items
        ///    and multiplicity is only used so that this task gets executed on all threads. However if the atomic variable
        ///    is exhausted, there is not more need to spawn new task, in this case the MultiplicityMode can be set to ABORT_ON_FIRST_FINISHED_TASK
        TaskId addTaskWithMultiplicity( hkTask* task, int multiplicity, hkTask::MultiplicityMode::Enum,
            hkTask::Priority::Enum priority = hkTask::Priority::MEDIUM );

        /// Add a reference counted task to the graph.
        /// The graph maintains a reference to the given task until the graph is cleared or destroyed.
        TaskId addReferencedTask( hkReferencedTask* task, hkTask::Priority::Enum priority = hkTask::Priority::MEDIUM );

        /// Add a reference counted task with multiplicity.
        /// The graph maintains a reference to the given task until the graph is cleared or destroyed.
        ///  - \a multiplicity defines the maximum number of times that the task should be processed.
        TaskId addReferencedTaskWithMultiplicity(_Inout_ hkReferencedTask* task, int multiplicity, hkTask::MultiplicityMode::Enum,
            hkTask::Priority::Enum priority = hkTask::Priority::MEDIUM );

        /// Add a dependency between two tasks. Both IDs must be valid.
        HK_INLINE void addDependency( TaskId predecessorId, TaskId successorId );

        /// Add a dependency if both task IDs are valid, otherwise do nothing.
        HK_INLINE bool tryAddDependency( TaskId predecessorId, TaskId successorId );

        /// Insert a task between two dependent tasks.
        /// The predecessorId and successorId must both be valid and form a dependency.
        void insertDependency( TaskId predecessorId, TaskId intermediaryId, TaskId successorId );

        /// Insert a task between two dependent tasks if predecessorId and successorId are both valid and form a dependency,
        /// otherwise do nothing.
        void tryInsertDependency( TaskId predecessorId, TaskId intermediaryId, TaskId successorId );

        /// Return true if a direct dependency exists between from predecessor to successor.
        /// Note: this is a rather slow debug function.
        bool hasDependency( TaskId predecessorId, TaskId successorId ) const;

        /// Return true if a dependency path exists from predecessor to successor.
        /// Note: this is a rather slow debug function.
        bool hasTransitiveDependency( TaskId predecessorId, TaskId successorId ) const;

        /// Adds another graph to this graph.
        TaskIdMapping append( const hkTaskGraph& other );

        /// Clear the graph.
        virtual void clear();

        //
        // Helper methods
        //

        /// Returns true if the graph is empty.
        HK_INLINE bool isEmpty() const;

        /// Get read-only access to the nodes array.
        HK_INLINE const hkArray<Node>& getNodes() const { return m_nodes; }

        /// Get read-only access to the dependencies array.
        HK_INLINE const hkArray<Dependency>& getDependencies() const { return m_dependencies; }

        /// Get read-only access to the referenced tasks
        HK_INLINE const hkArray< hkRefPtr< hkReferencedTask > >& getReferencedTasks() const {   return m_referencedTasks;   }

        /// Get read-write access to a given task.
        HK_INLINE hkTask* accessTask( TaskId id ) const { return m_nodes[ id.valueUnchecked() ].m_task; }

        /// Print the graph using the plain text DOT graph description language.
        /// The output can be turned into an image using e.g Graphviz.
        void printDot( hkOstream& outStream ) const;

        /// Print the graph in the Directed Graph Markup Language (DGML) format.
        /// The output can be opened in Microsoft Visual Studio (2010+).
        void printDgml( hkOstream& outStream ) const;

        //
        // Processing methods
        //

        /// Preallocate a handle in the given task queue for the given task.
        /// You can use this to set up dependencies with external handles before submitting the graph.
        hkTaskQueue::Handle preallocateTaskQueueHandle( hkTaskQueue& taskQueue, TaskId taskId );

        /// Submit handles to the given queue for processing, and returns a handle that you can wait for.
        /// This may add a dummy task to the graph to ensure that there is exactly one leaf handle to return.
        /// Notes:
        ///  - This call creates a reference point in the timer stream, so that all timers created by other threads
        ///    will inherit the same timer hierarchy that existed at the moment of calling this function in this thread.
        hkTaskQueue::Handle submitToTaskQueue( hkTaskQueue& taskQueue, hkTaskQueue::Order::Enum order = hkTaskQueue::Order::FIFO );

        /// Free all handles that were allocated in the given task queue for this graph.
        /// Note: This includes the handle that was returned by submitToTaskQueue().
        void freeTaskQueueHandles( hkTaskQueue& taskQueue );

        /// Helper function to submit and wait for the completion of all tasks in the graph.
        /// Also frees all handles from the queue.
        void submitToTaskQueueAndWait( hkTaskQueue& taskQueue, hkTaskQueue::Order::Enum order = hkTaskQueue::Order::FIFO );

    protected:

        /// The nodes of the graph.
        hkInplaceArray<Node,16> m_nodes;

        /// The edges of the graph.
        hkInplaceArray<Dependency,16> m_dependencies;

        /// Manages the reference count of any hkReferencedTasks that were added.
        hkArray< hkRefPtr< hkReferencedTask > > m_referencedTasks;
};

#include <Common/Base/Thread/TaskQueue/hkTaskGraph.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
