// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Thread/TaskQueue/hkTaskGraph.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Base/Thread/Async/hkAsyncHeartbeat.h>

// If this triggers, you need to change the storage of m_taskIdsToDependencyIdx
HK_COMPILE_TIME_ASSERT( sizeof( hkTaskGraph::TaskId ) == sizeof( hkUint16 ) );

// Extra internal functions
struct hkTaskGraphEx : public hkTaskGraph
{
    // Make sure the graph has a single leaf task. Returns the index of that task.
    int ensureSingleLeafTask()
    {
        int numLeaves = m_nodes.getSize();
        hkInplaceArray<hkUint8, 1024> isLeaf; isLeaf.setSize( numLeaves, 1 );
        for( int i = 0; i < m_dependencies.getSize(); ++i )
        {
            hkUint8& is = isLeaf[m_dependencies[i].m_predecessorId.value()];
            numLeaves -= is;    // this is no longer a leaf
            is = 0;
        }

        if( numLeaves == 1 )
        {
            for( int i = 0; i < m_nodes.getSize(); i++ )
            {
                if( isLeaf[i] )
                {
                    return i;
                }
            }
            return 0;   // error, cannot happen since we have 1 leaf
        }
        else
        {
            // Add a new task as a successor to all the leaves
            TaskId exitTaskId = addTask( HK_NULL );
            int exitTaskPriority = 0;
            for( int i = 0; i < m_nodes.getSize() - 1; i++ )    // -1 because we just added a final leaf
            {
                exitTaskPriority = hkMath::max2( exitTaskPriority, m_nodes[i].m_priority );
                if( isLeaf[i] )
                {
                    addDependency( hkTaskGraph::TaskId( i ), exitTaskId );
                }
            }
            m_nodes[exitTaskId.valueUnchecked()].m_priority = hkTask::Priority::Enum( exitTaskPriority );
            return exitTaskId.value();
        }
    }

private:

    hkTaskGraphEx() {}
};

void hkTaskGraph::insertDependency( TaskId predecessorId, TaskId intermediaryId, TaskId successorId )
{
    addDependency(predecessorId, intermediaryId);
    addDependency(intermediaryId, successorId);
}

void hkTaskGraph::tryInsertDependency( TaskId predecessorId, TaskId intermediaryId, TaskId successorId )
{
    tryAddDependency(predecessorId, intermediaryId);
    tryAddDependency(intermediaryId, successorId);
}

hkTaskGraph::TaskId hkTaskGraph::addTask( _Inout_ hkTask* task, hkTask::Priority::Enum prio )
{
    HK_ASSERT( 0x6ca69f2d, m_nodes.getSize() + 1 <= TaskId::Type(-1), "Graph is full" );
    HK_CHECK_ALIGN16(task);

    TaskId taskId( m_nodes.getSize() );

    Node* HK_RESTRICT node = &m_nodes.expandOne();
    node->m_task = task;
    node->m_multiplicity = 1;
    node->m_priority = prio;
    node->m_handle = HK_NULL;

    return taskId;
}

hkTaskGraph::TaskId hkTaskGraph::addTaskWithMultiplicity(
    hkTask* task, int multiplicity, hkTask::MultiplicityMode::Enum multiType, hkTask::Priority::Enum priority )
{
    HK_ASSERT(0x6ca69f2d, m_nodes.getSize() + 1 <= TaskId::Type(-1), "Graph is full");
    HK_ASSERT(0x6ca69f2e, multiplicity > 0, "multiplicity must be greater than zero");
    HK_CHECK_ALIGN16(task);

    TaskId taskId(m_nodes.getSize());

    Node* HK_RESTRICT node = &m_nodes.expandOne();
    node->m_task = task;
    node->m_multiplicity = (hkInt16)multiplicity;
    node->m_multiplicityMode = multiType;
    node->m_priority = priority;
    node->m_handle = HK_NULL;

    return taskId;
}

hkTaskGraph::TaskId hkTaskGraph::addReferencedTask(_Inout_ hkReferencedTask* task, hkTask::Priority::Enum priority )
{
    m_referencedTasks.pushBack( task );
    return addTask( task, priority );
}

hkTaskGraph::TaskId hkTaskGraph::addReferencedTaskWithMultiplicity(
    hkReferencedTask* task, int multiplicity, hkTask::MultiplicityMode::Enum multiplicityMode, hkTask::Priority::Enum priority )
{
    m_referencedTasks.pushBack(task);
    return addTaskWithMultiplicity(task, multiplicity, multiplicityMode, priority);
}

hkTaskGraph::TaskIdMapping hkTaskGraph::append( const hkTaskGraph& other )
{
    const int firstNewTaskIdx   = m_nodes.getSize();
    const int firstNewDepIdx    = m_dependencies.getSize();

    // Append nodes
    m_nodes.append(other.m_nodes);
    m_referencedTasks.append(other.m_referencedTasks);

    // Append handles
    m_dependencies.append(other.m_dependencies);

    // Re-index dependencies
    for (int k = m_dependencies.getSize() - 1; k >= firstNewDepIdx; k--)
    {
        Dependency& dep = m_dependencies[k];
        dep.m_predecessorId = TaskId(firstNewTaskIdx + dep.m_predecessorId.value());
        dep.m_successorId   = TaskId(firstNewTaskIdx + dep.m_successorId.value());
    }

    return TaskIdMapping(firstNewTaskIdx);
}

void hkTaskGraph::reserve( int numTasks, int numDependencies )
{
    m_nodes.reserve(numTasks);
    m_dependencies.reserve(numDependencies);
    m_referencedTasks.reserve(numTasks);
}

void hkTaskGraph::clear()
{
    m_nodes.clear();
    m_dependencies.clear();
    m_referencedTasks.clear();
}

hkTaskQueue::Handle hkTaskGraph::preallocateTaskQueueHandle( hkTaskQueue& taskQueue, TaskId taskId )
{
    Node& node = m_nodes[taskId.value()];
    if ( !node.m_handle )
    {
        taskQueue.allocateHandles( &node.m_handle, 1 );
    }
    return node.m_handle;
}


hkTaskQueue::Handle hkTaskGraph::submitToTaskQueue( hkTaskQueue& taskQueue, hkTaskQueue::Order::Enum order )
{
    hkAsyncHeartbeat::beat(0x86810496);

    // HK_TIMER_BEGIN_LIST("hkTaskGraph::submitToTaskQueue", "Find Leaf Node");

    // Make sure we have exactly one leaf node, so that we can return a single handle to wait for
    hkTaskGraphEx* self = static_cast<hkTaskGraphEx*>(this);
    int leafNodeIndex = self->ensureSingleLeafTask();

    hkAsyncHeartbeat::beat(0xf8fc114d);

#if 0
    // Print the task graph to a file (for debugging)
    {
        hkOstream os( "hkTaskGraph_submitToTaskQueue.dgml" );
        printDgml( os );
    }
#endif

    // HK_TIMER_SPLIT_LIST("Allocate Handles");

    const int numNodes = m_nodes.getSize();
    const int numDependencies = m_dependencies.getSize();

    // allocate a handle for all tasks with no handle yet
    {
        int numNewHandles = 0;
        for( int i=0; i<numNodes; ++i )
        {
            if( !m_nodes[i].m_handle )
            {
                numNewHandles++;
            }
        }

        hkInplaceArray<hkTaskQueue::Handle,256>::Temp handles;
        hkTaskQueue::Handle* newHandles = handles.expandBy( numNewHandles );
        taskQueue.allocateHandles( newHandles, numNewHandles );

        for( int i=0, h=0; i<numNodes; ++i )
        {
            if( !m_nodes[i].m_handle )
            {
                m_nodes[i].m_handle = handles[h++];
            }
        }
    }

    hkAsyncHeartbeat::beat(0x87a134ac);

    // Initialize all the handles (including preallocated ones)
    {
        for( int i=0; i<numNodes; ++i )
        {
            const Node& node = m_nodes[i];
            taskQueue.initHandle( node.m_handle, node.m_task, node.m_priority );
            if( node.m_multiplicity > 1 )
            {
                taskQueue.setMultiplicity( node.m_handle, node.m_multiplicity, node.m_multiplicityMode );
            }
        }
    }

    hkAsyncHeartbeat::beat(0x24526dfc);

    // HK_TIMER_SPLIT_LIST("Set Dependencies");

    // Add all the dependencies
    if( numDependencies > 0 )
    {
        hkInplaceArray<hkTaskQueue::Dependency, 256>::Temp dependencies;
        dependencies.setSize( numDependencies );
        for( int i=0; i<numDependencies; ++i )
        {
            const Dependency& d = m_dependencies[i];
            hkTaskQueue::Dependency& dep = dependencies[i];
            dep.m_predecessor = m_nodes[d.m_predecessorId.value()].m_handle;
            dep.m_successor   = m_nodes[d.m_successorId.value()].m_handle;
        }
        taskQueue.addDependencies( dependencies.begin(), numDependencies );
    }

    // HK_TIMER_SPLIT_LIST("Submit");
    hkAsyncHeartbeat::beat(0xeb30a1da);

    taskQueue.submitHandles( &m_nodes[0].m_handle, m_nodes.getSize(), order, sizeof(hkTaskGraph::Node) );

    // HK_TIMER_END_LIST();

    hkAsyncHeartbeat::beat(0x5707b48f);

    return m_nodes[leafNodeIndex].m_handle;
}

void hkTaskGraph::submitToTaskQueueAndWait( hkTaskQueue& taskQueue, hkTaskQueue::Order::Enum order )
{
    hkTaskQueue::Handle finalHandle = submitToTaskQueue( taskQueue, order );
    taskQueue.processUntilFinished( finalHandle );
    freeTaskQueueHandles( taskQueue );
    clear();
}

void hkTaskGraph::freeTaskQueueHandles( hkTaskQueue& taskQueue )
{
    taskQueue.freeHandles( &m_nodes[0].m_handle, m_nodes.getSize(), sizeof(hkTaskGraph::Node) );
    for( int i=0; i<m_nodes.getSize(); ++i )
    {
        m_nodes[i].m_handle = HK_NULL;
    }
}

bool hkTaskGraph::hasDependency( TaskId predecessorId, TaskId successorId ) const
{
    for (int i = 0; i < m_dependencies.getSize(); ++i)
    {
        if (m_dependencies[i].m_predecessorId == predecessorId &&
            m_dependencies[i].m_successorId == successorId)
        {
            return true;
        }
    }
    return false;
}

bool hkTaskGraph::hasTransitiveDependency( TaskId predecessorId, TaskId successorId ) const
{
    if (hasDependency(predecessorId, successorId))
    {
        return true;
    }

    for (int i = 0; i < m_dependencies.getSize(); ++i)
    {
        const Dependency& dep = m_dependencies[i];
        if (dep.m_predecessorId == predecessorId)
        {
            if (hasTransitiveDependency(dep.m_successorId, successorId))
            {
                return true;
            }
        }
    }
    return false;
}


// Helpers used by hkTaskGraph::printDot()
namespace
{
    struct SubGraph
    {
        hkStringPtr m_name;
        hkArray<int> m_members;
    };

    static void printTask( const hkTaskGraph::Node& task, int i, hkOstream &outStream, int nameOffset = 0 )
    {
        hkStringBuf label = "NULL";
        hkStringBuf dotFormat = "";
        if( task.m_task )
        {
            label = task.m_task->getName() + nameOffset;

            hkTask::DebugAttributes attributes;
            task.m_task->getDebugAttributes( attributes );

            if( !attributes.m_extraInfo.isEmpty() )
            {
                label.appendPrintf( "\\n%s", attributes.m_extraInfo.cString() );
            }

            switch( attributes.m_category )
            {
            case hkTask::DebugAttributes::CATEGORY_WORKER:
                dotFormat = "fillcolor=\"#e0ffe0\""; // light green
                break;
            case hkTask::DebugAttributes::CATEGORY_SYNC:
                dotFormat = "fillcolor=lightyellow";
                break;
            case hkTask::DebugAttributes::CATEGORY_SYNC_FIRING_SIGNALS:
                dotFormat = "fillcolor=\"#ff8080\""; // red-ish
                break;
            default:
                break;
            }
        }

        if( task.m_multiplicity > 1 )
        {
            label.appendPrintf( "\\n(&#215;%i)", task.m_multiplicity );
            if( dotFormat.getLength() > 0 )
            {
                dotFormat.append( "," );
            }
            dotFormat.append( "peripheries=2" );
        }

        outStream.printf( "\t%i [label=\"%s\",%s]\n", i, label.cString(), dotFormat.cString() );
    }
}

void hkTaskGraph::printDot( hkOstream& outStream ) const
{
    // See http://en.wikipedia.org/wiki/DOT_language for details on the output format.

    outStream.printf( "digraph {\n" );
    //outStream.printf( "\trankdir = LR\n" );
    outStream.printf( "\tgraph [fontsize=12,fontname=Helvetica,style=filled,fillcolor=gray90]\n" );
    outStream.printf( "\tnode [shape=box,fontsize=10,fontname=Helvetica,margin=0.1,height=0,style=filled,fillcolor=white]\n" );

    // Gather subgraphs
    hkArray<SubGraph> subGraphs;
    hkArray<int> subGraphIndices; subGraphIndices.setSize( m_nodes.getSize(), -1 );
    for( int i = 0; i < m_nodes.getSize(); ++i )
    {
        const hkTaskGraph::Node& task = m_nodes[i];
        if( task.m_task )
        {
            const char* name = task.m_task->getName();
            const char* pos = hkString::strChr(name, '/' );
            if( pos )
            {
                hkStringBuf groupName; groupName.append( name, int(pos - name) );
                // search for subgraph
                int subGraphIndex = -1;
                for( int k=0; k < subGraphs.getSize(); k++ )
                {
                    if( subGraphs[k].m_name == groupName.cString() )
                    {
                        subGraphIndex = k;
                        break;
                    }
                }
                if( subGraphIndex < 0 )
                {
                    subGraphIndex = subGraphs.getSize();
                    SubGraph& group = subGraphs.expandOne();
                    group.m_name = groupName;
                }
                SubGraph& group = subGraphs[subGraphIndex];
                group.m_members.pushBack( i );
                subGraphIndices[i] = subGraphIndex;
            }
        }
    }

    // Print subgraphs
    for( int sgi = 0; sgi < subGraphs.getSize(); sgi++ )
    {
        SubGraph& subGraph = subGraphs[sgi];
        outStream.printf( "\tsubgraph cluster%i {\n", sgi );
        outStream.printf( "\tlabel=\"%s\";\n", subGraph.m_name.cString() );
        int groupNameLen = subGraph.m_name.getLength();
        for( int j = 0; j < subGraph.m_members.getSize(); j++ )
        {
            int i = subGraph.m_members[j];
            const hkTaskGraph::Node& task = m_nodes[i];
            printTask( task, i, outStream, groupNameLen+1 );
        }

        outStream.printf( "\t}\n" );
    }

    // print all tasks not belonging to a subgraph
    for( int i = 0; i < m_nodes.getSize(); ++i )
    {
        if( subGraphIndices[i] < 0 )
        {
            const hkTaskGraph::Node& task = m_nodes[i];
            printTask( task, i, outStream );
        }
    }

    // Print dependencies
    for( int i = 0; i < m_dependencies.getSize(); ++i )
    {
        const hkTaskGraph::Dependency& d = m_dependencies[i];
        outStream.printf( "\t%i -> %i\n", d.m_predecessorId.valueUnchecked(), d.m_successorId.valueUnchecked() );
    }

    outStream.printf( "}\n" );
}

void hkTaskGraph::printDgml( hkOstream& outStream ) const
{
    // See https://en.wikipedia.org/wiki/DGML for details on the output format.

    outStream.printf( "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n" );
    outStream.printf( "<DirectedGraph GraphDirection=\"TopToBottom\" xmlns=\"http://schemas.microsoft.com/vs/2009/dgml\">\n" );
    outStream.printf( "\t<Nodes>\n" );

    hkArray< hkStringPtr >::Temp categories;
    hkArray< hkStringPtr >::Temp groupNodes;
    hkArray< hkTuple< int, int > > groupToNodeMapping;

    for( int i = 0; i < m_nodes.getSize(); ++i )
    {
        hkStringBuf nodeName;
        hkStringBuf nodeAttributes;

        const char* categoryName;
        hkStringBuf copyOfName;

        if( m_nodes[i].m_task )
        {
            hkTask::DebugAttributes attributes;
            m_nodes[i].m_task->getDebugAttributes( attributes );

            nodeName = m_nodes[i].m_task->getName();
            if( !attributes.m_extraInfo.isEmpty() )
            {
                nodeName.appendPrintf( " (%s)", attributes.m_extraInfo.cString() );
            }

            copyOfName = nodeName;

            // Hack: tokenize doesn't work with "\n" above for some reason, so manually take care of this case
            int indexOfNewLine = hkString::indexOf(copyOfName, '\n');
            if (indexOfNewLine != -1)
            {
                copyOfName.slice(0, indexOfNewLine);
            }
            categoryName = copyOfName.cString();

            // Search for a group ID.
            /*
            hkArray<const char*>::Temp attributeElements;
            const int numEntries = attributes.m_dotAttributes.split( ',', attributeElements );
            for( int entry = 0; entry < numEntries; ++entry )
            {
                if( hkString::strNcmp( attributeElements[entry], "groupId=", 8 ) == 0 )
                {
                    const char*  attributeValue = attributeElements[entry] + 8;

                    bool found = false;
                    int groupNodeIndex;
                    for( groupNodeIndex = 0; groupNodeIndex < groupNodes.getSize(); ++groupNodeIndex )
                    {
                        if( groupNodes[groupNodeIndex] == attributeValue )
                        {
                            found = true;
                            break;
                        }
                    }

                    hkTuple< int, int >& mapping = groupToNodeMapping.expandOne();
                    mapping.m_0 = groupNodeIndex;
                    mapping.m_1 = i;

                    if( !found )
                    {
                        groupNodes.pushBack( attributeValue );
                    }
                }
            }
            */
        }
        else
        {
            nodeName = "NULL";
            categoryName = "(NONE)";
        }

        nodeName.replace( "\n", "&#xA;" );
        nodeName.replace( "\"", "&#34;" );
        nodeName.replace( "<", "&lt;" );
        nodeName.replace( ">", "&gt;" );

        bool found = false;
        for( int c = 0; c < categories.getSize(); ++c )
        {
            if( hkString::strCmp( categories[c], categoryName ) == 0 )
            {
                found = true;
                break;
            }
        }

        if( !found )
        {
            categories.pushBack( categoryName );
        }

        outStream.printf( "\t\t<Node Id=\"N_%u\" Label=\"%s\" Category=\"%s\" />\n", i, nodeName.cString(), categoryName );
    }

    for( int i = 0; i < groupNodes.getSize(); ++i )
    {
        outStream.printf( "\t\t<Node Id=\"%s\" Label=\" \" Group=\"Expanded\" />\n", groupNodes[i].cString() );
    }

    outStream.printf( "\t</Nodes>\n" );

    // Print dependencies
    outStream.printf( "\t<Links>\n" );

    for( int i = 0; i < m_dependencies.getSize(); ++i )
    {
        const hkTaskGraph::Dependency& d = m_dependencies[i];
        outStream.printf( "\t\t<Link Source=\"N_%u\" Target=\"N_%u\" />\n", d.m_predecessorId.value(), d.m_successorId.value() );
    }

    for ( int i = 0; i < groupToNodeMapping.getSize(); ++i )
    {
        hkTuple< int, int > mapping = groupToNodeMapping[i];
        outStream.printf( "\t\t<Link Source=\"%s\" Target=\"N_%u\" Category=\"Contains\" />\n", groupNodes[mapping.m_0].cString(), (int)mapping.m_1 );
    }

    outStream.printf( "\t</Links>\n" );

    // Print categories
    outStream.printf( "\t<Categories>\n" );

    for( int i = 0; i < categories.getSize(); ++i )
    {
        const float hue = static_cast<float>(i) / static_cast<float>(categories.getSize());
        hkUint32 color = hkColor::rgbFromHSV( hue, 1.0f, 0.5f );

        outStream.printf( "\t\t<Category Id=\"%s\" Label=\"%s\" Background=\"#%x\" />\n", categories[i].cString(), categories[i].cString(), color );
    }
    outStream.printf( "\t</Categories>\n" );

    outStream.printf( "</DirectedGraph>" );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
