// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkBaseDefs.h>
#include <Common/Base/Object/hkReferencedObject.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Thread/Atomic/hkAtomicPrimitives.h>

class hkTaskQueue;
class hkMonitorStream;


/// Task interface.
/// A task is a self-contained unit of work that may be executed in any thread.
HK_CLASSALIGN16(class) HK_EXPORT_COMMON hkTask
{
    public:

        HK_DECLARE_CLASS( hkTask, New, ReflectVirtualBase );

        /// A generic interface for controlling the execution of a task
        struct HK_EXPORT_COMMON ExecutionContext
        {
            HK_DECLARE_CLASS(ExecutionContext, New);

            /// Destructor
            virtual ~ExecutionContext() {}

            /// Returns true if the execution of the task should be aborted
            virtual bool isAbortRequested() const = 0;
        };

        /// Some context passed to process()
        struct Input
        {
            HK_DECLARE_CLASS(Input, New);
            Input() : m_taskQueue(HK_NULL), m_monitorStream(HK_NULL), m_executionContext(HK_NULL), m_multiplicityIndex(0) {}

            hkTaskQueue* m_taskQueue;               ///< The task queue (if available)
            hkMonitorStream* m_monitorStream;       ///< The monitor stream (if available)
            ExecutionContext* m_executionContext;   ///< The execution context

            /// If a task requested a multiplicity > 1, this a unique ID between 0 and multiplicity for this iteration.
            int m_multiplicityIndex;                
        };

        /// Defines how many times a task with multiplicity will be called.
        struct MultiplicityMode
        {
            enum Enum
            {
                /// The task will be called exactly multiplicity times.
                PERFORM_ALL_CALLS,

                /// The task will be called up to multiplicity times,
                /// aborting as soon as any thread has finished processing it.
                ABORT_ON_FIRST_FINISHED_TASK
            };
        };

        /// Task priority
        struct Priority
        {
            enum Enum
            {
                LOWEST = 0,
                LOW  = 0x20,
                MEDIUM_LOW = 0x30,
                MEDIUM = 0x40,
                MEDIUM_HIGH = 0x50,
                HIGH = 0x80,
                MAX = 0xff,
            };
        };

        /// Some attributes used for debugging and instrumentation.
        struct DebugAttributes
        {
            HK_DECLARE_CLASS( DebugAttributes, New );

            inline DebugAttributes() : m_category(CATEGORY_WORKER) {}

            /// Arbitrary dynamic information.
            hkStringBuf m_extraInfo;

            /// High level task types.
            enum TaskCategory
            {
                CATEGORY_WORKER,                ///< the task is doing actual work.
                CATEGORY_SYNC,                  ///< the task is a short task used to synchronize other tasks.
                CATEGORY_SYNC_FIRING_SIGNALS,   ///< the task is a single threaded task which fires signals.
            };
            TaskCategory m_category;
        };

    public:

        /// Destructor.
        virtual ~hkTask() {}

        /// Process the task.
        virtual void process( const hkTask::Input& input ) = 0;

        /// Get a name for the task.
        virtual const char* getName() const;

        /// Get some optional debug attributes for the task.
        virtual void getDebugAttributes( DebugAttributes& attributesOut ) const;

        /// Returns whether the task can deal with its predecessors being aborted. If a task indicates it is abortable,
        /// it must check receivedAbortRequest() before it consumes any data produced by its direct or indirect
        /// predecessors, and should exit as soon as possible if receivedAbortRequest() returns true.
        virtual bool isAbortable() const { return false; }

        /// Helper function. Returns true if the given execution context has been requested to abort this task.
        static HK_INLINE bool HK_CALL receivedAbortRequest(_In_opt_ const ExecutionContext* execContext)
        {
            return execContext ? execContext->isAbortRequested() : false;
        }
        static HK_INLINE bool HK_CALL receivedAbortRequest( const Input& taskInput )
        {
            return receivedAbortRequest(taskInput.m_executionContext);
        }
};


/// A reference counted hkTask.
HK_CLASSALIGN16(class) HK_EXPORT_COMMON hkReferencedTask : public hkReferencedObject, public hkTask
{
    public:

        HK_DECLARE_CLASS( hkReferencedTask, New, ReflectIdentity );
};


/// An alternative task interface to be used if you need to execute a small function on an array of items.
class hkTaskBatchProcessor
{
    public:

        struct BatchInfo
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, BatchInfo );

            BatchInfo( int numItems = 0, int batchSize = 0 )
            :   m_numItems(numItems), m_maxBatchSize(batchSize), m_userData(HK_NULL) {}

            int m_numItems;     ///< The total number of items
            int m_maxBatchSize; ///< The (maximum) size of each batch of items

            void* m_userData;   ///< An optional pointer to some data
        };

    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, hkTaskBatchProcessor );

        virtual ~hkTaskBatchProcessor() {}

        /// Called before processing, once the batch size has been defined.
        virtual void prepare( const BatchInfo&, int numBatches ) {}

        /// Called when a task (worker thread) is beginning.
        virtual void beginTask( const hkTask::Input& input, const BatchInfo &bi ) {}

        /// Called (multiple times) by a thread to process a batch of items.
        /// Only for the last batch the numItems might be smaller than the bi.m_batchSize
        virtual void processBatch( const hkTask::Input& input, const BatchInfo &bi, int batchStart, int numItems ) = 0;

        /// Called when a task (worker thread) is ending.
        /// Every worker thread which was started, will call this function.
        virtual void endTask( const hkTask::Input& input, const BatchInfo &bi ) {}
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
