// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM       : ALL
// PRODUCT        : COMMON
// VISIBILITY     : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Thread/TaskQueue/hkTask.h>
#include <Common/Base/Thread/Atomic/hkIntegerDistributor.h>
#include <Common/Base/Monitor/hkMonitorStream.h>

/// Common code for hkBatchTask and hkReferencedBatchTask. This class should not be used directly.
template<typename BaseType, typename DerivedType>
class hkBatchTaskBase : public BaseType
{
public:
    /// Constructor.
    ///
    /// @param numThreads the number of expected invocations of this task. This should include the main thread, if it
    /// will be processing tasks.
    /// @param numItems the size of the range to invoke workers for.
    hkBatchTaskBase(int numThreads, int numItems)
        : m_numThreads(numThreads)
    {
        setNumItems(numItems);
    }

    /// Constructor which doesn't set a number of items. The number of items should be provided through setNumItems
    /// before processing begins.
    ///
    /// @param numThreads the number of expected invocations of this task. This should include the main thread, if it
    /// will be processing tasks.
    hkBatchTaskBase(int numThreads)
        : m_numThreads(numThreads)
    {
    }

    /// Set the number of times to call processItem(). This hsould only
    void setNumItems(int numItems)
    {
        m_distributor.initThreadInfo(m_numThreads, numItems);
    }

    virtual void process(const hkTask::Input& input) override
    {
        typename DerivedType::WorkerContext workerContext;
        static_cast<DerivedType *>(this)->initWorkerContext(workerContext);

        int effThreadId = input.m_multiplicityIndex;

        int counter = 0;
        while(true)
        {
            if(DerivedType::NUM_ITEMS_PER_ABORT_CHECK != 0)
            {
                if((counter++ & (DerivedType::NUM_ITEMS_PER_ABORT_CHECK-1)) == 0
                    && hkTask::receivedAbortRequest(input))
                {
                    return;
                }
            }

            hkUint32 i;
            hkResult res = m_distributor.getItem(effThreadId, &i);

            if(res.isFailure())
            {
                break;
            }

            static_cast<DerivedType *>(this)->processItem(workerContext, i);
        }

        static_cast<DerivedType *>(this)->finalizeWorkerContext(workerContext);
    }

    virtual bool isAbortable() const override
    {
        return DerivedType::NUM_ITEMS_PER_ABORT_CHECK != 0;
    }

    static const unsigned int NUM_ITEMS_PER_ABORT_CHECK = 0;

private:
    hkIntegerDistributor m_distributor;
    int m_numThreads;
};

/// A task which operates on a large number of work items, distributing them to threads. Its purpose is similar to that
/// of hkTaskBatchProcessor, but it can be used as part of a task graph and provides better distribution when there is
/// large variation in the computational cost of individual work items.
///
/// This task should be used through the CRTP. In each invocation of the task, a WorkerContext is allocated and
/// initialized through initWorkerContext(). For each integer i in the range [0, numItems), one task invoation calls
/// processItem(), passing that task's worker context. Each invocation of the task calls finalizeWorkerContext() when
/// there are no more items to process. Note that some invocations may call finalizeWorkerContext() while others are
/// still in processItem().
///
/// Batch tasks should be given a multiplicity equal to the numThreads specified in the constructor.
///
/// If the derived task returns true from isAbortable(), the task will check for abortion before every item by default,
/// or every NUM_ITEMS_PER_ABORT_CHECK if that is defined. (It must be a power of 2.)
template<typename DerivedType>
class hkBatchTask : public hkBatchTaskBase<hkTask, DerivedType>
{
public:
    typedef hkBatchTaskBase<hkTask, DerivedType> BaseType;

    hkBatchTask(int numThreads, int numItems) : BaseType(numThreads, numItems) {}
    hkBatchTask(int numThreads) : BaseType(numThreads) {}
};

/// A referenced version of hkBatchTask.
template<typename DerivedType>
class hkReferencedBatchTask : public hkBatchTaskBase<hkReferencedTask, DerivedType>
{
public:
    typedef hkBatchTaskBase<hkReferencedTask, DerivedType> BaseType;

    hkReferencedBatchTask(int numThreads, int numItems) : BaseType(numThreads, numItems) {}
    hkReferencedBatchTask(int numThreads) : BaseType(numThreads) {}
};



/// Helper macro to monitor the task execution time.
#define HK_BATCH_TASK_USE_MONITOR(NAME) \
    virtual void process(const hkTask::Input& input) override \
    { \
        HK_TIME_CODE_BLOCK(NAME, HK_NULL); \
        BaseType::process(input); \
    }

#if 0
/// A sample batch task, illustrating usage.
class AddNumbersTask : public hkBatchTask<AddNumbersTask>
{
public:
    AddNumbersTask(hkArrayView<hkUint32> addends, int numThreads)
        : hkBatchTask<AddNumbersTask>(numThreads, addends.getSize())
        , m_sum(0)
        , m_addends(addends)
    {
    }

    HK_BATCH_TASK_USE_MONITOR("AddNumbers");

    hkUint32 m_sum;

    hkArrayView<hkUint32> m_addends;

    struct WorkerContext
    {
        int m_localSum;
    };

    void initWorkerContext(WorkerContext & workerContext)
    {
        workerContext.m_localSum = 0;
    }

    void processItem(WorkerContext & workerContext, int i)
    {
        workerContext.m_localSum += m_addends[i];
    }

    void finalizeWorkerContext(WorkerContext & workerContext)
    {
        hkAtomic::exchangeAdd(&m_sum, workerContext.m_localSum);
    }

    static const int NUM_ITEMS_PER_ABORT_CHECK = 4;
};
#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
