// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Algorithm/Graph/hkDirectedGraph.h>
#include <Common/Base/Thread/TaskQueue/Default/hkDefaultTaskQueue.h>
#include <Common/Base/Thread/TaskQueue/Default/hkDefaultTaskQueue_HandleAllocator.h>

/// A graph of handles that can access both the input and output edges
class hkDefaultTaskQueue::HandleGraph : public hkDirectedGraph< hkDirectedGraphTypes::Array::VertexSet<hkDefaultTaskQueue::HandleImpl*>, hkDirectedGraphTypes::Array::EdgeSet<void> >
{
    public:

        HK_DECLARE_CLASS(HandleGraph, New);

    public:

        /// Builds the graph
        HandleGraph(_Inout_ HandleAllocator* handleAllocator)
        {
            // Allocate the vertices
            for (HandleAllocator::Iterator it = handleAllocator->getIterator(); it.isValid(); it.next())
            {
                HandleImpl* h = it.getHandle();

                if ( h->isAllocated() )
                {
                    const VertexId vid  = addVertex();
                    Vertex& v           = accessVertex(vid);
                    v.m_payload         = h;
                    h->m_graphVertexId  = vid.value();
                }
                else
                {
                    h->m_graphVertexId = VertexId::InvalidValue;
                }
            }

            // Allocate the edges
            for (VertexId fromVtxId = m_vertices.getFirstElement(); fromVtxId.isValid(); fromVtxId = m_vertices.getNextElement(fromVtxId))
            {
                const Vertex& v         = getVertex(fromVtxId);
                const HandleImpl* hFrom = v.m_payload;

                for (int si = hFrom->m_successors.getSize() - 1; si >= 0; si--)
                {
                    const HandleImpl* hTo   = hFrom->m_successors[si];
                    const VertexId toVtxId  (hTo->m_graphVertexId);

                    addEdge(fromVtxId, toVtxId);
                }

                for(int si = hFrom->m_waitingHandles.getSize() - 1; si >= 0; si--)
                {
                    const HandleImpl* hTo = hFrom->m_waitingHandles[si];
                    if(hTo)
                    {
                        const VertexId toVtxId(hTo->m_graphVertexId);
                        addEdge(fromVtxId, toVtxId);
                    }
                }
            }
        }

        /// Marks all appropriate handles as aborted (see hkTaskQueue::abortHandles for the exact criteria).
        void abortHandles(_Inout_updates_bytes_(numHandles * striding) hkTaskQueue::Handle* handles, int numHandles, int striding)
        {
            hkArray<VertexId>::Temp abortable; abortable.reserve(numHandles*2);

            // Add initial handles
            {
                hkTaskQueue::Handle* hPtr = handles;

                for(int k = numHandles - 1; k >= 0; k--, hPtr = hkAddByteOffset(hPtr, striding))
                {
                    const HandleImpl* h = reinterpret_cast<HandleImpl*>(*hPtr);
                    const VertexId vtxId(h->m_graphVertexId);

                    if(vtxId.isValid())
                    {
                        abortable.pushBack(vtxId);
                    }
                }
            }

            while(!abortable.isEmpty())
            {
                    // Pop stack top
                const VertexId vertexId = abortable.back(); abortable.popBack();
                Vertex& vertex = accessVertex(vertexId);

                HK_ASSERT(0x7a9adee2, isVertexAbortable(vertexId), "Aborted tasks and their descendants must be abortable.");

                if(isVertexAborted(vertexId))
                {
                    continue;
                }

                abortVertex(vertexId);

                // Queue successors
                for(EdgeId outEdgeId = vertex.getFirstOutgoingEdge(); outEdgeId.isValid(); /**/)
                {
                    const Edge& outEdge = getEdge(outEdgeId);
                    HK_ASSERT_NO_MSG(0x4b222652, outEdge.getStartVertex() == vertexId);

                    const VertexId successorId = outEdge.getEndVertex();
                    abortable.pushBack(successorId);
                    outEdgeId = outEdge.getNextOutgoingEdge();
                }

                // Queue predecessors which are abortable, not yet aborted, and have no non-aborted successors
                for(EdgeId inEdgeId = vertex.getFirstIncomingEdge(); inEdgeId.isValid(); /**/)
                {
                    const Edge& inEdge = getEdge(inEdgeId);
                    HK_ASSERT_NO_MSG(0x4b222653, inEdge.getEndVertex() == vertexId);

                    const VertexId predecessorId = inEdge.getStartVertex();
                    if(predecessorId.isValid())
                    {
                        if(!isVertexAborted(predecessorId)
                            && isVertexAbortable(predecessorId)
                            && areAllSuccessorsAborted(predecessorId))
                        {
                            abortable.pushBack(predecessorId);
                        }
                    }
                    inEdgeId = inEdge.getNextIncomingEdge();
                }
            }
        }

    private:
        bool isVertexAborted(VertexId vertexId) const
        {
            return getVertex(vertexId).m_payload->m_abortRequest;
        }

        bool isVertexAbortable(VertexId vertexId) const
        {
            return getVertex(vertexId).m_payload->m_task->isAbortable();
        }

        bool areAllSuccessorsAborted(VertexId vertexId) const
        {
            Vertex const& vertex = getVertex(vertexId);
            for(EdgeId outEdgeId = vertex.getFirstOutgoingEdge(); outEdgeId.isValid(); /**/)
            {
                const Edge& outEdge = getEdge(outEdgeId);
                const VertexId successorId = outEdge.getEndVertex();

                if(!isVertexAborted(successorId))
                {
                    return false;
                }

                outEdgeId = outEdge.getNextOutgoingEdge();
            }

            return true;
        }

        void abortVertex(VertexId vertexId)
        {
            HK_ASSERT_NO_MSG(0x4b2bece1, isVertexAbortable(vertexId));
            getVertex(vertexId).m_payload->m_abortRequest = true;
        }
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
