// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
//HK_REFLECTION_PARSER_EXCLUDE_FILE

#pragma once

#include <Common/Base/Thread/TaskQueue/Default/hkDefaultTaskQueue_HandleImpl.h>
#include <Common/Base/Thread/ThreadSafeIntegerPool/hkThreadSafeIntegerPool.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>


/// Handle allocator used by hkDefaultTaskQueue.
/// Allocates handles from a linked list of fixed size blocks.
class hkDefaultTaskQueue::HandleAllocator
{
    private:

        // A block of handles.
        struct Block
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, Block );

            enum
            {
                BLOCK_SIZE = 4096,  // default page size on many platforms
                NUM_HANDLES = (BLOCK_SIZE - sizeof(void*)) / sizeof(HandleImpl)
            };

            Block()
            {
                // Initialize the local free list
                HandleImpl* HK_RESTRICT h = &m_handles[0];
                for( int i=0; i<NUM_HANDLES; i++, h++ )
                {
                    h->m_dependencyCount = hkUint16(-2);    // mark as free
                    h->m_nextFreeHandle = h+1;              // link to next
                }
                m_handles[NUM_HANDLES-1].m_nextFreeHandle = HK_NULL;    // terminate list
                m_nextBlock = HK_NULL;
            }

            HandleImpl m_handles[NUM_HANDLES];
            Block* m_nextBlock;
        };

        HK_COMPILE_TIME_ASSERT( sizeof(Block) <= Block::BLOCK_SIZE );   // padding makes it not exactly equal

    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, HandleAllocator );

        /// Constructor.
        HandleAllocator( int spinCount = 0 ) : m_lock(spinCount)
        {
            m_firstFreeHandle = HK_NULL;
            m_firstBlock = HK_NULL;

            // Preallocate some blocks
            const int numPreallocatedHandles = 256;
            const int numPreallocatedBlocks = ( numPreallocatedHandles + Block::NUM_HANDLES - 1 ) / Block::NUM_HANDLES; // ceil
            for( int i=0; i<numPreallocatedBlocks; ++i )
            {
                allocateBlock();
            }
        }

        /// Destructor.
        ~HandleAllocator()
        {
            // Delete all blocks
            while( m_firstBlock )
            {
                Block* block = m_firstBlock->m_nextBlock;
                delete m_firstBlock;
                m_firstBlock = block;
            }
        }

        HK_INLINE void initializeHandle(_Inout_ HandleImpl* h)
        {
            h->m_dependencyCount = 1;   // an implicit dependency, removed when this handle is submitted
            h->m_multiplicity = 1;
            h->m_numActiveThreads = 0;
            h->m_onFinishZeroMultiplicity = false;
            h->m_waitingThreadsBitfield.clear();
            h->m_priority = hkTask::Priority::MEDIUM;
            h->m_sortKey = 0;               // indicates that it is not submitted
            h->m_task = HK_NULL;            // indicates that it is not initialized
            h->m_successors.clear();
            h->m_abortRequest = false;
        }

        /// Allocate a batch of handles. Thread safe.
        void allocate( _Inout_updates_bytes_(numHandles * striding) HandleImpl** handlesOut, _In_range_(>=, 0) int numHandles, _In_range_(>=, sizeof(HandleImpl*)) int striding )
        {
            if( numHandles > 0 )
            {
                m_lock.enter();

                for (int i = 0; i < numHandles; i++)
                {
                    if (!m_firstFreeHandle)
                    {
                        // Prepend a new block
                        allocateBlock();
                    }

                    // Take the free handle
                    HandleImpl* h = m_firstFreeHandle;
                    m_firstFreeHandle = h->m_nextFreeHandle;

                    // Initialize it
                    initializeHandle(h);

                    HK_ASSERT_NO_MSG(0x239f1f74, !h->m_successors.wasReallocated());

                    // Recomputing the full offset makes it easier for OACR to validate the memory accesses
                    HandleImpl** handleCur = hkAddByteOffset(handlesOut, i * striding);
                    *handleCur = h;
                }
                m_lock.leave();
            }
        }

        /// Free a batch of handles. Thread safe.
        void free(_Inout_updates_bytes_(numHandles * striding) HandleImpl** handles, int numHandles, int striding )
        {
            if( numHandles > 0 )
            {
                m_lock.enter();

                // Link them
                HandleImpl** ph = handles;
                HandleImpl* h = HK_NULL;
                HandleImpl* lastH = HK_NULL;
                for( int i=0; i<numHandles; i++, ph = hkAddByteOffset(ph,striding) )
                {
                    h = *ph;
                    HK_ASSERT_NO_MSG(0xe9a035d, !h->isSubmitted() || h->isFinished() );
                    HK_ASSERT_NO_MSG( 0x2ba44275, h->isAllocated() );

                    h->m_dependencyCount = hkUint16(-2);        // mark as free
                    h->m_nextFreeHandle = lastH;                // link to next (avoiding out of bounds)
                    HK_ASSERT_NO_MSG( 0x239f1f76, h->m_successors.isEmpty() );
                    if( h->m_successors.wasReallocated() )
                    {
                        h->m_successors.optimizeCapacity(0);    // reset to inplace storage
                    }
                    lastH = h;
                }

                // Prepend them to the free list
                handles[0]->m_nextFreeHandle = m_firstFreeHandle;
                m_firstFreeHandle = h;

                m_lock.leave();
            }
        }

        /// Free all handles. Thread safe.
        void freeAll()
        {
            m_lock.enter();
            m_firstFreeHandle = m_firstBlock ? &m_firstBlock->m_handles[0] : HK_NULL;
            Block* block = m_firstBlock;
            while( block )
            {
                // Free any external allocations made by the inplace arrays
                for( int i=0; i<Block::NUM_HANDLES; ++i )
                {
                    block->m_handles[i].m_successors.clearAndDeallocate();
                }

                // Reconstruct the block
                Block* next = block->m_nextBlock;
                new (block) Block;
                if( next )
                {
                    block->m_nextBlock = next;
                    block->m_handles[Block::NUM_HANDLES-1].m_nextFreeHandle = &next->m_handles[0];
                }
                block = next;
            }
            m_lock.leave();
        }

        /// Iterator over the handles
        struct Iterator
        {
            friend class hkDefaultTaskQueue::HandleAllocator;

            HK_INLINE bool isValid() const
            {
                return m_currentBlock && (m_currentHandleIdx < Block::NUM_HANDLES);
            }

            HK_INLINE void next()
            {
                if ( ++m_currentHandleIdx == Block::NUM_HANDLES )
                {
                    m_currentBlock      = m_currentBlock->m_nextBlock;
                    m_currentHandleIdx  = 0;
                }
            }

            HK_INLINE _Ret_notnull_ HandleImpl* getHandle()
            {
                return &m_currentBlock->m_handles[m_currentHandleIdx];
            }

        private:

            Block* m_currentBlock;
            int m_currentHandleIdx;
        };

        /// Returns an iterator over the handles
        Iterator getIterator()
        {
            Iterator it;
            it.m_currentBlock       = m_firstBlock;
            it.m_currentHandleIdx   = 0;

            return it;
        }

        /// Returns the capacity
        HK_INLINE int getCapacity() const
        {
            int numBlocks = 0;
            for (Block* b = m_firstBlock; b; b = b->m_nextBlock)
            {
                numBlocks++;
            }

            return numBlocks * Block::NUM_HANDLES;
        }

    private:

        /// Prepend a block of handles to the free list.
        HK_INLINE void allocateBlock()
        {
            Block* block = new Block;
            block->m_nextBlock = m_firstBlock;
            m_firstBlock = block;

            block->m_handles[Block::NUM_HANDLES-1].m_nextFreeHandle = m_firstFreeHandle;
            m_firstFreeHandle = &block->m_handles[0];
        }

    private:

        /// A linked list of blocks
        Block* m_firstBlock;

        /// A linked list of free handles
        HandleImpl* m_firstFreeHandle;

        /// A critical section, for thread safety
        hkCriticalSection m_lock;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
