// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// this: #include <Common/Base/Thread/TaskQueue/Default/hkDefaultTaskQueue.h>

#include <Common/Base/Thread/TaskQueue/hkTaskQueue.h>
#include <Common/Base/Thread/TaskQueue/hkTask.h>
#include <Common/Base/Thread/TaskQueue/hkTaskGraph.h>
#include <Common/Base/Container/MinHeap/hkMinHeap.h>
#include <Common/Base/Container/PointerMap/hkMap.h>

class hkAsyncThreadPool;


/// The default implementation of the hkTaskQueue interface.
/// This supports multiple threads processing the queue in parallel.
class HK_EXPORT_COMMON hkDefaultTaskQueue : public hkReferencedObject, public hkTaskQueue
{
    public:

        enum
        {
            /// The maximum number of threads that can process the queue in parallel.
            MAX_NUM_THREADS = 64    
        };

        /// Scheduling modes for the task queue.
        enum SchedulingMode
        {
            /// Blocking mode - default.
            /// Threads waiting for work will block and yield their processor. A full critical section
            /// is used to ensure mutually exclusive access to internal task queue structures.
            /// This mode allows sharing of physical cores with other threads trying to get processing resources.
            /// In this mode, m_spinCount can be used to configure some spinning before blocking.
            MODE_BLOCKING,

            /// Spinning mode (busy waiting).
            /// Threads that are waiting for work will spin atomically and indefinitely instead of blocking and
            /// yielding the processor, threads trying to access shared task queue data will also spin atomically
            /// waiting for the resource.
            /// Blocking operations will still happen when allocating task handles or using the task queue logger.
            /// m_spinCount is unused when this scheduling mode is selected.
            /// When using spinning scheduling, Havok threads running on the queue will not spontaneously yield
            /// their processors to other (non-Havok) work. In order to yield those processors and stop Havok
            /// execution, the task queue needs to be closed.
            /// This mode removes any overhead due to transitions between user and kernel mode, blocking and context
            /// switches.
            /// You may want to consider this mode if you have physical cores available exclusively for Havok workers
            /// during Havok workloads, since it will typically introduce a smaller overhead. We have seen
            /// large gains compared to blocking scheduling on consoles like PlayStation(R)4 and Xbox One.
            /// in order to reduce overhead due to SchedulingMode::BLOCKING.
            /// Note that this feature is currently in BETA.
            MODE_SPINNING,

            /// Async thread pool mode.
            /// Use the hkAsyncThreadPool aware locking primitives (\ref hkAsyncThreadPoolSimpleSemaphore and
            /// \ref hkAsyncThreadPoolMutex).
            /// This mode is similar to \ref MODE_BLOCKING, in that the thread return control to the operating system
            /// while it's waiting, so that the scheduler can run other threads, but because it uses the
            /// hkAsyncThreadPool aware locking primitives, it has the additional feature that a waiting worker thread
            /// won't block the hkAsyncThreadPool from switching between foreground and background mode.
            MODE_ASYNC_THREAD_POOL
        };

        /// Construction parameters.
        struct Cinfo
        {
            Cinfo() : m_schedulingMode(MODE_BLOCKING), m_spinCount(0), m_maxNumThreads(MAX_NUM_THREADS), m_asyncThreadPool(HK_NULL) {}

            /// Specifies the scheduling mode for the task queue.
            SchedulingMode m_schedulingMode;

            /// The spin count is used when acquiring critical sections.
            /// Optimal values vary depending on the platform and the use case. In case of doubt, use the default.
            /// This is only used when m_schedulingMode = SchedulingMode::BLOCKING.
            int m_spinCount;

            /// The maximum number of threads that can concurrently process the queue. Defaults to 64.
            /// Reduce this only if you need to reduce the number of allocated semaphores.
            int m_maxNumThreads;

            /// When using MODE_ASYNC_THREAD_POOL, this is the owning async thread pool.
            /// Advanced use only.
            hkAsyncThreadPool* m_asyncThreadPool;
        };

    public:

        HK_DECLARE_CLASS(hkDefaultTaskQueue, New);

        /// Constructor.
        hkDefaultTaskQueue( const Cinfo& cinfo = Cinfo() );

        /// Destructor.
        ~hkDefaultTaskQueue();

        /// Process a single task if one is immediately available.
        /// Returns false if no task was available.
        bool processOnce();

        /// Process any tasks that become available.
        /// Blocks until close() is called.
        void process();

        /// Instruct ALL threads to exit from process() as soon as there is no task available.
        /// Warning: Does NOT wait for those threads to actually exit.
        void close();

        /// Returns true if close() has been called.
        bool isClosed() const;

        /// Free all handles, reset the global serial number, and reopen the queue.
        void reset();

        /// Read monitor streams from all threads currently processing this queue (up to the most recent safe point).
        void getTimerData( hkArray<hkTimerData>& timerDataOut );

        /// Clear monitor streams of all threads currently processing this queue.
        void clearTimerData();

        /// Log all tasks and dependencies that are added after this call. Only available in debug builds.
        void startTrace();

        /// Export all data logged since startTrace(), and clear the log.
        void endTrace( hkTaskGraph& traceGraphOut );

        // Setter for m_numThreadsHint.
        inline void setNumThreadsHint( int numThreads ) { m_numThreadsHint = numThreads; }

        /// Check if the given handle is finished.
        static bool isHandleFinished( Handle handle );

        static bool isHandleAborted(Handle handle);

        /// hkTaskQueue implementation
        virtual void allocateHandles(_Inout_updates_bytes_(numHandles * striding) Handle* handlesOut, int numHandles, int striding = sizeof(Handle) ) HK_OVERRIDE;
        virtual void freeHandles(_Inout_updates_bytes_(numHandles * striding) Handle* handlesOut, int numHandles, int striding = sizeof(Handle) ) HK_OVERRIDE;
        virtual void initHandle( Handle handle, _In_opt_ hkTask* task, hkTask::Priority::Enum priority = hkTask::Priority::MEDIUM ) HK_OVERRIDE;
        virtual void setMultiplicity( Handle handle, int multiplicity, hkTask::MultiplicityMode::Enum mode = hkTask::MultiplicityMode::PERFORM_ALL_CALLS ) HK_OVERRIDE;
        virtual void addDependency( Handle predecessor, Handle successor ) HK_OVERRIDE;
        virtual void addDependencies(_Inout_updates_(numDependencies)  Dependency* dependencies, int numDependencies ) HK_OVERRIDE;
        virtual void submitHandles(_Inout_updates_bytes_(numHandles * striding) Handle* handles, int numHandles, Order::Enum order = Order::FIFO, int striding = sizeof(Handle) ) HK_OVERRIDE;
        virtual void processUntilFinished( Handle handles ) HK_OVERRIDE;
        virtual void processUntilFinished(_Inout_updates_(numHandles) Handle* handles, int numHandles ) HK_OVERRIDE;
        virtual void abortHandles(_Inout_updates_bytes_(numHandles * striding) Handle* handles, int numHandles, int striding = sizeof(Handle) ) HK_OVERRIDE;
        virtual int getNumThreadsHint() const HK_OVERRIDE { return m_numThreadsHint; }

        virtual bool isFinished( Handle handle ) HK_OVERRIDE;
        virtual void processAndFree(Handle handleToWait) HK_OVERRIDE;
        virtual Handle addAndSubmitTask(hkTask* task, int multiplicity, hkTask::MultiplicityMode::Enum mode, enum hkTask::Priority::Enum priority = hkTask::Priority::MEDIUM) HK_OVERRIDE;


        // Base critical section
        class CriticalSection
        {
        public:
            HK_DECLARE_CLASS( CriticalSection, New );
            virtual ~CriticalSection() {}

            void enter();
            void enter(_Inout_opt_ hkMonitorStream* monitorStream);
            void leave();
            void leave(_Inout_opt_ hkMonitorStream* monitorStream );

        private:
            virtual void enterImpl() = 0;
            virtual void leaveImpl() = 0;
        };

        // Opaque class, defined in the cpp file
        class ThreadContext;

    protected:

        // Opaque classes, defined in the cpp file
        class HandleImpl;
        class HandleAllocator;
        class HandleGraph;
        class Logger;

        /// An entry in m_queue.
        struct QueueEntry
        {
            HK_DECLARE_CLASS(QueueEntry, New, Pod);

            HK_INLINE hkBool32 operator<( const QueueEntry& other ) const
            {
                // Note: Using '>' here since we are actually using hkMinHeap as a MAX heap
                return ( this->m_priority > other.m_priority );
            }

            hkUint32 m_priority;    ///< The handle's priority
            HandleImpl* m_handle;   ///< The handle
        };

        /// An empty task, used when initHandle() is given a null task.
        struct EmptyTask : public hkTask
        {
            HK_DECLARE_CLASS(EmptyTask, New);

            virtual void process( const hkTask::Input& ) HK_OVERRIDE {}

            virtual const char* getName() const HK_OVERRIDE
            {
                return "EmptyTask";
            }

            virtual void getDebugAttributes( DebugAttributes& attributesOut ) const HK_OVERRIDE
            {
                attributesOut.m_category = DebugAttributes::CATEGORY_SYNC;
            }

            virtual bool isAbortable() const HK_OVERRIDE
            {
                return true;
            }
        };

    protected:

        /// A critical section to protect against concurrent access.
        CriticalSection* m_lock;

        /// Responsible for allocating handles.
        HandleAllocator* m_handleAllocator;

        /// An incremental serial number applied to each submitted handle.
        /// Used to order handles which have equal priority values.
        hkUint32 m_serial;

        /// All handles that are currently available for processing, sorted by priority.
        hkMinHeap<QueueEntry> m_queue;

        /// A pool of per-thread contexts.
        const int m_maxNumThreads;
        int m_maxUsedThreadContext;
        ThreadContext* m_threadContexts[MAX_NUM_THREADS];
        hkInplaceArray<int, MAX_NUM_THREADS> m_freeThreadContexts;

        /// Whether the queue has been closed.
        bool m_isClosed;

        /// An empty task, used as a placeholder when a handle has no real task.
        EmptyTask m_emptyTask;

        /// Storage for get/setNumThreadsHint().
        /// Note: This isn't actually used by this class, it is just piggybacking.
        int m_numThreadsHint;   

        /// See comments on SchedulingMode.
        const SchedulingMode m_schedulingMode;

        /// Optional debugging helper.
        HK_DEBUG_ONLY_MEMBER( Logger*, m_logger );
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
