// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#if defined (__HAVOK_PARSER__)
class hkSemaphore {};
#else

#include <Common/Base/Config/hkConfigThread.h>

#if defined(HK_PLATFORM_MAC)
#   include <semaphore.h>
#endif
#if (defined(HK_PLATFORM_MAC) || defined(HK_PLATFORM_IOS) || defined(HK_PLATFORM_LINUX) || \
    defined(HK_PLATFORM_LRBXENSIM) || defined(HK_PLATFORM_ANDROID) || defined(HK_PLATFORM_TIZEN) ) && (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
#   include <pthread.h>
#elif defined(HK_PLATFORM_WIIU)
#   include <cafe/os.h>
#elif defined(HK_PLATFORM_LRBSIM)
#   include <common/winpthread.h>
#elif defined(HK_PLATFORM_PS4)
#   include <kernel.h>
#elif defined(HK_PLATFORM_NX)
#   include <nn/os/os_Semaphore.h>
#endif

    /// A wrapper class for a semaphore.
    /// Semaphores are about 10 times slower than critical sections, but
    /// they are more flexible.
    /// You can think of a Semaphore as a set of tokens.
    /// A call to acquire() grabs a token and release puts a token back.
    /// If acquire() can not get a token, it simply waits until another thread calls release.
    /// If the number of tokens is maxCount, release will do nothing.
class HK_EXPORT_COMMON hkSemaphore
{
    public:

        HK_DECLARE_CLASS(hkSemaphore, New, NonCopyable);

            /// Create a semaphore with an initial count and a maximum count. The number of spin iterations is used in some
            /// implementations that use a critical section to perform polling with busy wait when entering it.
        hkSemaphore( int initialCount = 0, int maxCount = 1000, int numSpinIterations = 0 );

            /// Destruct the Semaphore
        ~hkSemaphore();

            /// This call will block until the semaphore is released.
        void acquire();

            /// Release the semaphore. Releases one or more threads blocked by acquire().
        void release(int count);

            /// Release the semaphore. Releases a thread blocked by acquire().
        HK_INLINE void release() { release(1); }

            /// If the semaphore can be released count times, do this and return HK_SUCCESS. If not, do nothing and return HK_FAILURE
        hkResult tryRelease(int count = 1);

            /// If the semaphore can be acquired, do this and return HK_SUCESS. If not, do nothing and return HK_FAILURE
        hkResult tryAcquire();

            // These static functions work on both simulated spu and cpu.
        static void HK_CALL acquire(_Inout_ hkSemaphore* semaphoreOnPpu);
        static void HK_CALL release(_Inout_ hkSemaphore* semaphoreOnPpu, int count = 1);

    protected:

#if defined(HK_PLATFORM_PS4)

        SceKernelSema m_semaphore;

#elif (defined(HK_PLATFORM_MAC) || defined(HK_PLATFORM_IOS) || defined(HK_PLATFORM_LINUX) || \
      defined(HK_PLATFORM_LRB) || defined(HK_PLATFORM_ANDROID) || defined(HK_PLATFORM_TIZEN)) && (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)

        //Mutex used to simulate the Semaphore
        struct hkSemaphorePosix
        {
            int             curCount;
            int             maxCount;
            int             numSpinIterations;
            pthread_mutex_t mutex;
            pthread_cond_t  cond;
        } m_semaphore;

#elif defined(HK_PLATFORM_WIIU) && (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)

        OSSemaphore m_semaphore;
        OSFastMutex m_mutex;
        int m_maxCount;

#elif defined(HK_PLATFORM_NX)
        nn::os::SemaphoreType m_semaphore;

#elif defined(HK_PLATFORM_SIM) // spu simulator
    public:     void* m_semaphore;
#else
    protected:  void* m_semaphore;
#endif

};
#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
