// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Thread/Pool/hkThreadPool.h>
#include <Common/Base/Thread/Thread/hkThread.h>
#include <Common/Base/Thread/Semaphore/hkSemaphore.h>

class hkWorkerThreadContext;
class hkTaskQueue;

/// Construction info for hkCpuThreadPool
struct HK_EXPORT_COMMON hkCpuThreadPoolCinfo
{
    HK_DECLARE_CLASS(hkCpuThreadPoolCinfo, New);

    hkCpuThreadPoolCinfo();

    /// How 'm_hardwareThreadMasksOrIds' is interpreted to execute this pool's threads on specific hardware threads
    struct HardwareThreadBinding
    {
        enum Enum
        {
            UNSPECIFIED,    // Threads may run on any hardware thread; expects m_hardwareThreadMasksOrIds to be empty
            SCHEDULER_HINT, // Threads request execution on specific hardware thread ID, solely via scheduler hint. Windows platforms only.
            HARD_AFFINITY   // Threads hard-affinitized to execute only on masked hardware thread IDs
        };
    };

    hkEnum<HardwareThreadBinding::Enum, hkUint8> m_hardwareThreadBinding;

    /// The number of threads to create.
    /// Defaults to 1.
    int m_numThreads;

    /// The hardware thread IDs or masks to use for each thread (see HardwareThreadBinding::SCHEDULER_HINT/HARD_AFFINITY, respectively).
    /// If empty array, assumes UNSPECIFIED hardware binding and assigns hardware thread IDs spanning all cores before assigning to
    /// more hardware threads in same core (see hkCpuThreadPool::calcHardwareThreadMaskOrId())
    /// These defaults assume that you are using hardware thread 0 as your main calling thread.
    hkArray<hkUint32> m_hardwareThreadMasksOrIds;

    /// The program stack size to use for each thread, if applicable.
    /// On Windows and Xbox360, this defaults to 0, meaning the thread inherits the executable's stack size.
    /// On PlayStation(R)3, this defaults to 256K.
    /// On PlayStation(R)4, this defaults to 512K in Debug/Dev configurations and 128K in Release.
    /// This has no effect on other platforms (e.g., posix).
    int m_stackSize;

    /// The buffer size to allocate to each thread for collecting timer information.
    /// To view timers in the Visual Debugger, you must set this buffer size to a non zero value.
    /// 2MB is a good recommended size. Smaller buffers are ok, but you may lose some timer info.
    /// Defaults to 0, which means that timers will be disabled.
    int m_timerBufferPerThreadAllocation;

    /// The thread name to be passed to each hkThread::startThread().
    const char* m_threadName;
};


/// A CPU implementation of the hkThreadPool interface. It can be used to process a job or a task queue or to process
/// other types of work loads providing a custom worker function to the processWorkLoad() method.
class HK_EXPORT_COMMON hkCpuThreadPool : public hkThreadPool
{
    public:

        /// Function to be run on each of the worker threads when a work load is processed via a processWorkLoad() call.
        typedef void (HK_CALL *WorkerFunction)(void* workLoad);

        /// Determines whether to start the worker threads on construction of the thread pool or manually via a
        /// startTreads() call. The later option is usefull when deriving from hkCpuThreadPool to prevent threads from
        /// starting before the derived class constructor finishes.
        enum ThreadStartMode
        {
            START_THREADS_ON_CONSTRUCTION,
            START_THREADS_MANUALLY
        };

    public:

        HK_DECLARE_CLASS(hkCpuThreadPool, New, ReflectIdentity);
        HK_RECORD_ATTR(hk::IncludeInMgd(false));

        /// Constructor.
        hkCpuThreadPool(
            const hkCpuThreadPoolCinfo& cinfo, ThreadStartMode threadStartMode = START_THREADS_ON_CONSTRUCTION);

        /// Destructor.
        ~hkCpuThreadPool();

        /// Starts the worker threads. Call only if the thread pool has been constructed with
        /// START_THREADS_MANUALLY thread start mode.
        void startThreads( const hkCpuThreadPoolCinfo& cinfo );

        /// Sets the worker function and work load and wakes up all worker threads, returning immediately.
        void processWorkLoad( WorkerFunction workerFunction, _Inout_ void* workLoad );

        //
        // hkThreadPool interface
        //

        virtual void processJobQueue(_Inout_ hkJobQueue* jobQueue, hkJobType notUsed = HK_JOB_TYPE_MAX ) HK_OVERRIDE;
        virtual void processTaskQueue(_Inout_ hkDefaultTaskQueue* taskQueue ) HK_OVERRIDE;
        virtual void waitForCompletion() HK_OVERRIDE;
        virtual bool isProcessing() const HK_OVERRIDE;
        virtual _Ret_maybenull_ hkJobQueue* getProcessingJobQueue() const HK_OVERRIDE;
        virtual _Ret_maybenull_ hkDefaultTaskQueue* getProcessingTaskQueue() const HK_OVERRIDE;
        virtual void appendTimerData( hkArray<hkTimerData>& timerDataOut ) HK_OVERRIDE;
        virtual void clearTimerData() HK_OVERRIDE;
        virtual int getNumThreads() const HK_OVERRIDE;
        virtual void setNumThreads( int numThreads ) HK_OVERRIDE;
        virtual void gcThreadMemoryOnNextCompletion() HK_OVERRIDE;

        // returns true if the passed in thread id belongs to a thread in this thread pool
        bool HK_CALL doesThreadBelongToPool(hkUint64 threadId) const;

    protected:

        /// State data per worker thread
        struct ThreadData
        {
            HK_DECLARE_CLASS(ThreadData, New);

            ThreadData();

            /// Pointer to the data shared between threads.
            hkCpuThreadPool* m_threadPool;

            /// System handle to the thread.
            hkThread m_thread;

            /// Thread Id from 1 - N (0 is reserved for main thread)
            int m_threadId;

            /// Hardware thread mask (or ID) onto which this software thread will be affinitized (or scheduler-hinted)
            int m_hardwareThreadMasksOrId;
            hkEnum<hkCpuThreadPoolCinfo::HardwareThreadBinding::Enum, hkUint8> m_hardwareThreadBinding;

            int m_hardwareThreadId;

            /// Flag is set to true when the thread is requested to close.
            bool m_killThread;

            /// Set this to true to request a reset of the thread's monitor stream
            bool m_clearTimers;

            /// Semaphore used to pause a thread after it has processed its workload.
            /// This semaphore is released by the main thread on every simulation step.
            hkSemaphore m_semaphore;

            // Internal buffer used for collecting timer information
            hkTimerData m_timerData;
        };

    protected:

        void addThread(hkCpuThreadPoolCinfo::HardwareThreadBinding::Enum hardwareBinding = hkCpuThreadPoolCinfo::HardwareThreadBinding::UNSPECIFIED,
            hkUint32 maskOrId = 0);
        void removeThread();

        /// Simple forwarder to the member function.
        static _Ret_null_ void* HK_CALL threadMainForwarder(_Inout_ void* workerThreadData);

        /// Worker thread main function. May be overridden in derived classes to customize thread setup and shutdown.
        virtual void threadMain(ThreadData* threadData);

        /// Worker thread processing loop. Must be called from the thread's main function to process work loads.
        void threadProcessingLoop(ThreadData* threadData, hkMemoryRouter* memoryRouter);

        /// Worker function for processing a job queue.
        static void HK_CALL jobQueueWorkerFunction(_Inout_ void* jobQueue);

        /// Worker function for processing a task queue.
        static void HK_CALL taskQueueWorkerFunction(_Inout_ void* taskQueue);

    protected:

        /// Data local to each thread.
        hkInplaceArray<ThreadData*,32> m_workerThreads;

        /// Function run by worker threads to process a work load.
        WorkerFunction m_workerFunction;

        /// Pointer to a generic work load to be processed by worker threads calling m_workerFunction.
        void* m_workLoad;

        /// The job queue being processed (if any)
        hkJobQueue* m_jobQueue;

        /// The task queue being processed (if any)
        hkDefaultTaskQueue* m_taskQueue;

        /// Semaphore used to pause the main thread when it waits for threads
        /// to finish their calculations.
        hkSemaphore m_workerThreadFinished;

        /// Monitor buffer size per thread.
        int m_timerBufferAllocation;

        /// True if garbage collection must be performed after the next work load is processed.
        hkBool m_gcThreadMemoryOnCompletion;

        /// Debugging flag set to true when worker threads are stepping
        hkBool m_isRunning;

        /// String for thread names (depending on platform support)
        const char* m_threadName;

        /// See comments for hkCpuThreadPoolCinfo::m_stackSize
        int m_stackSize;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
